<?php
// src/Twig/EncoreEntrypointExtension.php
namespace App\Twig;

use Twig\Extension\AbstractExtension;
use Twig\TwigFunction;

class EncoreEntrypointExtension extends AbstractExtension
{
    private string $entrypointsPath;
    private string $assetsBaseUrl;

    public function __construct(string $projectDir, string $assetsBaseUrl)
    {
        // default Encore output: public/build/entrypoints.json
        $this->entrypointsPath = $projectDir . '/public/build/entrypoints.json';
        $this->assetsBaseUrl = rtrim($assetsBaseUrl, '/');
    }

    public function getFunctions(): array
    {
        return [
            // usage: {{ encore_entry_link_tags_dynamic('app') | raw }}
            new TwigFunction('encore_entry_link_tags_dynamic', [$this, 'renderLinkTags'], ['is_safe' => ['html']]),
            new TwigFunction('encore_entry_script_tags_dynamic', [$this, 'renderScriptTags'], ['is_safe' => ['html']]),
        ];
    }

    private function getBasePath(): string
    {
        return $this->assetsBaseUrl;
    }

    private function loadEntrypoints(): array
    {
        if (!file_exists($this->entrypointsPath)) {
            return [];
        }

        $json = json_decode((string) file_get_contents($this->entrypointsPath), true);
        return is_array($json) ? $json : [];
    }

    public function renderLinkTags(string $entryName): string
    {
        $data = $this->loadEntrypoints();
        $base = $this->getBasePath();

        $html = '';
        $cssFiles = $data['entrypoints'][$entryName]['css'] ?? [];
        foreach ($cssFiles as $file) {
            // normalize: remove any leading slash(s)
            $filePath = ltrim($file, '/');
            $url = $base . '/' . $filePath;
            $html .= sprintf('<link rel="stylesheet" href="%s">', htmlspecialchars($url, ENT_QUOTES));
            $html .= "\n";
        }

        return $html;
    }

    public function renderScriptTags(string $entryName): string
    {
        $data = $this->loadEntrypoints();
        $base = $this->getBasePath();

        $html = '';
        // runtime + files — mimic the normal order Encore uses
        $jsFiles = $data['entrypoints'][$entryName]['js'] ?? [];
        foreach ($jsFiles as $file) {
            $filePath = ltrim($file, '/');
            $url = $base . '/' . $filePath;
            $html .= sprintf('<script src="%s"></script>', htmlspecialchars($url, ENT_QUOTES));
            $html .= "\n";
        }

        return $html;
    }
}
