<?php

namespace App\Service;

use App\Entity\Household;
use App\Entity\Event;
use App\Entity\Guest;
use App\Entity\HouseholdEventRsvp;
use App\Entity\Rsvp;
use App\Repository\HouseholdEventRsvpRepository;
use App\Repository\RsvpRepository;
use Doctrine\ORM\EntityManagerInterface;

class RsvpConverterService
{
    public function __construct(
        private EntityManagerInterface $em,
        private HouseholdEventRsvpRepository $householdRsvpRepo,
        private RsvpRepository $rsvpRepo
    ) {
    }

    /**
     * Convert household RSVP to individual guest RSVPs
     */
    public function convertHouseholdToGuests(Household $household, Event $event): void
    {
        $householdRsvp = $this->householdRsvpRepo->findByHouseholdAndEvent($household, $event);
        
        if (!$householdRsvp) {
            return;
        }

        // Get or create guests
        $guests = $household->getGuests();
        $totalNeeded = $householdRsvp->getTotalCount();
        
        // Parse names if available
        $names = [];
        if ($householdRsvp->getNamesText()) {
            $names = array_map('trim', explode(',', $householdRsvp->getNamesText()));
        }

        // Create missing guests
        $guestCount = count($guests);
        if ($guestCount < $totalNeeded) {
            for ($i = $guestCount; $i < $totalNeeded; $i++) {
                $guest = new Guest();
                $guest->setHousehold($household);
                
                if (isset($names[$i])) {
                    $nameParts = explode(' ', $names[$i], 2);
                    $guest->setFirstName($nameParts[0]);
                    $guest->setLastName($nameParts[1] ?? '');
                } else {
                    $guest->setFirstName('Guest ' . ($i + 1));
                }
                
                $guest->setIsChild($i >= $householdRsvp->getAdultsCount());
                $guest->setSide($household->getSide());
                
                $this->em->persist($guest);
                $guests[] = $guest;
            }
            $this->em->flush();
        }

        // Distribute meal choices
        $mealCounts = $householdRsvp->getMealCountsJson() ?? [];
        $mealDistribution = [];
        foreach ($mealCounts as $code => $count) {
            for ($i = 0; $i < $count; $i++) {
                $mealDistribution[] = $code;
            }
        }

        // Create individual RSVPs
        $index = 0;
        foreach ($guests as $guest) {
            if ($index >= $totalNeeded) break;
            
            $rsvp = $this->rsvpRepo->findByGuestAndEvent($guest, $event);
            if (!$rsvp) {
                $rsvp = new Rsvp();
                $rsvp->setGuest($guest);
                $rsvp->setEvent($event);
            }
            
            $rsvp->setStatus($householdRsvp->getStatus());
            $rsvp->setMealChoice($mealDistribution[$index] ?? null);
            $rsvp->setDietary($householdRsvp->getDietary());
            $rsvp->setRespondedAt($householdRsvp->getRespondedAt());
            
            $this->em->persist($rsvp);
            $index++;
        }

        $this->em->flush();
    }

    /**
     * Convert individual guest RSVPs back to household RSVP
     */
    public function convertGuestsToHousehold(Household $household, Event $event): void
    {
        $guests = $household->getGuests();
        
        $adultsCount = 0;
        $childrenCount = 0;
        $mealCounts = [];
        $dietaryNotes = [];
        $latestResponse = null;
        $status = null;
        $yesCount = 0;
        $noCount = 0;

        foreach ($guests as $guest) {
            $rsvp = $this->rsvpRepo->findByGuestAndEvent($guest, $event);
            
            if (!$rsvp || !$rsvp->getStatus()) {
                continue;
            }

            if ($rsvp->getStatus() === 'YES') {
                $yesCount++;
                if ($guest->isChild()) {
                    $childrenCount++;
                } else {
                    $adultsCount++;
                }

                if ($rsvp->getMealChoice()) {
                    $mealCounts[$rsvp->getMealChoice()] = ($mealCounts[$rsvp->getMealChoice()] ?? 0) + 1;
                }

                if ($rsvp->getDietary()) {
                    $dietaryNotes[] = $guest->getFullName() . ': ' . $rsvp->getDietary();
                }
            } elseif ($rsvp->getStatus() === 'NO') {
                $noCount++;
            }

            if ($rsvp->getRespondedAt() && (!$latestResponse || $rsvp->getRespondedAt() > $latestResponse)) {
                $latestResponse = $rsvp->getRespondedAt();
            }
        }

        // Determine overall status
        if ($yesCount > 0) {
            $status = 'YES';
        } elseif ($noCount > 0) {
            $status = 'NO';
        }

        if (!$status) {
            return;
        }

        $householdRsvp = $this->householdRsvpRepo->findByHouseholdAndEvent($household, $event);
        if (!$householdRsvp) {
            $householdRsvp = new HouseholdEventRsvp();
            $householdRsvp->setHousehold($household);
            $householdRsvp->setEvent($event);
        }

        $householdRsvp->setStatus($status);
        $householdRsvp->setAdultsCount($adultsCount);
        $householdRsvp->setChildrenCount($childrenCount);
        $householdRsvp->setMealCountsJson($mealCounts);
        $householdRsvp->setDietary(implode('; ', $dietaryNotes));
        $householdRsvp->setRespondedAt($latestResponse);

        $this->em->persist($householdRsvp);
        $this->em->flush();
    }
}
