<?php

namespace App\Service;

use App\Entity\Event;
use App\Repository\HouseholdRepository;
use App\Repository\GuestRepository;
use App\Repository\HouseholdEventRsvpRepository;
use App\Repository\RsvpRepository;
use App\Repository\SettingRepository;

class ExportService
{
    public function __construct(
        private HouseholdRepository $householdRepo,
        private GuestRepository $guestRepo,
        private HouseholdEventRsvpRepository $householdRsvpRepo,
        private RsvpRepository $rsvpRepo,
        private SettingRepository $settingRepo
    ) {
    }

    public function generateCatererCsv(Event $event): array
    {
        $settings = $this->settingRepo->findFirst();
        $mode = $settings->getPublicRsvpMode();
        
        $mealCounts = [];

        if ($mode === 'PER_GUEST') {
            $rsvps = $this->rsvpRepo->findBy(['event' => $event, 'status' => 'YES']);
            foreach ($rsvps as $rsvp) {
                if ($rsvp->getMealChoice()) {
                    $mealCounts[$rsvp->getMealChoice()] = ($mealCounts[$rsvp->getMealChoice()] ?? 0) + 1;
                }
            }
        } else {
            $rsvps = $this->householdRsvpRepo->findBy(['event' => $event, 'status' => 'YES']);
            foreach ($rsvps as $rsvp) {
                $meals = $rsvp->getMealCountsJson() ?? [];
                foreach ($meals as $code => $count) {
                    $mealCounts[$code] = ($mealCounts[$code] ?? 0) + $count;
                }
            }
        }

        $rows = [['meal_code', 'meal_name', 'total_count']];
        
        $mealOptions = $event->getMealOptionsJson() ?? [];
        $mealNames = [];
        foreach ($mealOptions as $option) {
            $mealNames[$option['code']] = $option['name'];
        }

        foreach ($mealCounts as $code => $count) {
            $rows[] = [
                $code,
                $mealNames[$code] ?? $code,
                $count
            ];
        }

        return $rows;
    }

    public function generateDietaryCsv(Event $event): array
    {
        $settings = $this->settingRepo->findFirst();
        $mode = $settings->getPublicRsvpMode();
        
        $rows = [['household', 'notes']];

        if ($mode === 'PER_GUEST') {
            $rsvps = $this->rsvpRepo->findBy(['event' => $event]);
            foreach ($rsvps as $rsvp) {
                if ($rsvp->getDietary()) {
                    $rows[] = [
                        $rsvp->getGuest()->getHousehold()->getDisplayName(),
                        $rsvp->getGuest()->getFullName() . ': ' . $rsvp->getDietary()
                    ];
                }
            }
        } else {
            $rsvps = $this->householdRsvpRepo->findBy(['event' => $event]);
            foreach ($rsvps as $rsvp) {
                if ($rsvp->getDietary()) {
                    $rows[] = [
                        $rsvp->getHousehold()->getDisplayName(),
                        $rsvp->getDietary()
                    ];
                }
            }
        }

        return $rows;
    }

    public function generateAttendanceCsv(?Event $event = null): array
    {
        $settings = $this->settingRepo->findFirst();
        $mode = $settings->getPublicRsvpMode();

        if ($mode === 'PER_GUEST') {
            $rows = [['guest_first', 'guest_last', 'household', 'event', 'status', 'meal_choice', 'dietary', 'responded_at']];
            
            $criteria = $event ? ['event' => $event] : [];
            $rsvps = $this->rsvpRepo->findBy($criteria);
            
            foreach ($rsvps as $rsvp) {
                $rows[] = [
                    $rsvp->getGuest()->getFirstName(),
                    $rsvp->getGuest()->getLastName(),
                    $rsvp->getGuest()->getHousehold()->getDisplayName(),
                    $rsvp->getEvent()->getName(),
                    $rsvp->getStatus() ?? '',
                    $rsvp->getMealChoice() ?? '',
                    $rsvp->getDietary() ?? '',
                    $rsvp->getRespondedAt() ? $rsvp->getRespondedAt()->format('Y-m-d H:i:s') : ''
                ];
            }
        } else {
            $rows = [['household', 'event', 'status', 'adults', 'children', 'responded_at']];
            
            $criteria = $event ? ['event' => $event] : [];
            $rsvps = $this->householdRsvpRepo->findBy($criteria);
            
            foreach ($rsvps as $rsvp) {
                $rows[] = [
                    $rsvp->getHousehold()->getDisplayName(),
                    $rsvp->getEvent()->getName(),
                    $rsvp->getStatus() ?? '',
                    $rsvp->getAdultsCount(),
                    $rsvp->getChildrenCount(),
                    $rsvp->getRespondedAt() ? $rsvp->getRespondedAt()->format('Y-m-d H:i:s') : ''
                ];
            }
        }

        return $rows;
    }

    public function generateGuestsCsv(): array
    {
        $rows = [['household', 'first_name', 'last_name', 'email', 'phone', 'is_child', 'side']];
        
        $guests = $this->guestRepo->findAll();
        foreach ($guests as $guest) {
            $rows[] = [
                $guest->getHousehold()->getDisplayName(),
                $guest->getFirstName(),
                $guest->getLastName() ?? '',
                $guest->getEmail() ?? '',
                $guest->getPhone() ?? '',
                $guest->isChild() ? 'yes' : 'no',
                $guest->getSide()
            ];
        }

        return $rows;
    }

    public function generateHouseholdsCsv(): array
    {
        $rows = [['display_name', 'email', 'phone', 'address', 'side', 'token', 'invited_at']];
        
        $households = $this->householdRepo->findAll();
        foreach ($households as $household) {
            $rows[] = [
                $household->getDisplayName(),
                $household->getEmail() ?? '',
                $household->getPhone() ?? '',
                $household->getAddress() ?? '',
                $household->getSide(),
                $household->getToken(),
                $household->getInvitedAt() ? $household->getInvitedAt()->format('Y-m-d H:i:s') : ''
            ];
        }

        return $rows;
    }
}
