<?php

namespace App\Service;

use App\Entity\User;
use App\Repository\SettingRepository;
use Doctrine\ORM\EntityManagerInterface;
use Symfony\Component\Mailer\Mailer;
use Symfony\Component\Mailer\Transport;
use Symfony\Component\Mime\Email;
use Symfony\Component\Security\Core\Authentication\Token\Storage\TokenStorageInterface;
use Twig\Environment;

class EmailService
{
    public function __construct(
        private SettingRepository $settingRepo,
        private TokenStorageInterface $tokenStorage,
        private Environment $twig,
        private EntityManagerInterface $em
    ) {
    }

    public function sendInviteEmail(string $toEmail, string $toName, string $rsvpUrl, ?string $eventName = null, bool $isReminder = false, ?string $venueName = null): bool
    {
        $user = $this->getCurrentUser();
        $settings = $this->settingRepo->findFirst();
        
        // Check if SMTP is configured
        if (!$user->getSmtpHost() || !$user->getSmtpFromEmail()) {
            throw new \Exception('SMTP settings not configured. Please configure SMTP in your profile.');
        }

        // Build SMTP DSN
        $dsn = $this->buildSmtpDsn($user);
        
        try {
            $transport = Transport::fromDsn($dsn);
            $mailer = new Mailer($transport);

            // Get formatted data for variable replacement
            $coupleNames = $settings->getCoupleNames() ?: '';
            $eventDate = $settings->getWeddingDate()?->format('F j, Y') ?: '';
            $eventNameFormatted = $eventName ?: 'our wedding';
            
            // Helper function for variable replacement
            $replaceVars = function($text) use ($eventNameFormatted, $toName, $eventDate, $venueName, $coupleNames) {
                return str_replace(
                    ['{eventName}', '{guestName}', '{eventDate}', '{venueName}', '{coupleNames}'],
                    [$eventNameFormatted, $toName, $eventDate, $venueName ?? '', $coupleNames],
                    $text
                );
            };

            // Get subject and message content from settings with variable replacement
            if ($isReminder) {
                $subject = $replaceVars($settings->getReminderEmailSubject() ?: "Reminder: Please RSVP to Our Wedding");
                $message = $replaceVars($settings->getReminderEmailBody() ?: 
                    "<p>We noticed you haven't responded to our wedding invitation yet. We'd love to know if you can join us!</p><p>Please take a moment to RSVP so we can finalize our arrangements.</p>");
                $template = 'emails/reminder-email.html.twig';
            } else {
                $subject = $replaceVars($settings->getInviteEmailSubject() ?: 
                    ($eventName ? "You're Invited to {eventName}!" : "You're Invited to Our Wedding!"));
                $message = $replaceVars($settings->getInviteEmailBody() ?: 
                    "<p>We are delighted to invite you to {eventName}!</p>");
                $template = 'emails/invite-email.html.twig';
            }

            // Render email using Twig template
            $htmlBody = $this->twig->render($template, [
                'guestName' => $toName,
                'rsvpUrl' => $rsvpUrl,
                'subject' => $subject,
                'message' => $message,
                'coupleNames' => $coupleNames,
                'weddingDate' => $eventDate,
                'eventName' => $eventName,
                // Customizable options
                'buttonText' => $settings->getEmailButtonText() ?: 'RSVP Now',
                'buttonColor' => $settings->getEmailButtonColor() ?: '#c2185b',
                'callToAction' => $settings->getEmailCallToAction() ?: 'Please RSVP by clicking the button below:',
                'closingText' => $settings->getEmailClosingText() ?: "We can't wait to celebrate with you!",
                'signature' => $settings->getEmailSignature(),
            ]);

            $email = (new Email())
                ->to($toEmail)
                ->subject($subject)
                ->html($htmlBody);

            // Set From address with optional name
            if ($user->getSmtpFromName()) {
                $email->from(new \Symfony\Component\Mime\Address($user->getSmtpFromEmail(), $user->getSmtpFromName()));
            } else {
                $email->from($user->getSmtpFromEmail());
            }

            $mailer->send($email);
            
            return true;
        } catch (\Exception $e) {
            throw new \Exception('Failed to send email: ' . $e->getMessage());
        }
    }

    public function sendBulkInvites(array $recipients, string $baseUrl, ?string $eventSlug = null, bool $isReminder = false): array
    {
        $results = [
            'sent' => 0,
            'failed' => 0,
            'errors' => []
        ];

        foreach ($recipients as $recipient) {
            try {
                // Use provided rsvpUrl or build from baseUrl
                $url = $recipient['rsvpUrl'] ?? ($eventSlug 
                    ? $baseUrl . '/rsvp/' . $eventSlug
                    : $baseUrl . '/rsvp');

                $this->sendInviteEmail(
                    $recipient['email'],
                    $recipient['name'],
                    $url,
                    $recipient['eventName'] ?? null,
                    $isReminder,
                    $recipient['venueName'] ?? null
                );
                
                $results['sent']++;
            } catch (\Exception $e) {
                $results['failed']++;
                $results['errors'][] = [
                    'email' => $recipient['email'],
                    'error' => $e->getMessage()
                ];
            }
        }

        return $results;
    }

    public function testSmtpConnection(): array
    {
        try {
            $user = $this->getCurrentUser();
            $dsn = $this->buildSmtpDsn($user);
            $transport = Transport::fromDsn($dsn);
            
            // Try to create mailer - if DSN is invalid, it will throw exception
            new Mailer($transport);
            
            return ['success' => true, 'message' => 'SMTP connection successful!'];
        } catch (\Exception $e) {
            return ['success' => false, 'message' => 'SMTP connection failed: ' . $e->getMessage()];
        }
    }

    private function getCurrentUser(): User
    {
        $token = $this->tokenStorage->getToken();
        if (!$token || !$token->getUser() instanceof User) {
            throw new \Exception('User not authenticated');
        }
        return $token->getUser();
    }

    public function sendPasswordResetEmail(string $toEmail, string $resetUrl): bool
    {
        $settings = $this->settingRepo->findFirst();
        
        // For password reset, we need to get SMTP from the first user or system config
        // since the user requesting reset is not authenticated
        $adminUser = $this->getFirstAdminUser();
        
        if (!$adminUser || !$adminUser->getSmtpHost() || !$adminUser->getSmtpFromEmail()) {
            throw new \Exception('SMTP settings not configured. Please configure SMTP settings.');
        }

        $dsn = $this->buildSmtpDsn($adminUser);
        
        try {
            $transport = Transport::fromDsn($dsn);
            $mailer = new Mailer($transport);

            $subject = 'Password Reset Request';
            
            // Render email using Twig template
            $htmlBody = $this->twig->render('emails/password-reset.html.twig', [
                'resetUrl' => $resetUrl,
                'subject' => $subject,
                'systemName' => $settings?->getCoupleNames() ?? 'RSVP Manager'
            ]);

            $email = (new Email())
                ->to($toEmail)
                ->subject($subject)
                ->html($htmlBody);

            // Set From address
            if ($adminUser->getSmtpFromName()) {
                $email->from(new \Symfony\Component\Mime\Address($adminUser->getSmtpFromEmail(), $adminUser->getSmtpFromName()));
            } else {
                $email->from($adminUser->getSmtpFromEmail());
            }

            $mailer->send($email);
            
            return true;
        } catch (\Exception $e) {
            throw new \Exception('Failed to send password reset email: ' . $e->getMessage());
        }
    }

    private function getFirstAdminUser(): ?User
    {
        // Get the first user with admin role (or any user with SMTP configured)
        $users = $this->em->getRepository(User::class)->findAll();
        
        foreach ($users as $user) {
            if ($user->getSmtpHost() && $user->getSmtpFromEmail()) {
                return $user;
            }
        }
        
        return null;
    }

    public function sendTicketEmail(string $toEmail, string $toName, string $ticketUrl, string $eventName, ?string $eventDate = null, ?string $venueName = null): bool
    {
        $user = $this->getCurrentUser();
        $settings = $this->settingRepo->findFirst();
        
        // Check if SMTP is configured
        if (!$user->getSmtpHost() || !$user->getSmtpFromEmail()) {
            throw new \Exception('SMTP settings not configured. Please configure SMTP in your profile.');
        }

        // Build SMTP DSN
        $dsn = $this->buildSmtpDsn($user);
        
        try {
            $transport = Transport::fromDsn($dsn);
            $mailer = new Mailer($transport);

            // Helper function for variable replacement
            $coupleNames = $settings->getCoupleNames() ?: '';
            $replaceVars = function($text) use ($eventName, $toName, $eventDate, $venueName, $coupleNames) {
                return str_replace(
                    ['{eventName}', '{guestName}', '{eventDate}', '{venueName}', '{coupleNames}'],
                    [$eventName, $toName, $eventDate ?? '', $venueName ?? '', $coupleNames],
                    $text
                );
            };
            
            // Get all customizable text from settings with defaults
            $subject = $replaceVars($settings->getTicketEmailSubject() ?: "Your QR Code Ticket for {eventName}");
            $greeting = $replaceVars($settings->getTicketEmailGreeting() ?: "Dear {guestName},");
            $intro = $replaceVars($settings->getTicketEmailIntro() ?: "<p>Thank you for confirming your attendance! We're so excited to celebrate with you.</p>");
            $message = $replaceVars($settings->getTicketEmailBody() ?: "<p>Here is your QR code ticket for check-in:</p>");
            $buttonText = $replaceVars($settings->getTicketEmailButtonText() ?: "View My Ticket");
            $instructions = $replaceVars($settings->getTicketEmailInstructions() ?: null);
            $closing = $replaceVars($settings->getTicketEmailClosing() ?: "We can't wait to see you there!");
            $footer = $replaceVars($settings->getTicketEmailFooter() ?: "<p>This is your personal QR code ticket. Please do not share this link.</p>");

            // Render email using Twig template
            $htmlBody = $this->twig->render('emails/ticket-email.html.twig', [
                'guestName' => $toName,
                'ticketUrl' => $ticketUrl,
                'eventName' => $eventName,
                'eventDate' => $eventDate,
                'venueName' => $venueName,
                'coupleNames' => $settings->getCoupleNames(),
                'subject' => $subject,
                'greeting' => $greeting,
                'intro' => $intro,
                'message' => $message,
                'buttonText' => $buttonText,
                'instructions' => $instructions,
                'closing' => $closing,
                'footer' => $footer,
            ]);

            $email = (new Email())
                ->to($toEmail)
                ->subject($subject)
                ->html($htmlBody);

            // Set From address with optional name
            if ($user->getSmtpFromName()) {
                $email->from(new \Symfony\Component\Mime\Address($user->getSmtpFromEmail(), $user->getSmtpFromName()));
            } else {
                $email->from($user->getSmtpFromEmail());
            }

            $mailer->send($email);
            
            return true;
        } catch (\Exception $e) {
            throw new \Exception('Failed to send ticket email: ' . $e->getMessage());
        }
    }

    public function sendBulkTickets(array $recipients): array
    {
        $results = [
            'sent' => 0,
            'failed' => 0,
            'errors' => []
        ];

        foreach ($recipients as $recipient) {
            try {
                $this->sendTicketEmail(
                    $recipient['email'],
                    $recipient['name'],
                    $recipient['ticketUrl'],
                    $recipient['eventName'],
                    $recipient['eventDate'] ?? null,
                    $recipient['venueName'] ?? null
                );
                
                $results['sent']++;
            } catch (\Exception $e) {
                $results['failed']++;
                $results['errors'][] = [
                    'email' => $recipient['email'],
                    'name' => $recipient['name'],
                    'error' => $e->getMessage()
                ];
            }
        }

        return $results;
    }

    private function buildSmtpDsn(User $user): string
    {
        $encryption = $user->getSmtpEncryption() ?: 'tls';
        $port = $user->getSmtpPort() ?: 587;
        $username = $user->getSmtpUsername();
        $password = $user->getSmtpPassword();
        $host = $user->getSmtpHost();

        if ($username && $password) {
            return sprintf(
                'smtp://%s:%s@%s:%d?encryption=%s',
                urlencode($username),
                urlencode($password),
                $host,
                $port,
                $encryption
            );
        }

        return sprintf('smtp://%s:%d?encryption=%s', $host, $port, $encryption);
    }

}
