<?php

namespace App\DataFixtures;

use App\Entity\Event;
use App\Entity\Guest;
use App\Entity\Household;
use App\Entity\HouseholdEventRsvp;
use App\Entity\User;
use Doctrine\Bundle\FixturesBundle\Fixture;
use Doctrine\Persistence\ObjectManager;
use Symfony\Component\PasswordHasher\Hasher\UserPasswordHasherInterface;

class AppFixtures extends Fixture
{
    private UserPasswordHasherInterface $passwordHasher;

    public function __construct(UserPasswordHasherInterface $passwordHasher)
    {
        $this->passwordHasher = $passwordHasher;
    }

    public function load(ObjectManager $manager): void
    {
        // Create admin user (if not using installer)
        $admin = new User();
        $admin->setEmail('admin@example.com');
        $admin->setName('Admin User');
        $admin->setRoles(['ROLE_ADMIN']);
        $admin->setPassword($this->passwordHasher->hashPassword($admin, 'admin123'));
        $manager->persist($admin);

        // Create Events
        $ceremonyEvent = new Event();
        $ceremonyEvent->setName('Wedding Ceremony');
        $ceremonyEvent->setSlug('ceremony');
        $ceremonyEvent->setNameOne('Sarah');
        $ceremonyEvent->setNameTwo('Michael');
        $ceremonyEvent->setAndText('&');
        $ceremonyEvent->setVenueName('Grand Oak Chapel');
        $ceremonyEvent->setVenueAddress('123 Chapel Lane, Austin, TX 78701');
        $ceremonyEvent->setDescription('Join us as we say "I do" in an intimate ceremony surrounded by our loved ones.');
        $ceremonyEvent->setStartAt(new \DateTime('+2 months 14:00'));
        $ceremonyEvent->setEndAt(new \DateTime('+2 months 15:30'));
        $ceremonyEvent->setCapacity(150);
        $ceremonyEvent->setIsPublic(true);
        $ceremonyEvent->setDisplayOrder(1);
        $ceremonyEvent->setMealsEnabled(true);
        $ceremonyEvent->setMealOptionsJson([
            ['code' => 'beef', 'name' => 'Herb-Crusted Beef Tenderloin'],
            ['code' => 'chicken', 'name' => 'Lemon Rosemary Chicken'],
            ['code' => 'salmon', 'name' => 'Pan-Seared Salmon'],
            ['code' => 'vegan', 'name' => 'Roasted Vegetable Wellington']
        ]);
        $manager->persist($ceremonyEvent);

        $receptionEvent = new Event();
        $receptionEvent->setName('Reception & Dinner');
        $receptionEvent->setSlug('reception');
        $receptionEvent->setNameOne('Sarah');
        $receptionEvent->setNameTwo('Michael');
        $receptionEvent->setVenueName('Riverside Ballroom');
        $receptionEvent->setVenueAddress('456 River Road, Austin, TX 78701');
        $receptionEvent->setDescription('Celebrate with us at an evening of dinner, dancing, and joy!');
        $receptionEvent->setStartAt(new \DateTime('+2 months 18:00'));
        $receptionEvent->setEndAt(new \DateTime('+2 months 23:00'));
        $receptionEvent->setCapacity(200);
        $receptionEvent->setIsPublic(true);
        $receptionEvent->setDisplayOrder(2);
        $receptionEvent->setMealsEnabled(true);
        $receptionEvent->setMealOptionsJson([
            ['code' => 'beef', 'name' => 'Herb-Crusted Beef Tenderloin'],
            ['code' => 'chicken', 'name' => 'Lemon Rosemary Chicken'],
            ['code' => 'salmon', 'name' => 'Pan-Seared Salmon'],
            ['code' => 'vegan', 'name' => 'Roasted Vegetable Wellington']
        ]);
        $manager->persist($receptionEvent);

        // Create Households with Guests and RSVPs
        $householdData = [
            [
                'name' => 'The Johnson Family',
                'email' => 'johnson@example.com',
                'phone' => '555-0101',
                'side' => 'BRIDE',
                'address' => '789 Oak Street, Austin, TX 78702',
                'guests' => [
                    ['firstName' => 'James', 'lastName' => 'Johnson', 'isChild' => false],
                    ['firstName' => 'Emily', 'lastName' => 'Johnson', 'isChild' => false],
                    ['firstName' => 'Sophie', 'lastName' => 'Johnson', 'isChild' => true]
                ],
                'rsvpStatus' => 'YES',
                'mealChoices' => ['beef', 'chicken', 'chicken']
            ],
            [
                'name' => 'The Martinez Family',
                'email' => 'martinez@example.com',
                'phone' => '555-0102',
                'side' => 'GROOM',
                'address' => '321 Pine Avenue, Austin, TX 78703',
                'guests' => [
                    ['firstName' => 'Carlos', 'lastName' => 'Martinez', 'isChild' => false],
                    ['firstName' => 'Maria', 'lastName' => 'Martinez', 'isChild' => false]
                ],
                'rsvpStatus' => 'YES',
                'mealChoices' => ['salmon', 'vegan']
            ],
            [
                'name' => 'The Thompson Family',
                'email' => 'thompson@example.com',
                'phone' => '555-0103',
                'side' => 'BRIDE',
                'address' => '567 Maple Drive, Austin, TX 78704',
                'guests' => [
                    ['firstName' => 'Robert', 'lastName' => 'Thompson', 'isChild' => false],
                    ['firstName' => 'Jennifer', 'lastName' => 'Thompson', 'isChild' => false],
                    ['firstName' => 'Lucas', 'lastName' => 'Thompson', 'isChild' => true],
                    ['firstName' => 'Emma', 'lastName' => 'Thompson', 'isChild' => true]
                ],
                'rsvpStatus' => 'YES',
                'mealChoices' => ['beef', 'salmon', 'chicken', 'chicken']
            ],
            [
                'name' => 'The Davis Family',
                'email' => 'davis@example.com',
                'phone' => '555-0104',
                'side' => 'GROOM',
                'address' => '890 Cedar Lane, Austin, TX 78705',
                'guests' => [
                    ['firstName' => 'Michael', 'lastName' => 'Davis', 'isChild' => false],
                    ['firstName' => 'Sarah', 'lastName' => 'Davis', 'isChild' => false]
                ],
                'rsvpStatus' => 'NO',
                'mealChoices' => []
            ],
            [
                'name' => 'The Anderson Family',
                'email' => 'anderson@example.com',
                'phone' => '555-0105',
                'side' => 'BRIDE',
                'address' => '234 Birch Street, Austin, TX 78706',
                'guests' => [
                    ['firstName' => 'David', 'lastName' => 'Anderson', 'isChild' => false],
                    ['firstName' => 'Lisa', 'lastName' => 'Anderson', 'isChild' => false],
                    ['firstName' => 'Noah', 'lastName' => 'Anderson', 'isChild' => true]
                ],
                'rsvpStatus' => 'MAYBE',
                'mealChoices' => []
            ],
            [
                'name' => 'The Wilson Family',
                'email' => 'wilson@example.com',
                'phone' => '555-0106',
                'side' => 'GROOM',
                'address' => '678 Elm Avenue, Austin, TX 78707',
                'guests' => [
                    ['firstName' => 'Thomas', 'lastName' => 'Wilson', 'isChild' => false],
                    ['firstName' => 'Jessica', 'lastName' => 'Wilson', 'isChild' => false]
                ],
                'rsvpStatus' => 'YES',
                'mealChoices' => ['chicken', 'vegan']
            ],
            [
                'name' => 'The Garcia Family',
                'email' => 'garcia@example.com',
                'phone' => '555-0107',
                'side' => 'BRIDE',
                'address' => '912 Spruce Road, Austin, TX 78708',
                'guests' => [
                    ['firstName' => 'Jose', 'lastName' => 'Garcia', 'isChild' => false],
                    ['firstName' => 'Ana', 'lastName' => 'Garcia', 'isChild' => false],
                    ['firstName' => 'Isabella', 'lastName' => 'Garcia', 'isChild' => true]
                ],
                'rsvpStatus' => null, // Not yet responded
                'mealChoices' => []
            ],
            [
                'name' => 'The Brown Family',
                'email' => 'brown@example.com',
                'phone' => '555-0108',
                'side' => 'GROOM',
                'address' => '345 Walnut Place, Austin, TX 78709',
                'guests' => [
                    ['firstName' => 'Christopher', 'lastName' => 'Brown', 'isChild' => false],
                    ['firstName' => 'Amanda', 'lastName' => 'Brown', 'isChild' => false]
                ],
                'rsvpStatus' => 'YES',
                'mealChoices' => ['beef', 'salmon']
            ],
            [
                'name' => 'The Lee Family',
                'email' => 'lee@example.com',
                'phone' => '555-0109',
                'side' => 'BRIDE',
                'address' => '789 Cherry Drive, Austin, TX 78710',
                'guests' => [
                    ['firstName' => 'Kevin', 'lastName' => 'Lee', 'isChild' => false],
                    ['firstName' => 'Michelle', 'lastName' => 'Lee', 'isChild' => false],
                    ['firstName' => 'Ethan', 'lastName' => 'Lee', 'isChild' => true]
                ],
                'rsvpStatus' => null, // Not yet responded
                'mealChoices' => []
            ],
            [
                'name' => 'The Miller Family',
                'email' => 'miller@example.com',
                'phone' => '555-0110',
                'side' => 'GROOM',
                'address' => '456 Poplar Lane, Austin, TX 78711',
                'guests' => [
                    ['firstName' => 'Daniel', 'lastName' => 'Miller', 'isChild' => false]
                ],
                'rsvpStatus' => 'YES',
                'mealChoices' => ['chicken']
            ]
        ];

        foreach ($householdData as $data) {
            $household = new Household();
            $household->setDisplayName($data['name']);
            $household->setEmail($data['email']);
            $household->setPhone($data['phone']);
            $household->setSide($data['side']);
            $household->setAddress($data['address']);
            $household->setNumberOfAdults(count(array_filter($data['guests'], fn($g) => !$g['isChild'])));
            $household->setNumberOfChildren(count(array_filter($data['guests'], fn($g) => $g['isChild'])));
            
            // Set invited date for some households
            if (rand(0, 1)) {
                $household->setInvitedAt(new \DateTime('-' . rand(7, 30) . ' days'));
            }

            // Create guests for household
            foreach ($data['guests'] as $guestData) {
                $guest = new Guest();
                $guest->setFirstName($guestData['firstName']);
                $guest->setLastName($guestData['lastName']);
                $guest->setIsChild($guestData['isChild']);
                $guest->setSide($data['side']);
                $guest->setEmail($data['email']); // Use household email
                $household->addGuest($guest);
                $manager->persist($guest);
            }

            $manager->persist($household);

            // Create RSVPs for both events if status is set
            if ($data['rsvpStatus']) {
                foreach ([$ceremonyEvent, $receptionEvent] as $event) {
                    $rsvp = new HouseholdEventRsvp();
                    $rsvp->setHousehold($household);
                    $rsvp->setEvent($event);
                    $rsvp->setStatus($data['rsvpStatus']);
                    
                    if ($data['rsvpStatus'] === 'YES') {
                        $rsvp->setAdultsCount($household->getNumberOfAdults());
                        $rsvp->setChildrenCount($household->getNumberOfChildren());
                        $rsvp->setRespondedAt(new \DateTime('-' . rand(1, 20) . ' days'));
                        
                        // Add guest details with meal choices
                        $guestDetails = [];
                        $mealIndex = 0;
                        foreach ($household->getGuests() as $guest) {
                            $detail = [
                                'name' => $guest->getFullName(),
                                'meal' => $data['mealChoices'][$mealIndex] ?? 'chicken',
                                'dietary' => rand(0, 5) === 0 ? 'Gluten-free' : ''
                            ];
                            if ($guest->isChild()) {
                                $guestDetails['children'][] = $detail;
                            } else {
                                $guestDetails['adults'][] = $detail;
                            }
                            $mealIndex++;
                        }
                        $rsvp->setGuestDetailsJson($guestDetails);
                        
                        // Some have messages
                        if (rand(0, 2) === 0) {
                            $messages = [
                                'So excited to celebrate with you!',
                                'Can\'t wait for this special day!',
                                'Looking forward to the celebration!',
                                'Congratulations! See you there!'
                            ];
                            $rsvp->setMessage($messages[array_rand($messages)]);
                        }
                    } elseif ($data['rsvpStatus'] === 'NO') {
                        $rsvp->setRespondedAt(new \DateTime('-' . rand(1, 15) . ' days'));
                    }
                    
                    $manager->persist($rsvp);
                }
            }
        }

        $manager->flush();

        echo "\n✅ Demo data created successfully!\n";
        echo "   - 2 Events (Ceremony & Reception)\n";
        echo "   - 10 Households\n";
        echo "   - 25 Guests (including children)\n";
        echo "   - Various RSVP statuses (YES, NO, MAYBE, Not Responded)\n\n";
        echo "📧 Admin Login: admin@example.com / admin123\n\n";
    }
}
