<?php

namespace App\Controller;

use App\Repository\HouseholdRepository;
use App\Repository\GuestRepository;
use App\Repository\EventRepository;
use App\Service\QrService;
use Endroid\QrCode\QrCode;
use Endroid\QrCode\Writer\SvgWriter;
use Symfony\Bundle\FrameworkBundle\Controller\AbstractController;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\Routing\Attribute\Route;
use Symfony\Component\Routing\Generator\UrlGeneratorInterface;

class QrController extends AbstractController
{
    public function __construct(
        private HouseholdRepository $householdRepo,
        private GuestRepository $guestRepo,
        private EventRepository $eventRepo,
        private QrService $qrService
    ) {
    }

    #[Route('/q/{householdId}-{eventId}', name: 'qr_household')]
    public function householdTicket(int $householdId, int $eventId): Response
    {
        $household = $this->householdRepo->find($householdId);
        $event = $this->eventRepo->find($eventId);

        if (!$household || !$event) {
            throw $this->createNotFoundException('Household or Event not found');
        }

        return $this->render('qr/ticket.html.twig', [
            'household' => $household,
            'event' => $event,
            'type' => 'household',
        ]);
    }

    #[Route('/q/g/{guestId}-{eventId}', name: 'qr_guest')]
    public function guestTicket(int $guestId, int $eventId): Response
    {
        $guest = $this->guestRepo->find($guestId);
        $event = $this->eventRepo->find($eventId);

        if (!$guest || !$event) {
            throw $this->createNotFoundException('Guest or Event not found');
        }

        return $this->render('qr/ticket.html.twig', [
            'guest' => $guest,
            'event' => $event,
            'type' => 'guest',
        ]);
    }

    #[Route('/qr/image/h/{householdId}-{eventId}.svg', name: 'qr_household_image')]
    public function householdQrImage(int $householdId, int $eventId): Response
    {
        $household = $this->householdRepo->find($householdId);
        $event = $this->eventRepo->find($eventId);

        if (!$household || !$event) {
            throw $this->createNotFoundException('Household or Event not found');
        }

        // Generate the absolute URL that the QR code will point to
        $qrData = $this->generateUrl('qr_household', [
            'householdId' => $householdId,
            'eventId' => $eventId
        ], UrlGeneratorInterface::ABSOLUTE_URL);

        // Build QR code
        $qrCode = new QrCode($qrData);
        $writer = new SvgWriter();
        $result = $writer->write($qrCode);

        return new Response($result->getString(), 200, [
            'Content-Type' => 'image/svg+xml',
            'Cache-Control' => 'public, max-age=3600',
        ]);
    }

    #[Route('/qr/image/g/{guestId}-{eventId}.svg', name: 'qr_guest_image')]
    public function guestQrImage(int $guestId, int $eventId): Response
    {
        $guest = $this->guestRepo->find($guestId);
        $event = $this->eventRepo->find($eventId);

        if (!$guest || !$event) {
            throw $this->createNotFoundException('Guest or Event not found');
        }

        // Generate the absolute URL that the QR code will point to
        $qrData = $this->generateUrl('qr_guest', [
            'guestId' => $guestId,
            'eventId' => $eventId
        ], UrlGeneratorInterface::ABSOLUTE_URL);

        // Build QR code
        $qrCode = new QrCode($qrData);
        $writer = new SvgWriter();
        $result = $writer->write($qrCode);

        return new Response($result->getString(), 200, [
            'Content-Type' => 'image/svg+xml',
            'Cache-Control' => 'public, max-age=3600',
        ]);
    }
}
