<?php

namespace App\Controller;

use App\Entity\Event;
use App\Entity\Guest;
use App\Entity\Household;
use App\Entity\HouseholdEventRsvp;
use App\Entity\Rsvp;
use App\Repository\EventRepository;
use App\Repository\HouseholdRepository;
use App\Repository\HouseholdEventRsvpRepository;
use App\Repository\RsvpRepository;
use App\Repository\SettingRepository;
use App\Repository\ThemeSettingRepository;
use Doctrine\ORM\EntityManagerInterface;
use Symfony\Bundle\FrameworkBundle\Controller\AbstractController;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\RateLimiter\RateLimiterFactory;
use Symfony\Component\Routing\Attribute\Route;

class InviteController extends AbstractController
{
    public function __construct(
        private HouseholdRepository $householdRepo,
        private EventRepository $eventRepo,
        private SettingRepository $settingRepo,
        private HouseholdEventRsvpRepository $householdRsvpRepo,
        private RsvpRepository $rsvpRepo,
        private ThemeSettingRepository $themeRepo,
        private EntityManagerInterface $em,
        private RateLimiterFactory $rsvpSubmitLimiter
    ) {
    }

    // PUBLIC RSVP (No token required)
    #[Route('/rsvp', name: 'public_rsvp')]
    public function publicRsvp(): Response
    {
        return $this->render('invite/public.html.twig', [
            'eventSlug' => null
        ]);
    }

    #[Route('/rsvp/{slug}', name: 'public_rsvp_event', requirements: ['slug' => '[a-z0-9-]+'])]
    public function publicRsvpEvent(string $slug): Response
    {
        $event = $this->eventRepo->findOneBy(['slug' => $slug]);
        
        if (!$event) {
            throw $this->createNotFoundException('Event not found');
        }

        return $this->render('invite/public.html.twig', [
            'eventSlug' => $slug
        ]);
    }

    #[Route('/api/rsvp/public/events', name: 'api_public_events_all', methods: ['GET'])]
    public function getAllPublicEvents(): JsonResponse
    {
        $settings = $this->settingRepo->findFirst();
        $events = $this->eventRepo->findBy(['isPublic' => true], ['displayOrder' => 'ASC']);
        
        $data = [
            'settings' => [
                'coupleNames' => $settings ? $settings->getCoupleNames() : null,
                'weddingDate' => $settings?->getWeddingDate()?->format('Y-m-d'),
                'locationText' => $settings?->getLocationText(),
                'rsvpDeadline' => $settings?->getRsvpDeadline()?->format('Y-m-d H:i:s'),
            ],
            'events' => [],
        ];

        foreach ($events as $event) {
            $data['events'][] = [
                'id' => $event->getId(),
                'name' => $event->getName(),
                'slug' => $event->getSlug(),
                'description' => $event->getDescription(),
                'venueName' => $event->getVenueName(),
                'venueAddress' => $event->getVenueAddress(),
                'startAt' => $event->getStartAt()?->format('Y-m-d H:i:s'),
                'endAt' => $event->getEndAt()?->format('Y-m-d H:i:s'),
                'mealsEnabled' => $event->isMealsEnabled(),
                'mealOptions' => $event->getMealOptionsJson() ?? [],
                'nameFont' => $event->getNameFont() ?? 'Bodoni Moda',
                'nameOne' => $event->getNameOne(),
                'nameTwo' => $event->getNameTwo(),
                'andText' => $event->getAndText() ?? 'AND',
            ];
        }

        return $this->json($data);
    }

    #[Route('/api/rsvp/public/events/{slug}', name: 'api_public_events_by_slug', methods: ['GET'])]
    public function getPublicEventsBySlug(string $slug): JsonResponse
    {
        $settings = $this->settingRepo->findFirst();
        $event = $this->eventRepo->findOneBy(['slug' => $slug, 'isPublic' => true]);
        
        if (!$event) {
            return $this->json(['error' => 'Event not found'], 404);
        }
        
        $data = [
            'settings' => [
                'coupleNames' => $settings ? $settings->getCoupleNames() : null,
                'weddingDate' => $settings?->getWeddingDate()?->format('Y-m-d'),
                'locationText' => $settings?->getLocationText(),
                'rsvpDeadline' => $settings?->getRsvpDeadline()?->format('Y-m-d H:i:s'),
            ],
            'events' => [
                [
                    'id' => $event->getId(),
                    'name' => $event->getName(),
                    'slug' => $event->getSlug(),
                    'description' => $event->getDescription(),
                    'venueName' => $event->getVenueName(),
                    'venueAddress' => $event->getVenueAddress(),
                    'startAt' => $event->getStartAt()?->format('Y-m-d H:i:s'),
                    'endAt' => $event->getEndAt()?->format('Y-m-d H:i:s'),
                    'mealsEnabled' => $event->isMealsEnabled(),
                    'mealOptions' => $event->getMealOptionsJson() ?? [],
                    'nameFont' => $event->getNameFont() ?? 'Bodoni Moda',
                    'nameOne' => $event->getNameOne(),
                    'nameTwo' => $event->getNameTwo(),
                    'andText' => $event->getAndText() ?? 'AND',
                ]
            ],
        ];

        return $this->json($data);
    }

    #[Route('/api/rsvp/public/theme', name: 'api_public_theme', methods: ['GET'])]
    public function getPublicTheme(): JsonResponse
    {
        // Get active theme or return default
        $theme = $this->themeRepo->findOneBy(['isActive' => true]);
        
        if (!$theme) {
            // Return default theme matching current color scheme
            return $this->json([
                'bg' => '#2A0713',
                'ink' => '#C9B699',
                'inkSoft' => 'rgba(201,182,153,0.8)',
                'fieldBorder' => '#7B5F51',
                'accent' => '#B6A37F',
                'accentHover' => '#C8B588',
                'error' => '#E8CBB5'
            ]);
        }
        
        return $this->json([
            'bg' => $theme->getBg(),
            'ink' => $theme->getText(),
            'inkSoft' => 'rgba(' . $this->hexToRgb($theme->getText()) . ', 0.8)',
            'fieldBorder' => $theme->getFieldBorder() ?? '#7B5F51',
            'accent' => $theme->getAccent(),
            'accentHover' => $theme->getAccentHover() ?? $theme->getAccent(),
            'error' => $theme->getError() ?? '#E8CBB5'
        ]);
    }

    private function hexToRgb(string $hex): string
    {
        $hex = ltrim($hex, '#');
        $r = hexdec(substr($hex, 0, 2));
        $g = hexdec(substr($hex, 2, 2));
        $b = hexdec(substr($hex, 4, 2));
        return "$r,$g,$b";
    }

    #[Route('/api/rsvp/public/submit', name: 'api_public_submit', methods: ['POST'])]
    public function submitPublicRsvp(Request $request): JsonResponse
    {
        // Rate limiting
        $limiter = $this->rsvpSubmitLimiter->create($request->getClientIp());
        if (false === $limiter->consume(1)->isAccepted()) {
            return $this->json(['error' => 'Too many RSVP submissions. Please try again later.'], 429);
        }

        $settings = $this->settingRepo->findFirst();
        
        // Check deadline
        if ($settings?->getRsvpDeadline() && new \DateTime() > $settings->getRsvpDeadline()) {
            return $this->json(['error' => 'RSVP deadline has passed'], 400);
        }

        $data = json_decode($request->getContent(), true);
        
        // Create or find household
        $email = $data['email'] ?? null;
        $household = null;
        
        if ($email) {
            $household = $this->householdRepo->findOneBy(['email' => $email]);
        }
        
        if (!$household) {
            $household = new Household();
            $household->setDisplayName($data['name'] ?? 'Guest');
            $household->setEmail($data['email'] ?? null);
            $household->setPhone($data['phone'] ?? null);
            $household->setNumberOfAdults($data['numberOfAdults'] ?? null);
            $household->setNumberOfChildren($data['numberOfChildren'] ?? null);
            $household->regenerateToken();
            $this->em->persist($household);
        } else {
            // Update existing household
            if (isset($data['numberOfAdults'])) {
                $household->setNumberOfAdults($data['numberOfAdults']);
            }
            if (isset($data['numberOfChildren'])) {
                $household->setNumberOfChildren($data['numberOfChildren']);
            }
        }

        // Create primary contact guest
        $guest = new Guest();
        $guest->setHousehold($household);
        $guest->setFirstName($data['firstName'] ?? '');
        $guest->setLastName($data['lastName'] ?? '');
        $guest->setEmail($data['email'] ?? null);
        $guest->setPhone($data['phone'] ?? null);
        $this->em->persist($guest);

        // Process individual adult guests
        $adults = $data['adults'] ?? [];
        foreach ($adults as $adultData) {
            if (empty($adultData['name'])) continue;
            
            $adultGuest = new Guest();
            $adultGuest->setHousehold($household);
            $nameParts = explode(' ', $adultData['name'], 2);
            $adultGuest->setFirstName($nameParts[0] ?? '');
            $adultGuest->setLastName($nameParts[1] ?? '');
            $this->em->persist($adultGuest);
            
            // Create RSVP for each event
            foreach ($data['events'] ?? [] as $eventData) {
                $event = $this->eventRepo->find($eventData['eventId']);
                if (!$event) continue;
                
                $rsvp = new Rsvp();
                $rsvp->setGuest($adultGuest);
                $rsvp->setEvent($event);
                $rsvp->setStatus($eventData['status'] ?? 'YES');
                $rsvp->setMealChoice($adultData['meal'] ?? null);
                $rsvp->setDietary($adultData['dietary'] ?? null);
                $rsvp->setRespondedAt(new \DateTime());
                $this->em->persist($rsvp);
            }
        }
        
        // Process individual child guests
        $children = $data['children'] ?? [];
        foreach ($children as $childData) {
            if (empty($childData['name'])) continue;
            
            $childGuest = new Guest();
            $childGuest->setHousehold($household);
            $nameParts = explode(' ', $childData['name'], 2);
            $childGuest->setFirstName($nameParts[0] ?? '');
            $childGuest->setLastName($nameParts[1] ?? '');
            $this->em->persist($childGuest);
            
            // Create RSVP for each event
            foreach ($data['events'] ?? [] as $eventData) {
                $event = $this->eventRepo->find($eventData['eventId']);
                if (!$event) continue;
                
                $rsvp = new Rsvp();
                $rsvp->setGuest($childGuest);
                $rsvp->setEvent($event);
                $rsvp->setStatus($eventData['status'] ?? 'YES');
                $rsvp->setMealChoice($childData['meal'] ?? null);
                $rsvp->setDietary($childData['dietary'] ?? null);
                $rsvp->setRespondedAt(new \DateTime());
                $this->em->persist($rsvp);
            }
        }

        // Also store in household_event_rsvp for consistency
        $events = $data['events'] ?? [];
        if (!empty($events)) {
            $firstEvent = $this->eventRepo->find($events[0]['eventId']);
            if ($firstEvent) {
                $householdRsvp = new HouseholdEventRsvp();
                $householdRsvp->setHousehold($household);
                $householdRsvp->setEvent($firstEvent);
                $householdRsvp->setStatus($events[0]['status'] ?? 'YES');
                $householdRsvp->setAdultsCount(count($adults));
                $householdRsvp->setChildrenCount(count($children));
                $householdRsvp->setMessageToCouple($data['message'] ?? null);
                
                // Store detailed guest info
                if (!empty($adults) || !empty($children)) {
                    $guestDetails = [];
                    if (!empty($adults)) {
                        $guestDetails['adults'] = $adults;
                    }
                    if (!empty($children)) {
                        $guestDetails['children'] = $children;
                    }
                    $householdRsvp->setGuestDetailsJson($guestDetails);
                }
                
                $householdRsvp->setRespondedAt(new \DateTime());
                $this->em->persist($householdRsvp);
            }
        }

        $this->em->flush();

        return $this->json(['success' => true, 'message' => 'Thank you for your RSVP!']);
    }

    // TOKEN-BASED RSVP (Original system)
    #[Route('/invite/{token}', name: 'invite_show', requirements: ['token' => '[a-zA-Z0-9]{20,}'])]
    #[Route('/rsvp/{token}', name: 'invite_show_legacy', requirements: ['token' => '[a-zA-Z0-9]{20,}'])]
    public function show(string $token): Response
    {
        $household = $this->householdRepo->findByToken($token);
        
        if (!$household) {
            throw $this->createNotFoundException('Invitation not found');
        }

        // Mark as opened
        if (!$household->getOpenedAt()) {
            $household->setOpenedAt(new \DateTime());
            $this->em->flush();
        }

        return $this->render('invite/show.html.twig', [
            'token' => $token,
        ]);
    }

    #[Route('/api/rsvp/{token}', name: 'api_invite_data', methods: ['GET'])]
    public function getData(string $token, Request $request): JsonResponse
    {
        $household = $this->householdRepo->findByToken($token);
        
        if (!$household) {
            return $this->json(['error' => 'Invitation not found'], 404);
        }

        $settings = $this->settingRepo->findFirst();
        
        // Create default settings if none exist
        if (!$settings) {
            $settings = new \App\Entity\Setting();
            $settings->setPublicRsvpMode('HOUSEHOLD');
            $settings->setCoupleNames('');
            $settings->setStaffPin('0000');
            $settings->setLocaleDefault('en');
            $settings->setMaxPlusOnesPerHousehold(2);
            $this->em->persist($settings);
            $this->em->flush();
        }
        
        // Check if filtering by specific event
        $specificEventId = $request->query->get('event');
        
        // Try to get events, but don't fail if events table doesn't exist
        try {
            if ($specificEventId) {
                // Filter to specific event only
                $event = $this->eventRepo->find($specificEventId);
                $events = $event ? [$event] : [];
            } else {
                // Get all public events
                $events = $this->eventRepo->findPublicEvents();
            }
        } catch (\Exception $e) {
            // Events table might not exist yet, use empty array
            $events = [];
        }
        
        $mode = $settings->getPublicRsvpMode();

        $data = [
            'mode' => $mode,
            'household' => [
                'id' => $household->getId(),
                'displayName' => $household->getDisplayName(),
            ],
            'settings' => [
                'coupleNames' => $settings->getCoupleNames(),
                'weddingDate' => $settings->getWeddingDate()?->format('Y-m-d'),
                'locationText' => $settings->getLocationText(),
                'rsvpDeadline' => $settings->getRsvpDeadline()?->format('Y-m-d H:i:s'),
            ],
            'events' => [],
        ];

        foreach ($events as $event) {
            try {
                $eventData = [
                    'id' => $event->getId(),
                    'name' => $event->getName(),
                    'venueName' => $event->getVenueName(),
                    'venueAddress' => $event->getVenueAddress(),
                    'startAt' => $event->getStartAt()?->format('Y-m-d H:i:s'),
                    'mealsEnabled' => $event->isMealsEnabled(),
                    'mealOptions' => $event->getMealOptionsJson() ?? [],
                    'nameFont' => $event->getNameFont() ?? 'Bodoni Moda',
                    'nameOne' => $event->getNameOne(),
                    'nameTwo' => $event->getNameTwo(),
                    'andText' => $event->getAndText() ?? 'AND',
                ];

                if ($mode === 'PER_GUEST') {
                    $eventData['guests'] = [];
                    foreach ($household->getGuests() as $guest) {
                        $rsvp = $this->rsvpRepo->findByGuestAndEvent($guest, $event);
                        $eventData['guests'][] = [
                            'id' => $guest->getId(),
                            'firstName' => $guest->getFirstName(),
                            'lastName' => $guest->getLastName(),
                            'status' => $rsvp?->getStatus(),
                            'mealChoice' => $rsvp?->getMealChoice(),
                            'dietary' => $rsvp?->getDietary(),
                        ];
                    }
                } else {
                    $rsvp = $this->householdRsvpRepo->findByHouseholdAndEvent($household, $event);
                    $eventData['rsvp'] = [
                        'status' => $rsvp?->getStatus(),
                        'adultsCount' => $rsvp?->getAdultsCount() ?? 0,
                        'childrenCount' => $rsvp?->getChildrenCount() ?? 0,
                        'mealCounts' => $rsvp?->getMealCountsJson() ?? [],
                        'guestDetails' => $rsvp?->getGuestDetailsJson() ?? null,
                        'dietary' => $rsvp?->getDietary(),
                        'namesText' => $rsvp?->getNamesText(),
                    ];
                }

                $data['events'][] = $eventData;
            } catch (\Exception $e) {
                // Skip this event if there's an error processing it
                continue;
            }
        }

        // Try to get existing RSVP message (might fail if tables don't exist yet)
        try {
            if ($mode === 'PER_GUEST') {
                // Add message from any RSVP
                $firstRsvp = $household->getGuests()->count() > 0 
                    ? $this->rsvpRepo->findOneBy(['guest' => $household->getGuests()->first()])
                    : null;
                $data['message'] = ''; // Per-guest doesn't store message in Rsvp entity as spec'd
            } else {
                $firstRsvp = $this->householdRsvpRepo->findOneBy(['household' => $household]);
                $data['message'] = $firstRsvp?->getMessageToCouple() ?? '';
            }
        } catch (\Exception $e) {
            // RSVP tables might not exist yet
            $data['message'] = '';
        }

        return $this->json($data);
    }

    #[Route('/api/rsvp/{token}', name: 'api_invite_submit', methods: ['POST'])]
    public function submit(string $token, Request $request): JsonResponse
    {
        // Rate limiting
        $limiter = $this->rsvpSubmitLimiter->create($request->getClientIp());
        if (false === $limiter->consume(1)->isAccepted()) {
            return $this->json(['error' => 'Too many RSVP submissions. Please try again later.'], 429);
        }

        $household = $this->householdRepo->findByToken($token);
        
        if (!$household) {
            return $this->json(['error' => 'Invitation not found'], 404);
        }

        $settings = $this->settingRepo->findFirst();
        
        // Create default settings if none exist
        if (!$settings) {
            $settings = new \App\Entity\Setting();
            $settings->setPublicRsvpMode('HOUSEHOLD');
            $settings->setCoupleNames('');
            $settings->setStaffPin('0000');
            $settings->setLocaleDefault('en');
            $settings->setMaxPlusOnesPerHousehold(2);
            $this->em->persist($settings);
            $this->em->flush();
        }
        
        $mode = $settings->getPublicRsvpMode();
        
        // Check deadline
        if ($settings->getRsvpDeadline() && new \DateTime() > $settings->getRsvpDeadline()) {
            return $this->json(['error' => 'RSVP deadline has passed'], 400);
        }

        $data = json_decode($request->getContent(), true);
        
        // Update household contact information if provided
        if (isset($data['firstName']) || isset($data['lastName'])) {
            $displayName = trim(($data['firstName'] ?? '') . ' ' . ($data['lastName'] ?? ''));
            if ($displayName) {
                $household->setDisplayName($displayName);
            }
        }
        if (isset($data['email'])) {
            $household->setEmail($data['email']);
        }
        if (isset($data['phone'])) {
            $household->setPhone($data['phone']);
        }
        
        if ($mode === 'PER_GUEST') {
            $this->handlePerGuestSubmit($household, $data);
        } else {
            $this->handleHouseholdSubmit($household, $data, $mode);
        }

        return $this->json(['success' => true]);
    }

    private function handleHouseholdSubmit(object $household, array $data, string $mode): void
    {
        $events = $data['events'] ?? [];
        
        // Prepare guest details if provided
        $guestDetails = null;
        if (!empty($data['adults']) || !empty($data['children'])) {
            $guestDetails = [];
            if (!empty($data['adults'])) {
                $guestDetails['adults'] = $data['adults'];
            }
            if (!empty($data['children'])) {
                $guestDetails['children'] = $data['children'];
            }
        }
        
        foreach ($events as $eventData) {
            $event = $this->eventRepo->find($eventData['eventId']);
            if (!$event) continue;

            $rsvp = $this->householdRsvpRepo->findByHouseholdAndEvent($household, $event);
            if (!$rsvp) {
                $rsvp = new HouseholdEventRsvp();
                $rsvp->setHousehold($household);
                $rsvp->setEvent($event);
            }

            $rsvp->setStatus($eventData['status'] ?? null);
            $rsvp->setAdultsCount($eventData['adults'] ?? 0);
            $rsvp->setChildrenCount($eventData['children'] ?? 0);
            $rsvp->setMealCountsJson($eventData['meals'] ?? []);
            $rsvp->setRespondedAt(new \DateTime());
            
            // Save guest details on ALL event RSVPs (not just first)
            if ($guestDetails !== null) {
                $rsvp->setGuestDetailsJson($guestDetails);
            }
            
            // Save dietary and message
            $rsvp->setDietary($data['dietary'] ?? null);
            $rsvp->setMessageToCouple($data['message'] ?? null);
            if ($mode === 'HOUSEHOLD_WITH_OPTIONAL_NAMES') {
                $rsvp->setNamesText($data['namesText'] ?? null);
            }

            $this->em->persist($rsvp);
        }

        $this->em->flush();
    }

    private function handlePerGuestSubmit(object $household, array $data): void
    {
        $guests = $data['guests'] ?? [];
        
        foreach ($guests as $guestData) {
            $guest = $this->em->getRepository(\App\Entity\Guest::class)->find($guestData['guestId']);
            if (!$guest || $guest->getHousehold() !== $household) continue;

            $event = $this->eventRepo->find($guestData['eventId']);
            if (!$event) continue;

            $rsvp = $this->rsvpRepo->findByGuestAndEvent($guest, $event);
            if (!$rsvp) {
                $rsvp = new Rsvp();
                $rsvp->setGuest($guest);
                $rsvp->setEvent($event);
            }

            $rsvp->setStatus($guestData['status'] ?? null);
            $rsvp->setMealChoice($guestData['mealChoice'] ?? null);
            $rsvp->setDietary($guestData['dietary'] ?? null);
            $rsvp->setRespondedAt(new \DateTime());

            $this->em->persist($rsvp);
        }

        $this->em->flush();
    }
}
