<?php

namespace App\Controller;

use App\Entity\Event;
use App\Entity\Guest;
use App\Entity\Household;
use App\Entity\HouseholdEventRsvp;
use App\Entity\User;
use Doctrine\ORM\EntityManagerInterface;
use Symfony\Bundle\FrameworkBundle\Controller\AbstractController;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\PasswordHasher\Hasher\UserPasswordHasherInterface;
use Symfony\Component\Routing\Attribute\Route;

class DevController extends AbstractController
{
    #[Route('/dev/load-demo-data', name: 'dev_load_demo_data')]
    public function loadDemoData(
        EntityManagerInterface $em,
        UserPasswordHasherInterface $passwordHasher
    ): Response {
        // Check if already has data
        $existingHouseholds = $em->getRepository(Household::class)->count([]);
        if ($existingHouseholds > 0) {
            return new Response(
                '<h1>Demo Data Already Loaded</h1><p>There are already ' . $existingHouseholds . ' households in the database.</p><p><a href="/admin">Go to Admin</a></p>',
                Response::HTTP_OK,
                ['Content-Type' => 'text/html']
            );
        }

        // Create admin user if doesn't exist
        $adminUser = $em->getRepository(User::class)->findOneBy(['email' => 'admin@example.com']);
        if (!$adminUser) {
            $adminUser = new User();
            $adminUser->setEmail('admin@example.com');
            $adminUser->setRoles(['ROLE_ADMIN']);
            $adminUser->setPassword($passwordHasher->hashPassword($adminUser, 'admin123'));
            $em->persist($adminUser);
            $em->flush();
        }

        // Create Events
        $ceremony = new Event();
        $ceremony->setName('Wedding Ceremony');
        $ceremony->setSlug('ceremony');
        $ceremony->setNameOne('Sarah');
        $ceremony->setNameTwo('Michael');
        $ceremony->setAndText('&');
        $ceremony->setVenueName('Grand Oak Chapel');
        $ceremony->setVenueAddress('123 Chapel Lane, Austin, TX 78701');
        $ceremony->setDescription('Join us as we say "I do" in an intimate ceremony surrounded by our loved ones.');
        $ceremony->setStartAt(new \DateTime('+2 months'));
        $ceremony->setEndAt(new \DateTime('+2 months +90 minutes'));
        $ceremony->setCapacity(150);
        $ceremony->setIsPublic(true);
        $ceremony->setDisplayOrder(1);
        $ceremony->setMealsEnabled(true);
        $ceremony->setMealOptionsJson([
            ['code' => 'beef', 'name' => 'Herb-Crusted Beef Tenderloin'],
            ['code' => 'chicken', 'name' => 'Lemon Rosemary Chicken'],
            ['code' => 'salmon', 'name' => 'Pan-Seared Salmon'],
            ['code' => 'vegan', 'name' => 'Roasted Vegetable Wellington']
        ]);
        $ceremony->setNameFont('Bodoni Moda');
        $em->persist($ceremony);

        $reception = new Event();
        $reception->setName('Reception & Dinner');
        $reception->setSlug('reception');
        $reception->setNameOne('Sarah');
        $reception->setNameTwo('Michael');
        $reception->setAndText('&');
        $reception->setVenueName('Riverside Ballroom');
        $reception->setVenueAddress('456 River Road, Austin, TX 78701');
        $reception->setDescription('Celebrate with us at an evening of dinner, dancing, and joy!');
        $reception->setStartAt(new \DateTime('+2 months +4 hours'));
        $reception->setEndAt(new \DateTime('+2 months +9 hours'));
        $reception->setCapacity(200);
        $reception->setIsPublic(true);
        $reception->setDisplayOrder(2);
        $reception->setMealsEnabled(true);
        $reception->setMealOptionsJson([
            ['code' => 'beef', 'name' => 'Herb-Crusted Beef Tenderloin'],
            ['code' => 'chicken', 'name' => 'Lemon Rosemary Chicken'],
            ['code' => 'salmon', 'name' => 'Pan-Seared Salmon'],
            ['code' => 'vegan', 'name' => 'Roasted Vegetable Wellington']
        ]);
        $reception->setNameFont('Bodoni Moda');
        $em->persist($reception);

        $em->flush();

        // Create Households and Guests
        $householdsData = [
            ['name' => 'The Johnson Family', 'email' => 'johnson@example.com', 'phone' => '555-0101', 'address' => '789 Oak Street, Austin, TX 78702', 'side' => 'BRIDE', 'adults' => 2, 'children' => 1, 'status' => 'YES', 'guests' => [
                ['first' => 'James', 'last' => 'Johnson', 'child' => false, 'meal' => 'beef'],
                ['first' => 'Emily', 'last' => 'Johnson', 'child' => false, 'meal' => 'chicken'],
                ['first' => 'Sophie', 'last' => 'Johnson', 'child' => true, 'meal' => 'chicken'],
            ]],
            ['name' => 'The Martinez Family', 'email' => 'martinez@example.com', 'phone' => '555-0102', 'address' => '321 Pine Avenue, Austin, TX 78703', 'side' => 'GROOM', 'adults' => 2, 'children' => 0, 'status' => 'YES', 'guests' => [
                ['first' => 'Carlos', 'last' => 'Martinez', 'child' => false, 'meal' => 'salmon'],
                ['first' => 'Maria', 'last' => 'Martinez', 'child' => false, 'meal' => 'vegan'],
            ]],
            ['name' => 'The Thompson Family', 'email' => 'thompson@example.com', 'phone' => '555-0103', 'address' => '567 Maple Drive, Austin, TX 78704', 'side' => 'BRIDE', 'adults' => 2, 'children' => 2, 'status' => 'YES', 'guests' => [
                ['first' => 'Robert', 'last' => 'Thompson', 'child' => false, 'meal' => 'beef'],
                ['first' => 'Jennifer', 'last' => 'Thompson', 'child' => false, 'meal' => 'salmon'],
                ['first' => 'Lucas', 'last' => 'Thompson', 'child' => true, 'meal' => 'chicken'],
                ['first' => 'Emma', 'last' => 'Thompson', 'child' => true, 'meal' => 'chicken'],
            ]],
            ['name' => 'The Davis Family', 'email' => 'davis@example.com', 'phone' => '555-0104', 'address' => '890 Cedar Lane, Austin, TX 78705', 'side' => 'GROOM', 'adults' => 2, 'children' => 0, 'status' => 'NO', 'guests' => [
                ['first' => 'Michael', 'last' => 'Davis', 'child' => false],
                ['first' => 'Sarah', 'last' => 'Davis', 'child' => false],
            ]],
            ['name' => 'The Anderson Family', 'email' => 'anderson@example.com', 'phone' => '555-0105', 'address' => '234 Birch Street, Austin, TX 78706', 'side' => 'BRIDE', 'adults' => 2, 'children' => 1, 'status' => 'MAYBE', 'guests' => [
                ['first' => 'David', 'last' => 'Anderson', 'child' => false],
                ['first' => 'Lisa', 'last' => 'Anderson', 'child' => false],
                ['first' => 'Noah', 'last' => 'Anderson', 'child' => true],
            ]],
            ['name' => 'The Wilson Family', 'email' => 'wilson@example.com', 'phone' => '555-0106', 'address' => '678 Elm Avenue, Austin, TX 78707', 'side' => 'GROOM', 'adults' => 2, 'children' => 0, 'status' => 'YES', 'guests' => [
                ['first' => 'Thomas', 'last' => 'Wilson', 'child' => false, 'meal' => 'chicken'],
                ['first' => 'Jessica', 'last' => 'Wilson', 'child' => false, 'meal' => 'vegan'],
            ]],
            ['name' => 'The Garcia Family', 'email' => 'garcia@example.com', 'phone' => '555-0107', 'address' => '912 Spruce Road, Austin, TX 78708', 'side' => 'BRIDE', 'adults' => 2, 'children' => 1, 'status' => 'PENDING', 'guests' => [
                ['first' => 'Jose', 'last' => 'Garcia', 'child' => false],
                ['first' => 'Ana', 'last' => 'Garcia', 'child' => false],
                ['first' => 'Isabella', 'last' => 'Garcia', 'child' => true],
            ]],
            ['name' => 'The Brown Family', 'email' => 'brown@example.com', 'phone' => '555-0108', 'address' => '345 Walnut Place, Austin, TX 78709', 'side' => 'GROOM', 'adults' => 2, 'children' => 0, 'status' => 'YES', 'guests' => [
                ['first' => 'Christopher', 'last' => 'Brown', 'child' => false, 'meal' => 'beef'],
                ['first' => 'Amanda', 'last' => 'Brown', 'child' => false, 'meal' => 'salmon'],
            ]],
            ['name' => 'The Lee Family', 'email' => 'lee@example.com', 'phone' => '555-0109', 'address' => '789 Cherry Drive, Austin, TX 78710', 'side' => 'BRIDE', 'adults' => 2, 'children' => 1, 'status' => 'PENDING', 'guests' => [
                ['first' => 'Kevin', 'last' => 'Lee', 'child' => false],
                ['first' => 'Michelle', 'last' => 'Lee', 'child' => false],
                ['first' => 'Ethan', 'last' => 'Lee', 'child' => true],
            ]],
            ['name' => 'The Miller Family', 'email' => 'miller@example.com', 'phone' => '555-0110', 'address' => '456 Poplar Lane, Austin, TX 78711', 'side' => 'GROOM', 'adults' => 1, 'children' => 0, 'status' => 'YES', 'guests' => [
                ['first' => 'Daniel', 'last' => 'Miller', 'child' => false, 'meal' => 'chicken'],
            ]],
        ];

        $messages = [
            'YES' => ['So excited to celebrate with you!', 'Looking forward to the celebration!', "Can't wait!", 'Congratulations!'],
            'NO' => ["Sorry we can't make it. Wishing you all the best!"],
            'MAYBE' => ['Will confirm soon!'],
        ];

        foreach ($householdsData as $idx => $data) {
            $household = new Household();
            $household->setDisplayName($data['name']);
            $household->setEmail($data['email']);
            $household->setPhone($data['phone']);
            $household->setAddress($data['address']);
            $household->setSide($data['side']);
            $household->setLocale('en');
            $household->setToken(bin2hex(random_bytes(10)));
            $household->setNumberOfAdults($data['adults']);
            $household->setNumberOfChildren($data['children']);
            
            if ($data['status'] !== 'PENDING') {
                $household->setInvitedAt(new \DateTime('-' . rand(5, 25) . ' days'));
            }
            
            $em->persist($household);

            // Create guests
            foreach ($data['guests'] as $guestData) {
                $guest = new Guest();
                $guest->setHousehold($household);
                $guest->setFirstName($guestData['first']);
                $guest->setLastName($guestData['last']);
                $guest->setEmail($data['email']);
                $guest->setPhone($data['phone']);
                $guest->setIsChild($guestData['child']);
                $guest->setSide($data['side']);
                $em->persist($guest);
            }

            // Create RSVPs for both events
            if ($data['status'] !== 'PENDING') {
                $guestDetails = ['adults' => [], 'children' => []];
                foreach ($data['guests'] as $guestData) {
                    $detail = [
                        'name' => $guestData['first'] . ' ' . $guestData['last'],
                        'meal' => $guestData['meal'] ?? '',
                        'dietary' => ''
                    ];
                    if ($guestData['child']) {
                        $guestDetails['children'][] = $detail;
                    } else {
                        $guestDetails['adults'][] = $detail;
                    }
                }

                // Ceremony RSVP
                $rsvp1 = new HouseholdEventRsvp();
                $rsvp1->setHousehold($household);
                $rsvp1->setEvent($ceremony);
                $rsvp1->setStatus($data['status']);
                $rsvp1->setAdultsCount($data['status'] === 'YES' ? $data['adults'] : 0);
                $rsvp1->setChildrenCount($data['status'] === 'YES' ? $data['children'] : 0);
                if ($data['status'] !== 'PENDING') {
                    $rsvp1->setRespondedAt(new \DateTime('-' . rand(2, 20) . ' days'));
                }
                if (isset($messages[$data['status']])) {
                    $msgArray = $messages[$data['status']];
                    $rsvp1->setMessageToCouple($msgArray[array_rand($msgArray)]);
                }
                if ($data['status'] === 'YES') {
                    $rsvp1->setGuestDetailsJson($guestDetails);
                }
                $em->persist($rsvp1);

                // Reception RSVP
                $rsvp2 = new HouseholdEventRsvp();
                $rsvp2->setHousehold($household);
                $rsvp2->setEvent($reception);
                $rsvp2->setStatus($data['status']);
                $rsvp2->setAdultsCount($data['status'] === 'YES' ? $data['adults'] : 0);
                $rsvp2->setChildrenCount($data['status'] === 'YES' ? $data['children'] : 0);
                if ($data['status'] !== 'PENDING') {
                    $rsvp2->setRespondedAt(new \DateTime('-' . rand(2, 20) . ' days'));
                }
                if (isset($messages[$data['status']])) {
                    $msgArray = $messages[$data['status']];
                    $rsvp2->setMessageToCouple($msgArray[array_rand($msgArray)]);
                }
                if ($data['status'] === 'YES') {
                    $rsvp2->setGuestDetailsJson($guestDetails);
                }
                $em->persist($rsvp2);
            }
        }

        $em->flush();

        return new Response(
            '<html><body style="font-family: sans-serif; max-width: 600px; margin: 50px auto; padding: 20px;">
                <h1 style="color: #10b981;">✓ Demo Data Loaded Successfully!</h1>
                <h2>What was created:</h2>
                <ul>
                    <li><strong>2 Events:</strong> Wedding Ceremony & Reception</li>
                    <li><strong>10 Households</strong> with complete contact information</li>
                    <li><strong>25 Guests</strong> (adults and children)</li>
                    <li><strong>RSVP Status Mix:</strong>
                        <ul>
                            <li>6 households confirmed (YES)</li>
                            <li>1 household declined (NO)</li>
                            <li>1 household maybe (MAYBE)</li>
                            <li>2 households not responded (PENDING)</li>
                        </ul>
                    </li>
                </ul>
                <h2>Admin Credentials:</h2>
                <ul>
                    <li><strong>Email:</strong> admin@example.com</li>
                    <li><strong>Password:</strong> admin123</li>
                </ul>
                <p><a href="/admin" style="display: inline-block; background: #3b82f6; color: white; padding: 10px 20px; text-decoration: none; border-radius: 5px; margin-top: 20px;">Go to Admin Panel</a></p>
            </body></html>',
            Response::HTTP_OK,
            ['Content-Type' => 'text/html']
        );
    }
}
