<?php

namespace App\Controller\Api;

use App\Entity\Event;
use App\Entity\Guest;
use App\Entity\Household;
use App\Entity\HouseholdEventRsvp;
use App\Entity\Rsvp;
use App\Entity\Seating;
use App\Entity\Checkin;
use App\Entity\PageContent;
use App\Entity\ThemeSetting;
use App\Repository\EventRepository;
use App\Repository\GuestRepository;
use App\Repository\HouseholdRepository;
use App\Repository\HouseholdEventRsvpRepository;
use App\Repository\RsvpRepository;
use App\Repository\SettingRepository;
use App\Repository\SeatingRepository;
use App\Repository\CheckinRepository;
use App\Repository\PageContentRepository;
use App\Repository\ThemeSettingRepository;
use App\Service\RsvpConverterService;
use App\Service\ExportService;
use App\Service\EmailService;
use Doctrine\ORM\EntityManagerInterface;
use Symfony\Bundle\FrameworkBundle\Controller\AbstractController;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\HttpFoundation\StreamedResponse;
use Symfony\Component\Routing\Attribute\Route;
use Symfony\Component\Security\Http\Attribute\IsGranted;

#[Route('/api')]
#[IsGranted('ROLE_ADMIN')]
class AdminApiController extends AbstractController
{
    public function __construct(
        private EntityManagerInterface $em,
        private SettingRepository $settingRepo,
        private EventRepository $eventRepo,
        private HouseholdRepository $householdRepo,
        private GuestRepository $guestRepo,
        private HouseholdEventRsvpRepository $householdRsvpRepo,
        private RsvpRepository $rsvpRepo,
        private SeatingRepository $seatingRepo,
        private CheckinRepository $checkinRepo,
        private PageContentRepository $pageRepo,
        private ThemeSettingRepository $themeRepo,
        private RsvpConverterService $converter,
        private ExportService $exportService,
        private EmailService $emailService
    ) {
    }

    // Settings
    #[Route('/settings', methods: ['GET'])]
    public function getSettings(): JsonResponse
    {
        $settings = $this->settingRepo->findFirst();
        
        // Create default settings if none exist
        if (!$settings || !$settings->getId()) {
            $settings = new \App\Entity\Setting();
            $settings->setPublicRsvpMode('HOUSEHOLD');
            $settings->setCoupleNames('');
            $settings->setStaffPin('0000');
            $settings->setLocaleDefault('en');
            $settings->setMaxPlusOnesPerHousehold(2);
            
            // Set default email customization values
            $settings->setEmailButtonText('RSVP Now');
            $settings->setEmailButtonColor('#c2185b');
            $settings->setEmailCallToAction('Please RSVP by clicking the button below:');
            $settings->setEmailClosingText("We can't wait to celebrate with you!");
            
            $this->em->persist($settings);
            $this->em->flush();
        }
        
        return $this->json($settings);
    }
    
    // User SMTP Settings
    #[Route('/settings/smtp', methods: ['GET'])]
    public function getSmtpSettings(): JsonResponse
    {
        $user = $this->getUser();
        
        return $this->json([
            'smtpHost' => $user->getSmtpHost(),
            'smtpPort' => $user->getSmtpPort(),
            'smtpUsername' => $user->getSmtpUsername(),
            'smtpPassword' => $user->getSmtpPassword(),
            'smtpEncryption' => $user->getSmtpEncryption(),
            'smtpFromEmail' => $user->getSmtpFromEmail(),
            'smtpFromName' => $user->getSmtpFromName(),
        ]);
    }

    #[Route('/settings', methods: ['PATCH'])]
    public function updateSettings(Request $request): JsonResponse
    {
        $data = json_decode($request->getContent(), true);
        $settings = $this->settingRepo->findFirst();
        
        if (!$settings->getId()) {
            $this->em->persist($settings);
        }

        // Update system settings only
        if (isset($data['publicRsvpMode'])) $settings->setPublicRsvpMode($data['publicRsvpMode']);
        if (isset($data['coupleNames'])) $settings->setCoupleNames($data['coupleNames']);
        if (isset($data['weddingDate'])) {
            $settings->setWeddingDate($data['weddingDate'] ? new \DateTime($data['weddingDate']) : null);
        }
        if (isset($data['locationText'])) $settings->setLocationText($data['locationText']);
        if (isset($data['rsvpDeadline'])) {
            $settings->setRsvpDeadline($data['rsvpDeadline'] ? new \DateTime($data['rsvpDeadline']) : null);
        }
        if (isset($data['staffPin'])) $settings->setStaffPin($data['staffPin']);
        if (isset($data['baseUrl'])) $settings->setBaseUrl($data['baseUrl']);
        if (isset($data['localeDefault'])) $settings->setLocaleDefault($data['localeDefault']);
        if (isset($data['maxPlusOnesPerHousehold'])) $settings->setMaxPlusOnesPerHousehold($data['maxPlusOnesPerHousehold']);
        
        // Email templates
        if (isset($data['inviteEmailSubject'])) $settings->setInviteEmailSubject($data['inviteEmailSubject']);
        if (isset($data['inviteEmailBody'])) $settings->setInviteEmailBody($data['inviteEmailBody']);
        if (isset($data['reminderEmailSubject'])) $settings->setReminderEmailSubject($data['reminderEmailSubject']);
        if (isset($data['reminderEmailBody'])) $settings->setReminderEmailBody($data['reminderEmailBody']);
        
        // Email customization options
        if (isset($data['emailButtonText'])) $settings->setEmailButtonText($data['emailButtonText']);
        if (isset($data['emailButtonColor'])) $settings->setEmailButtonColor($data['emailButtonColor']);
        if (isset($data['emailCallToAction'])) $settings->setEmailCallToAction($data['emailCallToAction']);
        if (isset($data['emailClosingText'])) $settings->setEmailClosingText($data['emailClosingText']);
        if (isset($data['emailSignature'])) $settings->setEmailSignature($data['emailSignature']);
        
        // Ticket email templates
        if (isset($data['ticketEmailSubject'])) $settings->setTicketEmailSubject($data['ticketEmailSubject']);
        if (isset($data['ticketEmailBody'])) $settings->setTicketEmailBody($data['ticketEmailBody']);
        if (isset($data['ticketEmailGreeting'])) $settings->setTicketEmailGreeting($data['ticketEmailGreeting']);
        if (isset($data['ticketEmailIntro'])) $settings->setTicketEmailIntro($data['ticketEmailIntro']);
        if (isset($data['ticketEmailButtonText'])) $settings->setTicketEmailButtonText($data['ticketEmailButtonText']);
        if (isset($data['ticketEmailInstructions'])) $settings->setTicketEmailInstructions($data['ticketEmailInstructions']);
        if (isset($data['ticketEmailClosing'])) $settings->setTicketEmailClosing($data['ticketEmailClosing']);
        if (isset($data['ticketEmailFooter'])) $settings->setTicketEmailFooter($data['ticketEmailFooter']);

        $this->em->flush();

        return $this->json($settings);
    }
    
    #[Route('/settings/smtp', methods: ['PATCH'])]
    public function updateSmtpSettings(Request $request): JsonResponse
    {
        $data = json_decode($request->getContent(), true);
        $user = $this->getUser();

        // Update User's SMTP settings only
        if (isset($data['smtpHost'])) $user->setSmtpHost($data['smtpHost']);
        if (isset($data['smtpPort'])) $user->setSmtpPort((int)$data['smtpPort']);
        if (isset($data['smtpUsername'])) $user->setSmtpUsername($data['smtpUsername']);
        if (isset($data['smtpPassword'])) $user->setSmtpPassword($data['smtpPassword']);
        if (isset($data['smtpEncryption'])) $user->setSmtpEncryption($data['smtpEncryption']);
        if (isset($data['smtpFromEmail'])) $user->setSmtpFromEmail($data['smtpFromEmail']);
        if (isset($data['smtpFromName'])) $user->setSmtpFromName($data['smtpFromName']);

        $this->em->flush();

        return $this->json([
            'smtpHost' => $user->getSmtpHost(),
            'smtpPort' => $user->getSmtpPort(),
            'smtpUsername' => $user->getSmtpUsername(),
            'smtpPassword' => $user->getSmtpPassword(),
            'smtpEncryption' => $user->getSmtpEncryption(),
            'smtpFromEmail' => $user->getSmtpFromEmail(),
            'smtpFromName' => $user->getSmtpFromName(),
        ]);
    }

    // Events
    #[Route('/events', methods: ['GET'])]
    public function getEvents(): JsonResponse
    {
        $events = $this->eventRepo->findAll();
        $data = [];
        
        foreach ($events as $event) {
            $data[] = [
                'id' => $event->getId(),
                'name' => $event->getName(),
                'slug' => $event->getSlug(),
                'venueName' => $event->getVenueName(),
                'venueAddress' => $event->getVenueAddress(),
                'description' => $event->getDescription(),
                'startAt' => $event->getStartAt()?->format('Y-m-d H:i:s'),
                'endAt' => $event->getEndAt()?->format('Y-m-d H:i:s'),
                'capacity' => $event->getCapacity(),
                'isPublic' => $event->isPublic(),
                'order' => $event->getDisplayOrder(),
                'mealsEnabled' => $event->isMealsEnabled(),
                'mealOptions' => $event->getMealOptionsJson(),
                'nameFont' => $event->getNameFont() ?? 'Bodoni Moda',
                'nameOne' => $event->getNameOne(),
                'nameTwo' => $event->getNameTwo(),
                'andText' => $event->getAndText() ?? 'AND',
            ];
        }
        
        return $this->json($data);
    }

    #[Route('/events', methods: ['POST'])]
    public function createEvent(Request $request): JsonResponse
    {
        $data = json_decode($request->getContent(), true);
        $event = new Event();
        
        $this->updateEventFromData($event, $data);
        $this->em->persist($event);
        $this->em->flush();

        return $this->json($event, 201);
    }

    #[Route('/events/{id}', methods: ['GET'])]
    public function getEvent(int $id): JsonResponse
    {
        $event = $this->eventRepo->find($id);
        if (!$event) return $this->json(['error' => 'Not found'], 404);
        
        return $this->json($event);
    }

    #[Route('/events/{id}', methods: ['PATCH'])]
    public function updateEvent(int $id, Request $request): JsonResponse
    {
        $event = $this->eventRepo->find($id);
        if (!$event) return $this->json(['error' => 'Not found'], 404);

        $data = json_decode($request->getContent(), true);
        $this->updateEventFromData($event, $data);
        $this->em->flush();

        return $this->json($event);
    }

    #[Route('/events/{id}', methods: ['DELETE'])]
    public function deleteEvent(int $id): JsonResponse
    {
        $event = $this->eventRepo->find($id);
        if (!$event) return $this->json(['error' => 'Not found'], 404);

        $this->em->remove($event);
        $this->em->flush();

        return $this->json(['success' => true]);
    }

    // Households
    #[Route('/households', methods: ['GET'])]
    public function getHouseholds(): JsonResponse
    {
        $households = $this->householdRepo->findAll();
        $data = [];
        
        foreach ($households as $household) {
            // Get guests with their RSVP details
            $guestsData = [];
            foreach ($household->getGuests() as $guest) {
                $guestRsvps = $guest->getRsvps();
                $mealChoice = null;
                $dietary = null;
                $attending = null;
                
                // Get meal and dietary from first RSVP
                foreach ($guestRsvps as $rsvp) {
                    if ($rsvp->getMealChoice()) {
                        $mealChoice = $rsvp->getMealChoice();
                    }
                    if ($rsvp->getDietary()) {
                        $dietary = $rsvp->getDietary();
                    }
                    if ($rsvp->getStatus()) {
                        $attending = $rsvp->getStatus() === 'YES';
                    }
                }
                
                $guestsData[] = [
                    'firstName' => $guest->getFirstName(),
                    'lastName' => $guest->getLastName(),
                    'mealChoice' => $mealChoice,
                    'dietary' => $dietary,
                    'attending' => $attending
                ];
            }
            
            // Get event RSVPs
            $eventRsvps = [];
            $householdRsvps = $household->getRsvps();
            if ($householdRsvps) {
                foreach ($householdRsvps as $rsvp) {
                    $event = $rsvp->getEvent();
                    if (!$event) continue;
                    
                    $eventName = $event->getName() ?? 'Unnamed Event';
                    
                    // Use couple names if both are set
                    if ($event->getNameOne() && $event->getNameTwo()) {
                        $andText = $event->getAndText() ?: '&';
                        $eventName = $event->getNameOne() . ' ' . $andText . ' ' . $event->getNameTwo();
                    }
                    
                    $eventRsvps[] = [
                        'eventId' => $event->getId(),
                        'eventName' => $eventName,
                        'status' => $rsvp->getStatus()
                    ];
                }
            }
            
            $data[] = [
                'id' => $household->getId(),
                'displayName' => $household->getDisplayName(),
                'email' => $household->getEmail(),
                'phone' => $household->getPhone(),
                'side' => $household->getSide(),
                'tags' => $household->getTags(),
                'token' => $household->getToken(),
                'guestCount' => $household->getGuests()->count(),
                'adultsCount' => $household->getNumberOfAdults(),
                'childrenCount' => $household->getNumberOfChildren(),
                'guests' => $guestsData,
                'eventRsvps' => $eventRsvps,
                'invitedAt' => $household->getInvitedAt()?->format('Y-m-d H:i:s'),
                'openedAt' => $household->getOpenedAt()?->format('Y-m-d H:i:s'),
            ];
        }

        return $this->json($data);
    }

    #[Route('/households', methods: ['POST'])]
    public function createHousehold(Request $request): JsonResponse
    {
        $data = json_decode($request->getContent(), true);
        $household = new Household();
        
        $this->updateHouseholdFromData($household, $data);
        $this->em->persist($household);
        $this->em->flush();

        return $this->json(['id' => $household->getId()], 201);
    }

    #[Route('/households/{id}', methods: ['PATCH'])]
    public function updateHousehold(int $id, Request $request): JsonResponse
    {
        $household = $this->householdRepo->find($id);
        if (!$household) return $this->json(['error' => 'Not found'], 404);

        $data = json_decode($request->getContent(), true);
        $this->updateHouseholdFromData($household, $data);
        $this->em->flush();

        return $this->json(['success' => true]);
    }

    #[Route('/households/{id}', methods: ['DELETE'])]
    public function deleteHousehold(int $id): JsonResponse
    {
        $household = $this->householdRepo->find($id);
        if (!$household) return $this->json(['error' => 'Not found'], 404);

        $this->em->remove($household);
        $this->em->flush();

        return $this->json(['success' => true]);
    }

    #[Route('/households/{id}/regen-token', methods: ['POST'])]
    public function regenerateToken(int $id): JsonResponse
    {
        $household = $this->householdRepo->find($id);
        if (!$household) return $this->json(['error' => 'Not found'], 404);

        $household->regenerateToken();
        $this->em->flush();

        return $this->json(['token' => $household->getToken()]);
    }

    // Guests
    #[Route('/guests', methods: ['GET'])]
    public function getGuests(Request $request): JsonResponse
    {
        $householdId = $request->query->get('householdId');
        
        if ($householdId) {
            $household = $this->householdRepo->find($householdId);
            $guests = $household ? $household->getGuests()->toArray() : [];
        } else {
            $guests = $this->guestRepo->findAll();
        }

        $data = [];
        foreach ($guests as $guest) {
            $data[] = [
                'id' => $guest->getId(),
                'householdId' => $guest->getHousehold()->getId(),
                'householdName' => $guest->getHousehold()->getDisplayName(),
                'firstName' => $guest->getFirstName(),
                'lastName' => $guest->getLastName(),
                'email' => $guest->getEmail(),
                'phone' => $guest->getPhone(),
                'isChild' => $guest->isChild(),
                'side' => $guest->getSide(),
                'tags' => $guest->getTags(),
            ];
        }

        return $this->json($data);
    }

    #[Route('/guests', methods: ['POST'])]
    public function createGuest(Request $request): JsonResponse
    {
        $data = json_decode($request->getContent(), true);
        $household = $this->householdRepo->find($data['householdId']);
        if (!$household) return $this->json(['error' => 'Household not found'], 404);

        $guest = new Guest();
        $guest->setHousehold($household);
        $this->updateGuestFromData($guest, $data);
        
        $this->em->persist($guest);
        $this->em->flush();

        return $this->json(['id' => $guest->getId()], 201);
    }

    #[Route('/guests/{id}', methods: ['PATCH'])]
    public function updateGuest(int $id, Request $request): JsonResponse
    {
        $guest = $this->guestRepo->find($id);
        if (!$guest) return $this->json(['error' => 'Not found'], 404);

        $data = json_decode($request->getContent(), true);
        $this->updateGuestFromData($guest, $data);
        $this->em->flush();

        return $this->json(['success' => true]);
    }

    #[Route('/guests/{id}', methods: ['DELETE'])]
    public function deleteGuest(int $id): JsonResponse
    {
        $guest = $this->guestRepo->find($id);
        if (!$guest) return $this->json(['error' => 'Not found'], 404);

        $this->em->remove($guest);
        $this->em->flush();

        return $this->json(['success' => true]);
    }

    // RSVPs - For Check-In Page (includes check-in status)
    #[Route('/rsvps/event/{eventId}', methods: ['GET'])]
    public function getRsvpsByEvent(int $eventId): JsonResponse
    {
        $rsvps = $this->householdRsvpRepo->findBy(['event' => $eventId]);

        $data = [];
        foreach ($rsvps as $rsvp) {
            $data[] = [
                'id' => $rsvp->getId(),
                'householdId' => $rsvp->getHousehold()->getId(),
                'householdName' => $rsvp->getHousehold()->getDisplayName(),
                'householdEmail' => $rsvp->getHousehold()->getEmail(),
                'eventId' => $rsvp->getEvent()->getId(),
                'eventName' => $rsvp->getEvent()->getName(),
                'status' => $rsvp->getStatus(),
                'adultsCount' => $rsvp->getAdultsCount(),
                'childrenCount' => $rsvp->getChildrenCount(),
                'checkedIn' => $rsvp->isCheckedIn(),
                'checkedInAt' => $rsvp->getCheckedInAt()?->format('Y-m-d H:i:s'),
                'respondedAt' => $rsvp->getRespondedAt()?->format('Y-m-d H:i:s'),
            ];
        }

        return $this->json($data);
    }

    // RSVPs - Household Mode
    #[Route('/rsvps/household', methods: ['GET'])]
    public function getHouseholdRsvps(Request $request): JsonResponse
    {
        $eventId = $request->query->get('eventId');
        $criteria = $eventId ? ['event' => $eventId] : [];
        $rsvps = $this->householdRsvpRepo->findBy($criteria);

        $data = [];
        foreach ($rsvps as $rsvp) {
            $data[] = [
                'id' => $rsvp->getId(),
                'householdId' => $rsvp->getHousehold()->getId(),
                'householdName' => $rsvp->getHousehold()->getDisplayName(),
                'householdEmail' => $rsvp->getHousehold()->getEmail(),
                'eventId' => $rsvp->getEvent()->getId(),
                'eventName' => $rsvp->getEvent()->getName(),
                'status' => $rsvp->getStatus(),
                'adultsCount' => $rsvp->getAdultsCount(),
                'childrenCount' => $rsvp->getChildrenCount(),
                'mealCounts' => $rsvp->getMealCountsJson(),
                'guestDetails' => $rsvp->getGuestDetailsJson(),
                'dietary' => $rsvp->getDietary(),
                'namesText' => $rsvp->getNamesText(),
                'message' => $rsvp->getMessageToCouple(),
                'respondedAt' => $rsvp->getRespondedAt()?->format('Y-m-d H:i:s'),
            ];
        }

        return $this->json($data);
    }

    // RSVPs - Per Guest Mode
    #[Route('/rsvps/guest', methods: ['GET'])]
    public function getGuestRsvps(Request $request): JsonResponse
    {
        $eventId = $request->query->get('eventId');
        $criteria = $eventId ? ['event' => $eventId] : [];
        $rsvps = $this->rsvpRepo->findBy($criteria);

        $data = [];
        foreach ($rsvps as $rsvp) {
            $data[] = [
                'id' => $rsvp->getId(),
                'guestId' => $rsvp->getGuest()->getId(),
                'guestName' => $rsvp->getGuest()->getFullName(),
                'householdName' => $rsvp->getGuest()->getHousehold()->getDisplayName(),
                'eventId' => $rsvp->getEvent()->getId(),
                'eventName' => $rsvp->getEvent()->getName(),
                'status' => $rsvp->getStatus(),
                'mealChoice' => $rsvp->getMealChoice(),
                'dietary' => $rsvp->getDietary(),
                'respondedAt' => $rsvp->getRespondedAt()?->format('Y-m-d H:i:s'),
            ];
        }

        return $this->json($data);
    }

    // Converters
    #[Route('/convert/household-to-guests', methods: ['POST'])]
    public function convertHouseholdToGuests(Request $request): JsonResponse
    {
        $data = json_decode($request->getContent(), true);
        $household = $this->householdRepo->find($data['householdId']);
        $event = $this->eventRepo->find($data['eventId']);

        if (!$household || !$event) {
            return $this->json(['error' => 'Not found'], 404);
        }

        $this->converter->convertHouseholdToGuests($household, $event);

        return $this->json(['success' => true]);
    }

    // Update Guest Details
    #[Route('/rsvps/{id}/guest-details', methods: ['PUT'])]
    public function updateGuestDetails(int $id, Request $request): JsonResponse
    {
        $rsvp = $this->householdRsvpRepo->find($id);
        
        if (!$rsvp) {
            return $this->json(['error' => 'RSVP not found'], 404);
        }

        $data = json_decode($request->getContent(), true);
        
        $guestDetails = [];
        if (!empty($data['adults'])) {
            $guestDetails['adults'] = $data['adults'];
        }
        if (!empty($data['children'])) {
            $guestDetails['children'] = $data['children'];
        }
        
        $rsvp->setGuestDetailsJson($guestDetails);
        $this->em->flush();

        return $this->json(['success' => true]);
    }

    #[Route('/convert/guests-to-household', methods: ['POST'])]
    public function convertGuestsToHousehold(Request $request): JsonResponse
    {
        $data = json_decode($request->getContent(), true);
        $household = $this->householdRepo->find($data['householdId']);
        $event = $this->eventRepo->find($data['eventId']);

        if (!$household || !$event) {
            return $this->json(['error' => 'Not found'], 404);
        }

        $this->converter->convertGuestsToHousehold($household, $event);

        return $this->json(['success' => true]);
    }

    // Checkin
    #[Route('/checkin', methods: ['POST'])]
    public function checkin(Request $request): JsonResponse
    {
        $data = json_decode($request->getContent(), true);
        $event = $this->eventRepo->find($data['eventId']);
        
        if (!$event) return $this->json(['error' => 'Event not found'], 404);

        $checkin = new Checkin();
        $checkin->setEvent($event);
        $checkin->setMethod($data['method'] ?? 'MANUAL');
        $checkin->setCountDelta($data['countDelta'] ?? 1);

        if (isset($data['householdId'])) {
            $household = $this->householdRepo->find($data['householdId']);
            if ($household) $checkin->setHousehold($household);
        }

        if (isset($data['guestId'])) {
            $guest = $this->guestRepo->find($data['guestId']);
            if ($guest) $checkin->setGuest($guest);
        }

        $this->em->persist($checkin);
        $this->em->flush();

        return $this->json(['success' => true, 'id' => $checkin->getId()]);
    }

    // Exports
    #[Route('/export/caterer/{eventId}', methods: ['GET'])]
    public function exportCaterer(int $eventId): StreamedResponse
    {
        $event = $this->eventRepo->find($eventId);
        if (!$event) throw $this->createNotFoundException();

        return $this->streamCsv(
            $this->exportService->generateCatererCsv($event),
            'caterer-' . $event->getName() . '.csv'
        );
    }

    #[Route('/export/dietary/{eventId}', methods: ['GET'])]
    public function exportDietary(int $eventId): StreamedResponse
    {
        $event = $this->eventRepo->find($eventId);
        if (!$event) throw $this->createNotFoundException();

        return $this->streamCsv(
            $this->exportService->generateDietaryCsv($event),
            'dietary-' . $event->getName() . '.csv'
        );
    }

    #[Route('/export/attendance', methods: ['GET'])]
    public function exportAttendance(): StreamedResponse
    {
        return $this->streamCsv(
            $this->exportService->generateAttendanceCsv(),
            'attendance.csv'
        );
    }

    #[Route('/export/guests', methods: ['GET'])]
    public function exportGuests(): StreamedResponse
    {
        return $this->streamCsv(
            $this->exportService->generateGuestsCsv(),
            'guests.csv'
        );
    }

    #[Route('/export/households', methods: ['GET'])]
    public function exportHouseholds(): StreamedResponse
    {
        return $this->streamCsv(
            $this->exportService->generateHouseholdsCsv(),
            'households.csv'
        );
    }

    // Theme Settings
    #[Route('/themes', methods: ['GET'])]
    public function getThemes(): JsonResponse
    {
        return $this->json($this->themeRepo->findAll());
    }

    #[Route('/themes', methods: ['POST'])]
    public function createTheme(Request $request): JsonResponse
    {
        $data = json_decode($request->getContent(), true);
        $theme = new ThemeSetting();
        $this->updateThemeFromData($theme, $data);
        
        $this->em->persist($theme);
        $this->em->flush();

        return $this->json($theme, 201);
    }

    #[Route('/themes/{id}', methods: ['PATCH'])]
    public function updateTheme(int $id, Request $request): JsonResponse
    {
        $theme = $this->themeRepo->find($id);
        if (!$theme) return $this->json(['error' => 'Not found'], 404);

        $data = json_decode($request->getContent(), true);
        $this->updateThemeFromData($theme, $data);
        $this->em->flush();

        return $this->json($theme);
    }

    // Page Content
    #[Route('/pages', methods: ['GET'])]
    public function getPages(): JsonResponse
    {
        return $this->json($this->pageRepo->findAll());
    }

    #[Route('/pages', methods: ['POST'])]
    public function createPage(Request $request): JsonResponse
    {
        $data = json_decode($request->getContent(), true);
        $page = new PageContent();
        $this->updatePageFromData($page, $data);
        
        $this->em->persist($page);
        $this->em->flush();

        return $this->json($page, 201);
    }

    #[Route('/pages/{id}', methods: ['PATCH'])]
    public function updatePage(int $id, Request $request): JsonResponse
    {
        $page = $this->pageRepo->find($id);
        if (!$page) return $this->json(['error' => 'Not found'], 404);

        $data = json_decode($request->getContent(), true);
        $this->updatePageFromData($page, $data);
        $this->em->flush();

        return $this->json($page);
    }

    // Dashboard Stats
    #[Route('/dashboard/stats', methods: ['GET'])]
    public function getDashboardStats(): JsonResponse
    {
        $households = $this->householdRepo->findAll();
        $totalInvited = count($households);
        $responded = 0;
        $yesCount = 0;
        $noCount = 0;

        $settings = $this->settingRepo->findFirst();
        $mode = $settings->getPublicRsvpMode();

        if ($mode === 'PER_GUEST') {
            $allRsvps = $this->rsvpRepo->findAll();
            $respondedGuests = [];
            foreach ($allRsvps as $rsvp) {
                if ($rsvp->getStatus()) {
                    $respondedGuests[$rsvp->getGuest()->getHousehold()->getId()] = true;
                    if ($rsvp->getStatus() === 'YES') $yesCount++;
                    if ($rsvp->getStatus() === 'NO') $noCount++;
                }
            }
            $responded = count($respondedGuests);
        } else {
            $allRsvps = $this->householdRsvpRepo->findAll();
            $respondedHouseholds = [];
            foreach ($allRsvps as $rsvp) {
                if ($rsvp->getStatus()) {
                    $respondedHouseholds[$rsvp->getHousehold()->getId()] = true;
                    if ($rsvp->getStatus() === 'YES') $yesCount++;
                    if ($rsvp->getStatus() === 'NO') $noCount++;
                }
            }
            $responded = count($respondedHouseholds);
        }

        return $this->json([
            'totalInvited' => $totalInvited,
            'responded' => $responded,
            'responseRate' => $totalInvited > 0 ? round(($responded / $totalInvited) * 100, 1) : 0,
            'yesCount' => $yesCount,
            'noCount' => $noCount,
        ]);
    }

    // Helper methods
    private function updateEventFromData(Event $event, array $data): void
    {
        if (isset($data['name'])) $event->setName($data['name']);
        if (isset($data['slug'])) $event->setSlug($data['slug']);
        if (isset($data['venueName'])) $event->setVenueName($data['venueName']);
        if (isset($data['venueAddress'])) $event->setVenueAddress($data['venueAddress']);
        if (isset($data['description'])) $event->setDescription($data['description']);
        if (isset($data['startAt'])) $event->setStartAt($data['startAt'] ? new \DateTime($data['startAt']) : null);
        if (isset($data['endAt'])) $event->setEndAt($data['endAt'] ? new \DateTime($data['endAt']) : null);
        if (isset($data['capacity'])) $event->setCapacity($data['capacity'] ?: null);
        if (isset($data['isPublic'])) $event->setIsPublic($data['isPublic']);
        if (isset($data['order'])) $event->setDisplayOrder($data['order']);
        if (isset($data['mealsEnabled'])) $event->setMealsEnabled($data['mealsEnabled']);
        if (isset($data['mealOptionsJson'])) $event->setMealOptionsJson($data['mealOptionsJson']);
        if (isset($data['nameFont'])) $event->setNameFont($data['nameFont']);
        if (isset($data['nameOne'])) $event->setNameOne($data['nameOne']);
        if (isset($data['nameTwo'])) $event->setNameTwo($data['nameTwo']);
        if (isset($data['andText'])) $event->setAndText($data['andText']);
        if (isset($data['mealOptions'])) $event->setMealOptionsJson($data['mealOptions']);
    }

    private function updateHouseholdFromData(Household $household, array $data): void
    {
        if (isset($data['displayName'])) $household->setDisplayName($data['displayName']);
        if (isset($data['email'])) $household->setEmail($data['email']);
        if (isset($data['phone'])) $household->setPhone($data['phone']);
        if (isset($data['address'])) $household->setAddress($data['address']);
        if (isset($data['side'])) $household->setSide($data['side']);
        if (isset($data['tags'])) $household->setTags($data['tags']);
        if (isset($data['locale'])) $household->setLocale($data['locale']);
        if (isset($data['notes'])) $household->setNotes($data['notes']);
        if (isset($data['adultsCount'])) $household->setNumberOfAdults($data['adultsCount']);
        if (isset($data['childrenCount'])) $household->setNumberOfChildren($data['childrenCount']);
    }

    private function updateGuestFromData(Guest $guest, array $data): void
    {
        if (isset($data['firstName'])) $guest->setFirstName($data['firstName']);
        if (isset($data['lastName'])) $guest->setLastName($data['lastName']);
        if (isset($data['email'])) $guest->setEmail($data['email']);
        if (isset($data['phone'])) $guest->setPhone($data['phone']);
        if (isset($data['isChild'])) $guest->setIsChild($data['isChild']);
        if (isset($data['side'])) $guest->setSide($data['side']);
        if (isset($data['tags'])) $guest->setTags($data['tags']);
        if (isset($data['notes'])) $guest->setNotes($data['notes']);
    }

    private function updateThemeFromData(ThemeSetting $theme, array $data): void
    {
        if (isset($data['name'])) $theme->setName($data['name']);
        if (isset($data['primary'])) $theme->setPrimaryColor($data['primary']);
        if (isset($data['accent'])) $theme->setAccent($data['accent']);
        if (isset($data['bg'])) $theme->setBg($data['bg']);
        if (isset($data['text'])) $theme->setText($data['text']);
        if (isset($data['fontHeading'])) $theme->setFontHeading($data['fontHeading']);
        if (isset($data['fontBody'])) $theme->setFontBody($data['fontBody']);
        if (isset($data['fieldBorder'])) $theme->setFieldBorder($data['fieldBorder']);
        if (isset($data['accentHover'])) $theme->setAccentHover($data['accentHover']);
        if (isset($data['error'])) $theme->setError($data['error']);
        
        // If setting this theme as active, deactivate all others
        if (isset($data['isActive']) && $data['isActive'] === true) {
            $allThemes = $this->themeRepo->findAll();
            foreach ($allThemes as $t) {
                $t->setIsActive(false);
            }
            $theme->setIsActive(true);
        }
    }

    // Email/Invites endpoints
    #[Route('/email/test-smtp', methods: ['POST'])]
    public function testSmtp(): JsonResponse
    {
        $result = $this->emailService->testSmtpConnection();
        return $this->json($result);
    }

    #[Route('/email/send-invite', methods: ['POST'])]
    public function sendInvite(Request $request): JsonResponse
    {
        $data = json_decode($request->getContent(), true);
        $settings = $this->settingRepo->findFirst();
        
        try {
            $baseUrl = $settings->getBaseUrl() ?: $request->getSchemeAndHttpHost();
            $rsvpUrl = $baseUrl . '/rsvp/' . ($data['eventSlug'] ?? '');
            
            $this->emailService->sendInviteEmail(
                $data['email'],
                $data['name'],
                $rsvpUrl,
                $data['eventName'] ?? null,
                false, // isReminder
                $data['venueName'] ?? null
            );
            
            return $this->json(['success' => true, 'message' => 'Invite sent successfully!']);
        } catch (\Exception $e) {
            return $this->json(['success' => false, 'message' => $e->getMessage()], 400);
        }
    }

    #[Route('/email/send-bulk', methods: ['POST'])]
    public function sendBulkInvites(Request $request): JsonResponse
    {
        $data = json_decode($request->getContent(), true);
        $settings = $this->settingRepo->findFirst();
        
        try {
            $baseUrl = $settings->getBaseUrl() ?: $request->getSchemeAndHttpHost();
            
            // Fetch event details if eventId provided
            $event = null;
            if (isset($data['eventId'])) {
                $event = $this->eventRepo->find($data['eventId']);
            }
            
            // Add event name to each recipient
            $recipients = $data['recipients'];
            if ($event) {
                // Use couples' names if both are provided, otherwise use event name
                $eventDisplayName = $event->getName();
                if ($event->getNameOne() && $event->getNameTwo()) {
                    $andText = $event->getAndText() ?: 'AND';
                    $eventDisplayName = $event->getNameOne() . ' ' . $andText . ' ' . $event->getNameTwo();
                }
                
                foreach ($recipients as &$recipient) {
                    $recipient['eventName'] = $eventDisplayName;
                }
            }
            
            $results = $this->emailService->sendBulkInvites(
                $recipients,
                $baseUrl,
                $event?->getSlug(),
                $data['isReminder'] ?? false
            );
            
            // Update invitedAt timestamp for successfully sent invites
            if (!($data['isReminder'] ?? false) && $results['sent'] > 0) {
                foreach ($data['recipients'] as $recipient) {
                    if (isset($recipient['id'])) {
                        $household = $this->householdRepo->find($recipient['id']);
                        if ($household && !$household->getInvitedAt()) {
                            $household->setInvitedAt(new \DateTime());
                        }
                    }
                }
                $this->em->flush();
            }
            
            return $this->json($results);
        } catch (\Exception $e) {
            return $this->json(['success' => false, 'message' => $e->getMessage()], 400);
        }
    }

    #[Route('/email/send-tickets', methods: ['POST'])]
    public function sendBulkTickets(Request $request): JsonResponse
    {
        $data = json_decode($request->getContent(), true);
        $settings = $this->settingRepo->findFirst();
        
        try {
            $baseUrl = $settings->getBaseUrl() ?: $request->getSchemeAndHttpHost();
            
            // Fetch event details
            $event = null;
            if (isset($data['eventId'])) {
                $event = $this->eventRepo->find($data['eventId']);
            }
            
            if (!$event) {
                return $this->json(['success' => false, 'message' => 'Event not found'], 400);
            }
            
            // Prepare recipients with ticket URLs
            $recipients = [];
            foreach ($data['recipients'] as $recipient) {
                if (!isset($recipient['id']) || !isset($recipient['email'])) {
                    continue;
                }
                
                $ticketUrl = $baseUrl . '/q/' . $recipient['id'] . '-' . $event->getId();
                
                // Use couple names if both are provided, otherwise use event name
                $eventName = ($event->getNameOne() && $event->getNameTwo())
                    ? $event->getNameOne() . ' ' . ($event->getAndText() ?: 'AND') . ' ' . $event->getNameTwo()
                    : $event->getName();
                
                $recipients[] = [
                    'email' => $recipient['email'],
                    'name' => $recipient['name'] ?? 'Guest',
                    'ticketUrl' => $ticketUrl,
                    'eventName' => $eventName,
                    'eventDate' => $event->getStartAt()?->format('F j, Y g:i A'),
                    'venueName' => $event->getVenueName()
                ];
            }
            
            if (empty($recipients)) {
                return $this->json(['success' => false, 'message' => 'No valid recipients provided'], 400);
            }
            
            $results = $this->emailService->sendBulkTickets($recipients);
            
            return $this->json($results);
        } catch (\Exception $e) {
            return $this->json(['success' => false, 'message' => $e->getMessage()], 400);
        }
    }

    private function updatePageFromData(PageContent $page, array $data): void
    {
        if (isset($data['slug'])) $page->setSlug($data['slug']);
        if (isset($data['title'])) $page->setTitle($data['title']);
        if (isset($data['body'])) $page->setBody($data['body']);
        if (isset($data['published'])) $page->setPublished($data['published']);
    }

    private function streamCsv(array $rows, string $filename): StreamedResponse
    {
        $response = new StreamedResponse(function() use ($rows) {
            $handle = fopen('php://output', 'w');
            foreach ($rows as $row) {
                fputcsv($handle, $row);
            }
            fclose($handle);
        });

        $response->headers->set('Content-Type', 'text/csv');
        $response->headers->set('Content-Disposition', 'attachment; filename="' . $filename . '"');

        return $response;
    }

    #[Route('/qr-checkin/{householdId}/{eventId}', methods: ['POST'])]
    public function qrCheckIn(int $householdId, int $eventId): JsonResponse
    {
        $rsvp = $this->householdRsvpRepo->findOneBy([
            'household' => $householdId,
            'event' => $eventId
        ]);
        
        if (!$rsvp) {
            return $this->json(['error' => 'RSVP not found'], 404);
        }
        
        if ($rsvp->getStatus() !== 'attending') {
            return $this->json(['error' => 'Not marked as attending'], 400);
        }
        
        if ($rsvp->isCheckedIn()) {
            return $this->json([
                'message' => 'Already checked in',
                'household' => [
                    'name' => $rsvp->getHousehold()->getDisplayName(),
                    'guests' => $rsvp->getTotalCount(),
                    'email' => $rsvp->getHousehold()->getEmail()
                ],
                'checked_in_at' => $rsvp->getCheckedInAt()->format('Y-m-d H:i:s')
            ]);
        }
        
        // Mark as checked in
        $rsvp->setCheckedIn(true);
        $rsvp->setCheckedInAt(new \DateTime());
        $rsvp->setCheckedInByUser($this->getUser());
        
        $this->em->flush();
        
        return $this->json([
            'success' => true,
            'household' => [
                'name' => $rsvp->getHousehold()->getDisplayName(),
                'guests' => $rsvp->getTotalCount(),
                'email' => $rsvp->getHousehold()->getEmail()
            ],
            'checked_in_at' => $rsvp->getCheckedInAt()->format('Y-m-d H:i:s')
        ]);
    }

    #[Route('/qr-checkin/undo/{householdId}/{eventId}', methods: ['POST'])]
    public function undoQrCheckIn(int $householdId, int $eventId): JsonResponse
    {
        $rsvp = $this->householdRsvpRepo->findOneBy([
            'household' => $householdId,
            'event' => $eventId
        ]);
        
        if (!$rsvp) {
            return $this->json(['error' => 'RSVP not found'], 404);
        }
        
        $rsvp->setCheckedIn(false);
        $rsvp->setCheckedInAt(null);
        $rsvp->setCheckedInByUser(null);
        
        $this->em->flush();
        
        return $this->json(['success' => true]);
    }
}
