<?php
/**
 * One-Time Cache Clear Script
 * 
 * This script clears the production cache and then deletes itself.
 * Visit this page once after installation to clear the cache.
 * 
 * SECURITY: This file will automatically delete itself after running.
 */

// Function to recursively delete a directory
function deleteDirectory($dir) {
    if (!is_dir($dir)) {
        return false;
    }
    
    $items = array_diff(scandir($dir), ['.', '..']);
    
    foreach ($items as $item) {
        $path = $dir . DIRECTORY_SEPARATOR . $item;
        
        if (is_dir($path)) {
            deleteDirectory($path);
        } else {
            @unlink($path);
        }
    }
    
    return @rmdir($dir);
}

// Get paths - we're in the root directory
$projectDir = __DIR__;
$cacheDir = $projectDir . '/var/cache/prod';
$selfFile = __FILE__;

$cacheCleared = false;
$cacheMessage = '';
$selfDeleted = false;

// Clear the cache
if (is_dir($cacheDir)) {
    if (deleteDirectory($cacheDir)) {
        $cacheCleared = true;
        $cacheMessage = 'Production cache cleared successfully!';
    } else {
        $cacheMessage = 'Failed to clear cache. You may need to delete it manually.';
    }
} else {
    $cacheCleared = true;
    $cacheMessage = 'No cache directory found. Cache is already clear.';
}

// Delete this script file
if ($cacheCleared) {
    $selfDeleted = @unlink($selfFile);
}

?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Cache Cleared</title>
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }
        
        body {
            font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, 'Helvetica Neue', Arial, sans-serif;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            min-height: 100vh;
            display: flex;
            align-items: center;
            justify-content: center;
            padding: 20px;
        }
        
        .container {
            background: white;
            border-radius: 12px;
            box-shadow: 0 20px 60px rgba(0, 0, 0, 0.3);
            max-width: 600px;
            width: 100%;
            padding: 40px;
        }
        
        h1 {
            color: #1a202c;
            font-size: 32px;
            margin-bottom: 20px;
            text-align: center;
        }
        
        .success-icon {
            text-align: center;
            font-size: 64px;
            margin-bottom: 20px;
        }
        
        .message {
            padding: 20px;
            border-radius: 8px;
            margin-bottom: 20px;
            font-size: 16px;
            line-height: 1.6;
        }
        
        .success {
            background: #d4edda;
            color: #155724;
            border: 1px solid #c3e6cb;
        }
        
        .error {
            background: #f8d7da;
            color: #721c24;
            border: 1px solid #f5c6cb;
        }
        
        .info {
            background: #d1ecf1;
            color: #0c5460;
            border: 1px solid #bee5eb;
            margin-top: 20px;
            font-size: 14px;
        }
        
        .button {
            display: inline-block;
            background: #667eea;
            color: white;
            padding: 14px 32px;
            border-radius: 6px;
            text-decoration: none;
            font-weight: 600;
            transition: all 0.3s;
            text-align: center;
            margin-top: 20px;
            font-size: 16px;
        }
        
        .button:hover {
            background: #5568d3;
            transform: translateY(-2px);
            box-shadow: 0 4px 12px rgba(102, 126, 234, 0.4);
        }
        
        .text-center {
            text-align: center;
        }
        
        ul {
            margin-left: 20px;
            margin-top: 10px;
        }
        
        li {
            margin-bottom: 8px;
        }
    </style>
</head>
<body>
    <div class="container">
        <?php if ($cacheCleared): ?>
            <div class="success-icon">✅</div>
            <h1>All Done!</h1>
            
            <div class="message success">
                <?php echo htmlspecialchars($cacheMessage); ?>
            </div>
            
            <?php if ($selfDeleted): ?>
                <div class="message info">
                    <strong>🔒 Security:</strong> This script has automatically deleted itself for security reasons.
                </div>
            <?php else: ?>
                <div class="message error">
                    <strong>⚠️ Important:</strong> Please manually delete the file <code>clear-prod-cache.php</code> from your server for security.
                </div>
            <?php endif; ?>
            
            <div class="message info">
                <strong>✓ Next Steps:</strong>
                <ul>
                    <li>Your application is now ready to use</li>
                    <li>You can close this window</li>
                    <li>Visit your application homepage</li>
                </ul>
            </div>
            
            <!-- <div class="text-center">
                <a href="/public" class="button">Go to Application →</a>
            </div> -->
            
        <?php else: ?>
            <div class="success-icon">❌</div>
            <h1>Cache Clear Failed</h1>
            
            <div class="message error">
                <?php echo htmlspecialchars($cacheMessage); ?>
            </div>
            
            <div class="message info">
                <strong>Manual Solution:</strong><br>
                Please delete the <code>/var/cache/prod</code> folder manually using your file manager or FTP client.
            </div>
            
            <div class="text-center">
                <a href="javascript:location.reload()" class="button">Try Again</a>
            </div>
        <?php endif; ?>
    </div>
</body>
</html>
