/**
 * Get the application base path from the window object
 * This allows the app to work in subdirectories
 */
function getBasePath() {
    return window.APP_BASE_PATH || '';
}

/**
 * Build a full API URL with the base path
 * @param {string} path - The API path (e.g., '/api/dashboard/stats')
 * @returns {string} - The full URL with base path
 */
export function buildApiUrl(path) {
    const basePath = getBasePath();
    // Ensure path starts with /
    const normalizedPath = path.startsWith('/') ? path : `/${path}`;
    return `${basePath}${normalizedPath}`;
}

/**
 * Build a full URL with origin and base path
 * @param {string} path - The path (e.g., '/invite/abc123')
 * @returns {string} - The full URL with origin and base path
 */
export function buildFullUrl(path) {
    const basePath = getBasePath();
    const normalizedPath = path.startsWith('/') ? path : `/${path}`;
    return `${window.location.origin}${basePath}${normalizedPath}`;
}

/**
 * Enhanced fetch that automatically prepends the base path
 * @param {string} path - The API path
 * @param {RequestInit} options - Fetch options
 * @returns {Promise<Response>}
 */
export function apiFetch(path, options = {}) {
    return fetch(buildApiUrl(path), options);
}
