import React, { useState } from 'react';
import { Button } from '@/components/ui/button';

export default function PerGuestForm({ events, onSubmit }) {
    const [selectedEvent, setSelectedEvent] = useState(events[0]?.id || null);
    const [formData, setFormData] = useState({
        guests: [],
        message: ''
    });

    // Initialize guest RSVPs from events
    useState(() => {
        const guestRsvps = [];
        events.forEach(event => {
            event.guests?.forEach(guest => {
                guestRsvps.push({
                    guestId: guest.id,
                    eventId: event.id,
                    guestName: `${guest.firstName} ${guest.lastName || ''}`.trim(),
                    status: guest.status || null,
                    mealChoice: guest.mealChoice || '',
                    dietary: guest.dietary || ''
                });
            });
        });
        setFormData({ guests: guestRsvps, message: '' });
    }, []);

    const updateGuest = (guestId, eventId, updates) => {
        setFormData(prev => ({
            ...prev,
            guests: prev.guests.map(g =>
                g.guestId === guestId && g.eventId === eventId
                    ? { ...g, ...updates }
                    : g
            )
        }));
    };

    const handleSubmit = (e) => {
        e.preventDefault();

        // Validation
        for (const guestData of formData.guests) {
            const event = events.find(e => e.id === guestData.eventId);
            
            if (guestData.status === 'YES' && event.mealsEnabled && !guestData.mealChoice) {
                alert(`Please select a meal choice for ${guestData.guestName} at ${event.name}`);
                return;
            }
        }

        onSubmit(formData);
    };

    const currentEvent = events.find(e => e.id === selectedEvent);
    const currentGuests = formData.guests.filter(g => g.eventId === selectedEvent);

    return (
        <form onSubmit={handleSubmit} className="space-y-6">
            <div className="border-b">
                <div className="flex gap-2 overflow-x-auto">
                    {events.map(event => (
                        <button
                            key={event.id}
                            type="button"
                            onClick={() => setSelectedEvent(event.id)}
                            className={`cursor-pointer px-4 py-2 whitespace-nowrap border-b-2 ${
                                selectedEvent === event.id
                                    ? 'border-primary text-primary font-semibold'
                                    : 'border-transparent hover:border-border'
                            }`}
                        >
                            {event.name}
                        </button>
                    ))}
                </div>
            </div>

            {currentEvent && (
                <div>
                    <div className="mb-4">
                        <h3 className="text-xl font-semibold">{currentEvent.name}</h3>
                        {currentEvent.venueName && (
                            <p className="text-sm text-muted-foreground">{currentEvent.venueName}</p>
                        )}
                        {currentEvent.startAt && (
                            <p className="text-sm text-muted-foreground">
                                {new Date(currentEvent.startAt).toLocaleString()}
                            </p>
                        )}
                    </div>

                    <div className="space-y-4">
                        {currentGuests.map(guestData => (
                            <div key={guestData.guestId} className="border rounded-lg p-4">
                                <h4 className="font-semibold mb-3">{guestData.guestName}</h4>

                                <div className="space-y-3">
                                    <div>
                                        <label className="block text-sm font-medium mb-2">Attending?</label>
                                        <div className="flex gap-2">
                                            {['YES', 'NO', 'MAYBE'].map(status => (
                                                <button
                                                    key={status}
                                                    type="button"
                                                    onClick={() => updateGuest(guestData.guestId, currentEvent.id, { status })}
                                                    className={`cursor-pointer px-4 py-2 rounded border ${
                                                        guestData.status === status
                                                            ? 'bg-primary text-primary-foreground'
                                                            : 'bg-background hover:bg-accent'
                                                    }`}
                                                >
                                                    {status}
                                                </button>
                                            ))}
                                        </div>
                                    </div>

                                    {guestData.status === 'YES' && currentEvent.mealsEnabled && currentEvent.mealOptions && (
                                        <div>
                                            <label className="block text-sm font-medium mb-1">Meal Choice</label>
                                            <select
                                                value={guestData.mealChoice}
                                                onChange={(e) => updateGuest(guestData.guestId, currentEvent.id, { mealChoice: e.target.value })}
                                                className="w-full px-3 py-2 border rounded"
                                            >
                                                <option value="">Select a meal...</option>
                                                {currentEvent.mealOptions.map(meal => (
                                                    <option key={meal.code} value={meal.code}>
                                                        {meal.name}
                                                    </option>
                                                ))}
                                            </select>
                                        </div>
                                    )}

                                    {guestData.status === 'YES' && (
                                        <div>
                                            <label className="block text-sm font-medium mb-1">Dietary Restrictions</label>
                                            <input
                                                type="text"
                                                value={guestData.dietary}
                                                onChange={(e) => updateGuest(guestData.guestId, currentEvent.id, { dietary: e.target.value })}
                                                className="w-full px-3 py-2 border rounded"
                                                placeholder="Any allergies or restrictions?"
                                            />
                                        </div>
                                    )}
                                </div>
                            </div>
                        ))}
                    </div>
                </div>
            )}

            <div>
                <label className="block text-sm font-medium mb-1">Message to Couple (optional)</label>
                <textarea
                    value={formData.message}
                    onChange={(e) => setFormData({ ...formData, message: e.target.value })}
                    className="w-full px-3 py-2 border rounded"
                    rows="3"
                    placeholder="Send a message to the happy couple!"
                />
            </div>

            <Button type="submit" className="w-full">Submit RSVP</Button>
        </form>
    );
}
