import React, { useState } from 'react';
import { Button } from '@/components/ui/button';

export default function HouseholdForm({ events, mode, onSubmit }) {
    const [formData, setFormData] = useState({
        events: events.map(event => ({
            eventId: event.id,
            status: event.rsvp?.status || null,
            adultsCount: event.rsvp?.adultsCount || 0,
            childrenCount: event.rsvp?.childrenCount || 0
        })),
        adults: [],
        children: [],
        message: events[0]?.rsvp?.message || ''
    });

    const updateEvent = (index, updates) => {
        const newEvents = [...formData.events];
        newEvents[index] = { ...newEvents[index], ...updates };
        
        // If adultsCount changed, adjust adults array
        if ('adultsCount' in updates) {
            const newCount = updates.adultsCount;
            const currentAdults = [...formData.adults];
            if (newCount > currentAdults.length) {
                // Add new adults
                for (let i = currentAdults.length; i < newCount; i++) {
                    currentAdults.push({ name: '', meal: '', dietary: '' });
                }
            } else {
                // Remove excess adults
                currentAdults.splice(newCount);
            }
            setFormData({ ...formData, events: newEvents, adults: currentAdults });
        } else if ('childrenCount' in updates) {
            // If childrenCount changed, adjust children array
            const newCount = updates.childrenCount;
            const currentChildren = [...formData.children];
            if (newCount > currentChildren.length) {
                // Add new children
                for (let i = currentChildren.length; i < newCount; i++) {
                    currentChildren.push({ name: '', meal: '', dietary: '' });
                }
            } else {
                // Remove excess children
                currentChildren.splice(newCount);
            }
            setFormData({ ...formData, events: newEvents, children: currentChildren });
        } else {
            setFormData({ ...formData, events: newEvents });
        }
    };

    const updateAdult = (index, field, value) => {
        const newAdults = [...formData.adults];
        newAdults[index] = { ...newAdults[index], [field]: value };
        setFormData({ ...formData, adults: newAdults });
    };

    const updateChild = (index, field, value) => {
        const newChildren = [...formData.children];
        newChildren[index] = { ...newChildren[index], [field]: value };
        setFormData({ ...formData, children: newChildren });
    };

    const handleSubmit = (e) => {
        e.preventDefault();
        
        // Validation
        for (const eventData of formData.events) {
            const event = events.find(e => e.id === eventData.eventId);
            
            if (eventData.status === 'YES') {
                const total = eventData.adultsCount + eventData.childrenCount;
                if (total < 1) {
                    alert('Please specify at least one adult or child for events you\'re attending.');
                    return;
                }

                // Validate meal choices if meals enabled
                if (event.mealsEnabled) {
                    for (let i = 0; i < formData.adults.length; i++) {
                        if (!formData.adults[i].meal) {
                            alert(`Please select a meal for Adult ${i + 1}`);
                            return;
                        }
                    }
                    for (let i = 0; i < formData.children.length; i++) {
                        if (!formData.children[i].meal) {
                            alert(`Please select a meal for Child ${i + 1}`);
                            return;
                        }
                    }
                }
            }
        }

        onSubmit(formData);
    };

    return (
        <form onSubmit={handleSubmit} className="space-y-6">
            {events.map((event, index) => {
                const eventData = formData.events[index];
                const total = eventData.adultsCount + eventData.childrenCount;

                return (
                    <div key={event.id} className="border rounded-lg p-4">
                        <h3 className="text-xl font-semibold mb-2">{event.name}</h3>
                        {event.venueName && (
                            <p className="text-sm text-muted-foreground mb-2">{event.venueName}</p>
                        )}
                        {event.startAt && (
                            <p className="text-sm text-muted-foreground mb-4">
                                {new Date(event.startAt).toLocaleString()}
                            </p>
                        )}

                        <div className="space-y-4">
                            <div>
                                <label className="block text-sm font-medium mb-2">Will you attend?</label>
                                <div className="flex gap-2">
                                    {['YES', 'NO', 'MAYBE'].map(status => (
                                        <button
                                            key={status}
                                            type="button"
                                            onClick={() => updateEvent(index, { status })}
                                            className={`cursor-pointer px-4 py-2 rounded border ${
                                                eventData.status === status
                                                    ? 'bg-primary text-primary-foreground'
                                                    : 'bg-background hover:bg-accent'
                                            }`}
                                        >
                                            {status}
                                        </button>
                                    ))}
                                </div>
                            </div>

                            {eventData.status === 'YES' && (
                                <>
                                    <div className="grid grid-cols-2 gap-4">
                                        <div>
                                            <label className="block text-sm font-medium mb-1">Number of Adults</label>
                                            <input
                                                type="number"
                                                min="0"
                                                max="20"
                                                value={eventData.adultsCount}
                                                onChange={(e) => updateEvent(index, { adultsCount: parseInt(e.target.value) || 0 })}
                                                className="w-full px-3 py-2 border rounded"
                                            />
                                        </div>
                                        <div>
                                            <label className="block text-sm font-medium mb-1">Number of Children</label>
                                            <input
                                                type="number"
                                                min="0"
                                                max="20"
                                                value={eventData.childrenCount}
                                                onChange={(e) => updateEvent(index, { childrenCount: parseInt(e.target.value) || 0 })}
                                                className="w-full px-3 py-2 border rounded"
                                            />
                                        </div>
                                    </div>

                                    {/* Adult Details */}
                                    {formData.adults.length > 0 && (
                                        <div className="space-y-3">
                                            <h4 className="font-medium text-sm">Adult Details</h4>
                                            {formData.adults.map((adult, adultIdx) => (
                                                <div key={adultIdx} className="border rounded-lg p-3 space-y-2 bg-muted/30">
                                                    <p className="text-sm font-medium">Adult {adultIdx + 1}</p>
                                                    <input
                                                        type="text"
                                                        placeholder="Full Name"
                                                        value={adult.name}
                                                        onChange={(e) => updateAdult(adultIdx, 'name', e.target.value)}
                                                        className="w-full px-3 py-2 border rounded text-sm"
                                                    />
                                                    {event.mealsEnabled && event.mealOptions && (
                                                        <select
                                                            value={adult.meal}
                                                            onChange={(e) => updateAdult(adultIdx, 'meal', e.target.value)}
                                                            className="w-full px-3 py-2 border rounded text-sm"
                                                        >
                                                            <option value="">Select Meal</option>
                                                            {event.mealOptions.map(meal => (
                                                                <option key={meal.code} value={meal.code}>
                                                                    {meal.name}
                                                                </option>
                                                            ))}
                                                        </select>
                                                    )}
                                                    <input
                                                        type="text"
                                                        placeholder="Dietary Restrictions (optional)"
                                                        value={adult.dietary}
                                                        onChange={(e) => updateAdult(adultIdx, 'dietary', e.target.value)}
                                                        className="w-full px-3 py-2 border rounded text-sm"
                                                    />
                                                </div>
                                            ))}
                                        </div>
                                    )}

                                    {/* Children Details */}
                                    {formData.children.length > 0 && (
                                        <div className="space-y-3">
                                            <h4 className="font-medium text-sm">Children Details</h4>
                                            {formData.children.map((child, childIdx) => (
                                                <div key={childIdx} className="border rounded-lg p-3 space-y-2 bg-muted/30">
                                                    <p className="text-sm font-medium">Child {childIdx + 1}</p>
                                                    <input
                                                        type="text"
                                                        placeholder="Full Name"
                                                        value={child.name}
                                                        onChange={(e) => updateChild(childIdx, 'name', e.target.value)}
                                                        className="w-full px-3 py-2 border rounded text-sm"
                                                    />
                                                    {event.mealsEnabled && event.mealOptions && (
                                                        <select
                                                            value={child.meal}
                                                            onChange={(e) => updateChild(childIdx, 'meal', e.target.value)}
                                                            className="w-full px-3 py-2 border rounded text-sm"
                                                        >
                                                            <option value="">Select Meal</option>
                                                            {event.mealOptions.map(meal => (
                                                                <option key={meal.code} value={meal.code}>
                                                                    {meal.name}
                                                                </option>
                                                            ))}
                                                        </select>
                                                    )}
                                                    <input
                                                        type="text"
                                                        placeholder="Dietary Restrictions (optional)"
                                                        value={child.dietary}
                                                        onChange={(e) => updateChild(childIdx, 'dietary', e.target.value)}
                                                        className="w-full px-3 py-2 border rounded text-sm"
                                                    />
                                                </div>
                                            ))}
                                        </div>
                                    )}
                                </>
                            )}
                        </div>
                    </div>
                );
            })}


            <div>
                <label className="block text-sm font-medium mb-1">Message to Couple (optional)</label>
                <textarea
                    value={formData.message}
                    onChange={(e) => setFormData({ ...formData, message: e.target.value })}
                    className="w-full px-3 py-2 border rounded"
                    rows="3"
                    placeholder="Send a message to the happy couple!"
                />
            </div>

            <Button type="submit" className="w-full">Submit RSVP</Button>
        </form>
    );
}
