import React, { useState, useEffect } from 'react';
import { Button } from '@/components/ui/button';
import { Tabs, TabsContent, TabsList, TabsTrigger } from '@/components/ui/tabs';
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from '@/components/ui/card';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from '@/components/ui/select';
import { Textarea } from '@/components/ui/textarea';
import { toast } from 'sonner';
import { apiFetch } from '../../../../utils/api';

const DEFAULT_INVITE_MESSAGE = `<p>We are delighted to invite you to our wedding!</p>`;

const DEFAULT_REMINDER_MESSAGE = `<p>We noticed you haven't responded to our wedding invitation yet. We'd love to know if you can join us!</p>
<p>Please take a moment to RSVP so we can finalize our arrangements.</p>`;

const DEFAULT_TICKET_MESSAGE = `<p>Please present this QR code at the entrance to {eventName}.</p>
<p>Show this ticket on your mobile device or print it out to bring with you.</p>`;

const DEFAULT_TICKET_INSTRUCTIONS = `<h3 style="margin-top: 0; color: #f57c00;">📋 How to Use Your Ticket:</h3>
<ol style="margin: 10px 0; padding-left: 20px;">
    <li><strong>Open the link</strong> on your smartphone</li>
    <li><strong>Your QR code</strong> will be displayed</li>
    <li><strong>Show it to staff</strong> at the venue entrance</li>
    <li><strong>They'll scan it</strong> for instant check-in</li>
</ol>
<p style="margin-bottom: 0;"><strong>💡 Tip:</strong> Bookmark this link or take a screenshot for easy access at the event!</p>`;

const DEFAULT_TICKET_FOOTER = `<p>This is your personal QR code ticket. Please do not share this link.</p>
<p style="color: #999; font-size: 12px; margin-top: 10px;">Lost your ticket? Contact us and we'll resend it.</p>`;

export default function Settings() {
    const [settings, setSettings] = useState(null);
    const [smtpSettings, setSmtpSettings] = useState(null);
    const [loading, setLoading] = useState(true);
    const [saving, setSaving] = useState(false);

    useEffect(() => {
        // Load system settings
        apiFetch('/api/settings')
            .then(res => res.json())
            .then(data => {
                setSettings(data);
            });
        
        // Load SMTP settings from User entity
        apiFetch('/api/settings/smtp')
            .then(res => res.json())
            .then(data => {
                setSmtpSettings(data);
                setLoading(false);
            });
    }, []);

    const handleSubmit = async (e) => {
        e.preventDefault();
        setSaving(true);

        try {
            // Save system settings
            await apiFetch('/api/settings', {
                method: 'PATCH',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify(settings)
            });
            
            // Save SMTP settings separately
            await apiFetch('/api/settings/smtp', {
                method: 'PATCH',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify(smtpSettings)
            });
            
            toast.success('Settings saved successfully!');
        } catch (err) {
            toast.error('Failed to save settings');
        }

        setSaving(false);
    };

    const updateField = (field, value) => {
        // Check if it's an SMTP field
        const smtpFields = ['smtpHost', 'smtpPort', 'smtpUsername', 'smtpPassword', 'smtpEncryption', 'smtpFromEmail', 'smtpFromName'];
        
        if (smtpFields.includes(field)) {
            setSmtpSettings({ ...smtpSettings, [field]: value });
        } else {
            setSettings({ ...settings, [field]: value });
        }
    };
    
    const getColorValue = () => settings?.emailButtonColor || '#c2185b';

    if (loading) return <div className="flex items-center justify-center h-96">Loading...</div>;

    return (
        <div className="space-y-6">
            <div>
                <h1 className="text-3xl font-bold tracking-tight">Settings</h1>
                <p className="text-muted-foreground">Configure email settings and customize invitation templates</p>
            </div>

            <form onSubmit={handleSubmit}>
                <Tabs defaultValue="email" className="space-y-4">
                    <TabsList>
                        <TabsTrigger value="email">📧 Email</TabsTrigger>
                        <TabsTrigger value="templates">✉️ Email Templates</TabsTrigger>
                    </TabsList>
                    <TabsContent value="email">
                        <Card>
                            <CardHeader>
                                <CardTitle>Email & SMTP Settings</CardTitle>
                                <CardDescription>Configure email server for sending invitations</CardDescription>
                            </CardHeader>
                            <CardContent className="space-y-4">
                                <div className="space-y-2">
                                    <Label htmlFor="smtpHost">SMTP Host</Label>
                                    <Input
                                        id="smtpHost"
                                        value={smtpSettings?.smtpHost || ''}
                                        onChange={(e) => updateField('smtpHost', e.target.value)}
                                        placeholder="smtp.gmail.com"
                                    />
                                </div>

                                <div className="grid grid-cols-2 gap-4">
                                    <div className="space-y-2">
                                        <Label htmlFor="smtpPort">SMTP Port</Label>
                                        <Input
                                            id="smtpPort"
                                            type="number"
                                            value={smtpSettings?.smtpPort || ''}
                                            onChange={(e) => updateField('smtpPort', e.target.value ? parseInt(e.target.value) : null)}
                                            placeholder="587"
                                        />
                                    </div>
                                    <div className="space-y-2">
                                        <Label htmlFor="smtpEncryption">Encryption</Label>
                                        <Select 
                                            value={smtpSettings?.smtpEncryption || 'none'} 
                                            onValueChange={(value) => updateField('smtpEncryption', value === 'none' ? '' : value)}
                                        >
                                            <SelectTrigger id="smtpEncryption">
                                                <SelectValue />
                                            </SelectTrigger>
                                            <SelectContent>
                                                <SelectItem value="tls">TLS</SelectItem>
                                                <SelectItem value="ssl">SSL</SelectItem>
                                                <SelectItem value="none">None</SelectItem>
                                            </SelectContent>
                                        </Select>
                                    </div>
                                </div>

                                <div className="space-y-2">
                                    <Label htmlFor="smtpUsername">SMTP Username</Label>
                                    <Input
                                        id="smtpUsername"
                                        type="text"
                                        value={smtpSettings?.smtpUsername || ''}
                                        onChange={(e) => updateField('smtpUsername', e.target.value)}
                                        placeholder="your-email@gmail.com"
                                        autoComplete="username"
                                    />
                                </div>

                                <div className="space-y-2">
                                    <Label htmlFor="smtpPassword">SMTP Password</Label>
                                    <Input
                                        id="smtpPassword"
                                        type="password"
                                        value={smtpSettings?.smtpPassword || ''}
                                        onChange={(e) => updateField('smtpPassword', e.target.value)}
                                        placeholder="••••••••"
                                        autoComplete="current-password"
                                    />
                                    <p className="text-sm text-muted-foreground">
                                        For Gmail, use an App Password instead of your regular password
                                    </p>
                                </div>

                                <div className="space-y-2">
                                    <Label htmlFor="smtpFromEmail">From Email Address</Label>
                                    <Input
                                        id="smtpFromEmail"
                                        type="email"
                                        value={smtpSettings?.smtpFromEmail || ''}
                                        onChange={(e) => updateField('smtpFromEmail', e.target.value)}
                                        placeholder="noreply@yourwedding.com"
                                    />
                                </div>

                                <div className="space-y-2">
                                    <Label htmlFor="smtpFromName">From Name</Label>
                                    <Input
                                        id="smtpFromName"
                                        value={smtpSettings?.smtpFromName || ''}
                                        onChange={(e) => updateField('smtpFromName', e.target.value)}
                                        placeholder="John & Jane Wedding"
                                    />
                                </div>
                            </CardContent>
                        </Card>
                    </TabsContent>

                    <TabsContent value="templates">
                        <div className="space-y-6">
                            <Card>
                                <CardHeader>
                                    <CardTitle>Invitation Email Content</CardTitle>
                                    <CardDescription>
                                        Customize the message in your invitation emails. Variables: {'{eventName}'}, {'{guestName}'}, {'{eventDate}'}, {'{venueName}'}
                                    </CardDescription>
                                </CardHeader>
                                <CardContent className="space-y-4">
                                    <div className="space-y-2">
                                        <Label htmlFor="inviteEmailSubject">Subject Line</Label>
                                        <Input
                                            id="inviteEmailSubject"
                                            value={settings?.inviteEmailSubject || "You're Invited to Our Wedding!"}
                                            onChange={(e) => updateField('inviteEmailSubject', e.target.value)}
                                            placeholder="You're Invited to Our Wedding!"
                                        />
                                    </div>
                                    <div className="space-y-2">
                                        <Label htmlFor="inviteEmailBody">Message Content (HTML)</Label>
                                        <Textarea
                                            id="inviteEmailBody"
                                            value={settings?.inviteEmailBody || DEFAULT_INVITE_MESSAGE}
                                            onChange={(e) => updateField('inviteEmailBody', e.target.value)}
                                            placeholder="Enter your invitation message..."
                                            rows={6}
                                            className="font-mono text-sm"
                                        />
                                        <p className="text-xs text-muted-foreground">
                                            Supports HTML tags (&lt;p&gt;, &lt;strong&gt;, &lt;em&gt;) and variables: {'{eventName}'}, {'{guestName}'}, {'{eventDate}'}, {'{venueName}'}
                                        </p>
                                    </div>
                                </CardContent>
                            </Card>

                            <Card>
                                <CardHeader>
                                    <CardTitle>Reminder Email Content</CardTitle>
                                    <CardDescription>
                                        Customize the message for reminder emails. Variables: {'{eventName}'}, {'{guestName}'}, {'{eventDate}'}, {'{venueName}'}
                                    </CardDescription>
                                </CardHeader>
                                <CardContent className="space-y-4">
                                    <div className="space-y-2">
                                        <Label htmlFor="reminderEmailSubject">Subject Line</Label>
                                        <Input
                                            id="reminderEmailSubject"
                                            value={settings?.reminderEmailSubject || "Reminder: Please RSVP to Our Wedding"}
                                            onChange={(e) => updateField('reminderEmailSubject', e.target.value)}
                                            placeholder="Reminder: Please RSVP to Our Wedding"
                                        />
                                    </div>
                                    <div className="space-y-2">
                                        <Label htmlFor="reminderEmailBody">Message Content (HTML)</Label>
                                        <Textarea
                                            id="reminderEmailBody"
                                            value={settings?.reminderEmailBody || DEFAULT_REMINDER_MESSAGE}
                                            onChange={(e) => updateField('reminderEmailBody', e.target.value)}
                                            placeholder="Enter your reminder message..."
                                            rows={6}
                                            className="font-mono text-sm"
                                        />
                                        <p className="text-xs text-muted-foreground">
                                            Supports HTML tags (&lt;p&gt;, &lt;strong&gt;, &lt;em&gt;) and variables: {'{eventName}'}, {'{guestName}'}, {'{eventDate}'}, {'{venueName}'}
                                        </p>
                                    </div>
                                </CardContent>
                            </Card>

                            <Card>
                                <CardHeader>
                                    <CardTitle>QR Ticket Email Content</CardTitle>
                                    <CardDescription>
                                        Customize ALL text in the QR code ticket email. Variables: {'{eventName}'}, {'{guestName}'}, {'{eventDate}'}, {'{venueName}'}
                                    </CardDescription>
                                </CardHeader>
                                <CardContent className="space-y-4">
                                    <div className="space-y-2">
                                        <Label htmlFor="ticketEmailSubject">Subject Line</Label>
                                        <Input
                                            id="ticketEmailSubject"
                                            value={settings?.ticketEmailSubject || "Your QR Code Ticket for {eventName}"}
                                            onChange={(e) => updateField('ticketEmailSubject', e.target.value)}
                                            placeholder="Your QR Code Ticket for {eventName}"
                                        />
                                    </div>
                                    <div className="space-y-2">
                                        <Label htmlFor="ticketEmailGreeting">Greeting</Label>
                                        <Input
                                            id="ticketEmailGreeting"
                                            value={settings?.ticketEmailGreeting || "Dear {guestName},"}
                                            onChange={(e) => updateField('ticketEmailGreeting', e.target.value)}
                                            placeholder="Dear {guestName},"
                                        />
                                    </div>
                                    <div className="space-y-2">
                                        <Label htmlFor="ticketEmailIntro">Introduction Text (HTML)</Label>
                                        <Textarea
                                            id="ticketEmailIntro"
                                            value={settings?.ticketEmailIntro || "<p>Thank you for confirming your attendance! We're so excited to celebrate with you.</p>"}
                                            onChange={(e) => updateField('ticketEmailIntro', e.target.value)}
                                            placeholder="<p>Thank you for confirming...</p>"
                                            rows={3}
                                            className="font-mono text-sm"
                                        />
                                    </div>
                                    <div className="space-y-2">
                                        <Label htmlFor="ticketEmailBody">Main Message (HTML)</Label>
                                        <Textarea
                                            id="ticketEmailBody"
                                            value={settings?.ticketEmailBody || DEFAULT_TICKET_MESSAGE}
                                            onChange={(e) => updateField('ticketEmailBody', e.target.value)}
                                            placeholder="Enter your ticket email message..."
                                            rows={4}
                                            className="font-mono text-sm"
                                        />
                                    </div>
                                    <div className="space-y-2">
                                        <Label htmlFor="ticketEmailButtonText">Button Text</Label>
                                        <Input
                                            id="ticketEmailButtonText"
                                            value={settings?.ticketEmailButtonText || "View My Ticket"}
                                            onChange={(e) => updateField('ticketEmailButtonText', e.target.value)}
                                            placeholder="View My Ticket"
                                        />
                                    </div>
                                    <div className="space-y-2">
                                        <Label htmlFor="ticketEmailInstructions">Instructions Section (HTML, optional)</Label>
                                        <Textarea
                                            id="ticketEmailInstructions"
                                            value={settings?.ticketEmailInstructions || DEFAULT_TICKET_INSTRUCTIONS}
                                            onChange={(e) => updateField('ticketEmailInstructions', e.target.value)}
                                            placeholder="Leave blank to hide instructions..."
                                            rows={8}
                                            className="font-mono text-sm"
                                        />
                                        <p className="text-xs text-muted-foreground">
                                            Leave blank to hide the instructions section
                                        </p>
                                    </div>
                                    <div className="space-y-2">
                                        <Label htmlFor="ticketEmailClosing">Closing Text</Label>
                                        <Input
                                            id="ticketEmailClosing"
                                            value={settings?.ticketEmailClosing || "We can't wait to see you there!"}
                                            onChange={(e) => updateField('ticketEmailClosing', e.target.value)}
                                            placeholder="We can't wait to see you there!"
                                        />
                                    </div>
                                    <div className="space-y-2">
                                        <Label htmlFor="ticketEmailFooter">Footer Text (HTML)</Label>
                                        <Textarea
                                            id="ticketEmailFooter"
                                            value={settings?.ticketEmailFooter || DEFAULT_TICKET_FOOTER}
                                            onChange={(e) => updateField('ticketEmailFooter', e.target.value)}
                                            placeholder="Footer disclaimer..."
                                            rows={4}
                                            className="font-mono text-sm"
                                        />
                                    </div>
                                    <p className="text-xs text-muted-foreground mt-4">
                                        <strong>Tip:</strong> All fields support HTML and these variables: {'{eventName}'}, {'{guestName}'}, {'{eventDate}'}, {'{venueName}'}
                                    </p>
                                </CardContent>
                            </Card>

                            <Card>
                                <CardHeader>
                                    <CardTitle>Email Customization</CardTitle>
                                    <CardDescription>
                                        Customize colors, button text, and other elements used in both email templates
                                    </CardDescription>
                                </CardHeader>
                                <CardContent className="space-y-4">
                                    <div className="grid grid-cols-2 gap-4">
                                        <div className="space-y-2">
                                            <Label htmlFor="emailButtonText">Button Text</Label>
                                            <Input
                                                id="emailButtonText"
                                                value={settings?.emailButtonText || "RSVP Now"}
                                                onChange={(e) => updateField('emailButtonText', e.target.value)}
                                                placeholder="RSVP Now"
                                            />
                                        </div>
                                        <div className="space-y-2">
                                            <Label htmlFor="emailButtonColor">Button Color</Label>
                                            <div className="flex gap-2">
                                                <Input
                                                    id="emailButtonColor"
                                                    type="color"
                                                    value={getColorValue()}
                                                    onChange={(e) => updateField('emailButtonColor', e.target.value)}
                                                    className="w-20 h-10"
                                                />
                                                <Input
                                                    type="text"
                                                    value={getColorValue()}
                                                    onChange={(e) => updateField('emailButtonColor', e.target.value)}
                                                    placeholder="#c2185b"
                                                />
                                            </div>
                                        </div>
                                    </div>
                                    <div className="space-y-2">
                                        <Label htmlFor="emailCallToAction">Call to Action Text</Label>
                                        <Input
                                            id="emailCallToAction"
                                            value={settings?.emailCallToAction || "Please RSVP by clicking the button below:"}
                                            onChange={(e) => updateField('emailCallToAction', e.target.value)}
                                            placeholder="Please RSVP by clicking the button below:"
                                        />
                                    </div>
                                    <div className="space-y-2">
                                        <Label htmlFor="emailClosingText">Closing Text</Label>
                                        <Input
                                            id="emailClosingText"
                                            value={settings?.emailClosingText || "We can't wait to celebrate with you!"}
                                            onChange={(e) => updateField('emailClosingText', e.target.value)}
                                            placeholder="We can't wait to celebrate with you!"
                                        />
                                    </div>
                                    <div className="space-y-2">
                                        <Label htmlFor="emailSignature">Custom Signature (optional HTML)</Label>
                                        <Textarea
                                            id="emailSignature"
                                            value={settings?.emailSignature || ''}
                                            onChange={(e) => updateField('emailSignature', e.target.value)}
                                            placeholder="Leave blank to use default: With love, {coupleNames}"
                                            rows={3}
                                            className="font-mono text-sm"
                                        />
                                        <p className="text-xs text-muted-foreground">
                                            Override the default signature. Leave blank to use "With love, [Couple Names]"
                                        </p>
                                    </div>
                                </CardContent>
                            </Card>
                        </div>
                    </TabsContent>
                </Tabs>

                <div className="flex justify-end mt-6">
                    <Button type="submit" disabled={saving} size="lg">
                        {saving ? 'Saving...' : 'Save All Settings'}
                    </Button>
                </div>
            </form>
        </div>
    );
}
