import React, { useState, useEffect } from 'react';
import { Button } from '@/components/ui/button';
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from '@/components/ui/card';
import { Input } from '@/components/ui/input';
import { Badge } from '@/components/ui/badge';
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from '@/components/ui/select';
import { Checkbox } from '@/components/ui/checkbox';
import { Mail, Users, CheckCircle2, AlertTriangle, Send } from 'lucide-react';
import { toast } from 'sonner';
import { apiFetch } from '../../../../utils/api';

export default function Invites() {
    const [events, setEvents] = useState([]);
    const [households, setHouseholds] = useState([]);
    const [selectedEvent, setSelectedEvent] = useState('');
    const [selectedHouseholds, setSelectedHouseholds] = useState([]);
    const [sending, setSending] = useState(false);
    const [smtpConfigured, setSmtpConfigured] = useState(false);
    const [searchTerm, setSearchTerm] = useState('');

    useEffect(() => {
        loadData();
    }, []);

    const loadData = async () => {
        const eventsRes = await apiFetch('/api/events');
        const eventsData = await eventsRes.json();
        setEvents(eventsData);

        const householdsRes = await apiFetch('/api/households');
        const householdsData = await householdsRes.json();
        setHouseholds(householdsData);
    };

    const testSmtp = async () => {
        const response = await apiFetch('/api/email/test-smtp', { method: 'POST' });
        const result = await response.json();
        
        if (result.success) {
            toast.success(result.message);
            setSmtpConfigured(true);
        } else {
            toast.error(result.message);
            setSmtpConfigured(false);
        }
    };

    const sendInvites = async () => {
        if (selectedHouseholds.length === 0) {
            toast.error('Please select at least one household');
            return;
        }

        setSending(true);
        const event = events.find(e => e.id === parseInt(selectedEvent));
        
        const recipients = selectedHouseholds.map(id => {
            const household = households.find(h => h.id === id);
            return {
                email: household.email,
                name: household.displayName,
                eventName: event?.name,
                eventSlug: event?.slug
            };
        }).filter(r => r.email); // Only include households with email

        try {
            const response = await apiFetch('/api/email/send-bulk', {
                method: 'POST',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify({ 
                    recipients, 
                    eventSlug: event?.slug 
                })
            });

            const result = await response.json();
            
            if (result.sent !== undefined) {
                toast.success('Invitations sent', {
                    description: `Sent: ${result.sent}, Failed: ${result.failed}`
                });
                if (result.errors && result.errors.length > 0) {
                    console.error('Failed emails:', result.errors);
                }
            } else {
                toast.success('Invitations sent successfully!');
            }

            setSelectedHouseholds([]);
        } catch (error) {
            toast.error('Failed to send invitations', {
                description: error.message
            });
        } finally {
            setSending(false);
        }
    };

    const selectAll = () => {
        const withEmail = households.filter(h => h.email).map(h => h.id);
        setSelectedHouseholds(withEmail);
    };

    const clearSelection = () => {
        setSelectedHouseholds([]);
    };

    const filteredHouseholds = households.filter(h => 
        h.displayName?.toLowerCase().includes(searchTerm.toLowerCase()) ||
        h.email?.toLowerCase().includes(searchTerm.toLowerCase())
    );

    const householdsWithEmail = households.filter(h => h.email);

    return (
        <div className="space-y-6">
            {/* Header */}
            <div className="flex flex-col gap-4 sm:flex-row sm:items-center sm:justify-between">
                <div>
                    <h1 className="text-3xl font-bold tracking-tight">Send Invitations</h1>
                    <p className="text-muted-foreground">Send RSVP invites via email</p>
                </div>
                <Button variant="outline" onClick={testSmtp}>
                    <Send className="h-4 w-4 mr-2" />
                    Test SMTP
                </Button>
            </div>

            {/* Warning if no SMTP */}
            {!smtpConfigured && (
                <Card className="border-yellow-200 bg-yellow-50">
                    <CardContent className="flex items-center gap-3 pt-6">
                        <AlertTriangle className="h-5 w-5 text-yellow-600" />
                        <p className="text-sm text-yellow-800">
                            SMTP not configured. Please set up SMTP settings before sending emails.
                        </p>
                    </CardContent>
                </Card>
            )}

            {/* Stats */}
            <div className="grid gap-4 grid-cols-1 sm:grid-cols-3">
                <Card>
                    <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                        <CardTitle className="text-sm font-medium">Total Households</CardTitle>
                        <Users className="h-4 w-4 text-muted-foreground" />
                    </CardHeader>
                    <CardContent>
                        <div className="text-2xl font-bold">{households.length}</div>
                    </CardContent>
                </Card>
                <Card>
                    <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                        <CardTitle className="text-sm font-medium">With Email</CardTitle>
                        <Mail className="h-4 w-4 text-green-600" />
                    </CardHeader>
                    <CardContent>
                        <div className="text-2xl font-bold text-green-600">{householdsWithEmail.length}</div>
                    </CardContent>
                </Card>
                <Card>
                    <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                        <CardTitle className="text-sm font-medium">Selected</CardTitle>
                        <CheckCircle2 className="h-4 w-4 text-blue-600" />
                    </CardHeader>
                    <CardContent>
                        <div className="text-2xl font-bold text-blue-600">{selectedHouseholds.length}</div>
                    </CardContent>
                </Card>
            </div>

            {/* Event Selection */}
            <Card>
                <CardHeader>
                    <CardTitle>Select Event (Optional)</CardTitle>
                    <CardDescription>
                        Choose a specific event or leave blank to send general wedding invitation
                    </CardDescription>
                </CardHeader>
                <CardContent>
                    <Select value={selectedEvent || "all"} onValueChange={(value) => setSelectedEvent(value === "all" ? "" : value)}>
                        <SelectTrigger>
                            <SelectValue placeholder="All Events (/rsvp)" />
                        </SelectTrigger>
                        <SelectContent>
                            <SelectItem value="all">All Events (/rsvp)</SelectItem>
                            {events.map(event => (
                                <SelectItem key={event.id} value={event.id.toString()}>
                                    {event.name} (/rsvp/{event.slug})
                                </SelectItem>
                            ))}
                        </SelectContent>
                    </Select>
                </CardContent>
            </Card>

            {/* Recipients Selection */}
            <Card>
                <CardHeader>
                    <div className="flex justify-between items-center">
                        <div>
                            <CardTitle>Select Recipients</CardTitle>
                            <CardDescription>Choose households to receive invitations</CardDescription>
                        </div>
                        <div className="flex gap-2">
                            <Button variant="outline" size="sm" onClick={selectAll}>
                                <CheckCircle2 className="h-4 w-4 mr-2" />
                                All ({householdsWithEmail.length})
                            </Button>
                            <Button variant="outline" size="sm" onClick={clearSelection}>
                                Clear
                            </Button>
                        </div>
                    </div>
                </CardHeader>
                <CardContent>
                    {/* Search */}
                    <Input
                        placeholder="Search households..."
                        value={searchTerm}
                        onChange={(e) => setSearchTerm(e.target.value)}
                        className="mb-4"
                    />

                    {/* Household List */}
                    <div className="space-y-2 max-h-96 overflow-y-auto mb-4">
                        {filteredHouseholds.filter(h => h.email).length === 0 ? (
                            <div className="flex flex-col items-center justify-center py-8 text-muted-foreground">
                                <Mail className="h-12 w-12 mb-2 opacity-50" />
                                <p>No households with email addresses found</p>
                            </div>
                        ) : (
                            filteredHouseholds.filter(h => h.email).map(household => (
                                <label 
                                    key={household.id} 
                                    className="flex items-start gap-3 p-3 border rounded hover:bg-accent cursor-pointer transition-colors"
                                >
                                    <Checkbox
                                        checked={selectedHouseholds.includes(household.id)}
                                        onCheckedChange={(checked) => {
                                            if (checked) {
                                                setSelectedHouseholds([...selectedHouseholds, household.id]);
                                            } else {
                                                setSelectedHouseholds(selectedHouseholds.filter(id => id !== household.id));
                                            }
                                        }}
                                        className="mt-0.5"
                                    />
                                    <div className="flex-1">
                                        <div className="font-medium">{household.displayName}</div>
                                        <div className="text-sm text-muted-foreground flex items-center gap-1">
                                            <Mail className="h-3 w-3" />
                                            {household.email}
                                        </div>
                                        {household.tags && (
                                            <div className="flex gap-1 mt-1 flex-wrap">
                                                {household.tags.split(',').map((tag, idx) => (
                                                    <Badge key={idx} variant="secondary" className="text-xs">
                                                        {tag.trim()}
                                                    </Badge>
                                                ))}
                                            </div>
                                        )}
                                    </div>
                                    {household.invitedAt && (
                                        <Badge variant="outline" className="text-green-600">
                                            <CheckCircle2 className="h-3 w-3 mr-1" />
                                            Invited
                                        </Badge>
                                    )}
                                </label>
                            ))
                        )}
                    </div>

                    {/* Send Button */}
                    <div className="space-y-4">
                        <Button 
                            onClick={sendInvites}
                            disabled={sending || selectedHouseholds.length === 0}
                            className="w-full"
                            size="lg"
                        >
                            {sending ? (
                                <>
                                    <Send className="h-4 w-4 mr-2 animate-pulse" />
                                    Sending...
                                </>
                            ) : (
                                <>
                                    <Mail className="h-4 w-4 mr-2" />
                                    Send to {selectedHouseholds.length} Recipient{selectedHouseholds.length !== 1 ? 's' : ''}
                                </>
                            )}
                        </Button>

                        {selectedHouseholds.length > 0 && (
                            <p className="text-sm text-muted-foreground text-center">
                                💡 Each email will contain a personalized RSVP link
                            </p>
                        )}
                    </div>
                </CardContent>
            </Card>
        </div>
    );
}
