import React, { useState, useEffect } from 'react';
import { Button } from '@/components/ui/button';
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from '@/components/ui/card';
import { Input } from '@/components/ui/input';
import { Badge } from '@/components/ui/badge';
import { Table, TableBody, TableCell, TableHead, TableHeader, TableRow } from '@/components/ui/table';
import { DropdownMenu, DropdownMenuContent, DropdownMenuItem, DropdownMenuLabel, DropdownMenuSeparator, DropdownMenuTrigger } from '@/components/ui/dropdown-menu';
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from '@/components/ui/select';
import { Skeleton } from '@/components/ui/skeleton';
import { Dialog, DialogContent, DialogDescription, DialogHeader, DialogTitle } from '@/components/ui/dialog';
import { Users, UserCheck, UserX, Mail, Eye, Link, MoreVertical, Plus, Download, RefreshCw, Edit, Trash2, Send, Filter, ChevronDown, ChevronUp, Utensils, AlertCircle, ChevronLeft, ChevronRight, ChevronsLeft, ChevronsRight, QrCode, Calendar } from 'lucide-react';
import { toast } from 'sonner';
import { apiFetch, buildFullUrl, buildApiUrl } from '../../../../utils/api';

export default function Guests() {
    const [households, setHouseholds] = useState([]);
    const [loading, setLoading] = useState(true);
    const [searchTerm, setSearchTerm] = useState('');
    const [filterSide, setFilterSide] = useState('all');
    const [filterStatus, setFilterStatus] = useState('all');
    const [viewingHousehold, setViewingHousehold] = useState(null);
    const [expandedRows, setExpandedRows] = useState([]);
    const [currentPage, setCurrentPage] = useState(1);
    const [itemsPerPage, setItemsPerPage] = useState(10);
    const [qrModalGuest, setQrModalGuest] = useState(null);
    const [events, setEvents] = useState([]);

    useEffect(() => {
        loadHouseholds();
        loadEvents();
    }, []);

    const loadHouseholds = () => {
        apiFetch('/api/households')
            .then(res => res.json())
            .then(data => {
                setHouseholds(data);
                setLoading(false);
            });
    };

    const loadEvents = () => {
        apiFetch('/api/events')
            .then(res => res.json())
            .then(data => {
                setEvents(data);
            });
    };

    const copyInviteLink = (token) => {
        const url = buildFullUrl(`/invite/${token}`);
        navigator.clipboard.writeText(url);
        toast.success('Invite link copied to clipboard!');
    };

    const regenerateToken = async (id) => {
        if (!confirm('Regenerate invite token? The old link will stop working.')) return;
        
        const res = await apiFetch(`/api/households/${id}/regen-token`, { method: 'POST' });
        const data = await res.json();
        
        loadHouseholds();
        toast.success('Token regenerated!');
    };

    const filteredHouseholds = households.filter(h => {
        const matchesSearch = h.displayName.toLowerCase().includes(searchTerm.toLowerCase()) ||
                            h.email?.toLowerCase().includes(searchTerm.toLowerCase());
        const matchesSide = filterSide === 'all' || h.side === filterSide;
        const matchesStatus = filterStatus === 'all' ||
            (filterStatus === 'opened' && h.openedAt) ||
            (filterStatus === 'invited' && h.invitedAt && !h.openedAt) ||
            (filterStatus === 'not_sent' && !h.invitedAt);
        return matchesSearch && matchesSide && matchesStatus;
    });

    // Reset to page 1 when filters change
    useEffect(() => {
        setCurrentPage(1);
    }, [searchTerm, filterSide, filterStatus]);

    // Calculate pagination
    const totalPages = Math.ceil(filteredHouseholds.length / itemsPerPage);
    const startIndex = (currentPage - 1) * itemsPerPage;
    const endIndex = startIndex + itemsPerPage;
    const paginatedHouseholds = filteredHouseholds.slice(startIndex, endIndex);

    const stats = {
        total: households.reduce((sum, h) => sum + (h.guestCount || 0), 0),
        households: households.length,
        responded: households.filter(h => h.openedAt).length,
        bride: households.filter(h => h.side === 'BRIDE').reduce((sum, h) => sum + (h.guestCount || 0), 0),
        groom: households.filter(h => h.side === 'GROOM').reduce((sum, h) => sum + (h.guestCount || 0), 0),
    };

    if (loading) {
        return (
            <div className="space-y-6">
                <Skeleton className="h-10 w-64" />
                <div className="grid grid-cols-2 md:grid-cols-4 gap-4">
                    {[...Array(4)].map((_, i) => (
                        <Card key={i}>
                            <CardHeader className="pb-2">
                                <Skeleton className="h-4 w-20" />
                            </CardHeader>
                            <CardContent>
                                <Skeleton className="h-8 w-16" />
                            </CardContent>
                        </Card>
                    ))}
                </div>
                <Skeleton className="h-96 w-full" />
            </div>
        );
    }

    return (
        <div className="space-y-6">
            {/* Header */}
            <div className="flex flex-col gap-4 sm:flex-row sm:items-center sm:justify-between">
                <div>
                    <h1 className="text-3xl font-bold tracking-tight">Guests</h1>
                    <p className="text-muted-foreground">View and manage all guests across households</p>
                </div>
                <div className="flex gap-2 flex-wrap">
                    <Button variant="outline" size="sm" onClick={() => window.location.href = buildApiUrl('/api/export/households')}>
                        <Download className="h-4 w-4 mr-2" />
                        Export
                    </Button>
                    <Button size="sm" onClick={() => window.location.href = '/admin/households'}>
                        <Plus className="h-4 w-4 mr-2" />
                        Add Household
                    </Button>
                </div>
            </div>

            {/* Stats Cards */}
            <div className="grid gap-4 grid-cols-2 sm:grid-cols-4">
                <Card>
                    <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                        <CardTitle className="text-sm font-medium">Total Guests</CardTitle>
                        <Users className="h-4 w-4 text-muted-foreground" />
                    </CardHeader>
                    <CardContent>
                        <div className="text-2xl font-bold">{stats.total}</div>
                        <p className="text-xs text-muted-foreground">{stats.households} households</p>
                    </CardContent>
                </Card>
                <Card>
                    <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                        <CardTitle className="text-sm font-medium">Responded</CardTitle>
                        <UserCheck className="h-4 w-4 text-green-600" />
                    </CardHeader>
                    <CardContent>
                        <div className="text-2xl font-bold text-green-600">{stats.responded}</div>
                        <p className="text-xs text-muted-foreground">households</p>
                    </CardContent>
                </Card>
                <Card>
                    <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                        <CardTitle className="text-sm font-medium">Bride's Side</CardTitle>
                        <Users className="h-4 w-4 text-pink-600" />
                    </CardHeader>
                    <CardContent>
                        <div className="text-2xl font-bold">{stats.bride}</div>
                        <p className="text-xs text-muted-foreground">guests</p>
                    </CardContent>
                </Card>
                <Card>
                    <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                        <CardTitle className="text-sm font-medium">Groom's Side</CardTitle>
                        <Users className="h-4 w-4 text-blue-600" />
                    </CardHeader>
                    <CardContent>
                        <div className="text-2xl font-bold">{stats.groom}</div>
                        <p className="text-xs text-muted-foreground">guests</p>
                    </CardContent>
                </Card>
            </div>

            {/* Filters */}
            <Card>
                <CardContent className="pt-6">
                    <div className="flex flex-col gap-4">
                        <div className="flex flex-col sm:flex-row gap-4">
                            <div className="flex-1">
                                <Input
                                    placeholder="Search by household or email..."
                                    value={searchTerm}
                                    onChange={(e) => setSearchTerm(e.target.value)}
                                />
                            </div>
                            <div className="flex gap-2">
                                <Select value={filterSide} onValueChange={setFilterSide}>
                                    <SelectTrigger className="w-full sm:w-[150px]">
                                        <SelectValue />
                                    </SelectTrigger>
                                    <SelectContent>
                                        <SelectItem value="all">All Sides</SelectItem>
                                        <SelectItem value="BRIDE">Bride's Side</SelectItem>
                                        <SelectItem value="GROOM">Groom's Side</SelectItem>
                                    </SelectContent>
                                </Select>
                                <Select value={filterStatus} onValueChange={setFilterStatus}>
                                    <SelectTrigger className="w-full sm:w-[150px]">
                                        <SelectValue />
                                    </SelectTrigger>
                                    <SelectContent>
                                        <SelectItem value="all">All Status</SelectItem>
                                        <SelectItem value="opened">Opened</SelectItem>
                                        <SelectItem value="invited">Invited</SelectItem>
                                        <SelectItem value="not_sent">Not Sent</SelectItem>
                                    </SelectContent>
                                </Select>
                            </div>
                        </div>
                        {(searchTerm || filterSide !== 'all' || filterStatus !== 'all') && (
                            <div className="flex items-center gap-2 text-sm text-muted-foreground">
                                <Filter className="h-4 w-4" />
                                Showing {filteredHouseholds.length} of {households.length} households
                                <Button variant="ghost" size="sm" onClick={() => {
                                    setSearchTerm('');
                                    setFilterSide('all');
                                    setFilterStatus('all');
                                }}>
                                    Clear filters
                                </Button>
                            </div>
                        )}
                    </div>
                </CardContent>
            </Card>

            {/* Guests Table */}
            <Card>
                <Table>
                    <TableHeader>
                        <TableRow>
                            <TableHead>Household</TableHead>
                            <TableHead className="w-[100px]">Guests</TableHead>
                            <TableHead className="w-[120px]">Side</TableHead>
                            <TableHead className="w-[150px]">Events</TableHead>
                            <TableHead>Contact</TableHead>
                            <TableHead className="w-[120px]">Status</TableHead>
                            <TableHead className="w-[80px] text-right">Actions</TableHead>
                        </TableRow>
                    </TableHeader>
                    <TableBody>
                        {paginatedHouseholds.length === 0 ? (
                            <TableRow>
                                <TableCell colSpan={7} className="h-24 text-center">
                                    <div className="flex flex-col items-center gap-2 text-muted-foreground">
                                        <Users className="h-8 w-8 opacity-50" />
                                        <p>No guests found matching your filters.</p>
                                        <Button variant="outline" size="sm" onClick={() => {
                                            setSearchTerm('');
                                            setFilterSide('all');
                                            setFilterStatus('all');
                                        }}>
                                            Clear filters
                                        </Button>
                                    </div>
                                </TableCell>
                            </TableRow>
                        ) : (
                            paginatedHouseholds.map(household => (
                                <TableRow key={household.id}>
                                    <TableCell>
                                        <div className="font-medium">{household.displayName}</div>
                                    </TableCell>
                                    <TableCell>
                                        <Badge variant="outline">
                                            <Users className="h-3 w-3 mr-1" />
                                            {household.guestCount}
                                        </Badge>
                                    </TableCell>
                                    <TableCell>
                                        <Badge variant={household.side === 'BRIDE' ? 'default' : 'secondary'}>
                                            {household.side}
                                        </Badge>
                                    </TableCell>
                                    <TableCell>
                                        <div className="flex gap-1 flex-wrap">
                                            {household.eventRsvps && household.eventRsvps.length > 0 ? (
                                                household.eventRsvps.map(rsvp => (
                                                    <Badge 
                                                        key={rsvp.eventId} 
                                                        variant="outline"
                                                        className={
                                                            rsvp.status === 'YES' ? 'border-green-500 text-green-700' :
                                                            rsvp.status === 'NO' ? 'border-red-500 text-red-700' :
                                                            rsvp.status === 'MAYBE' ? 'border-yellow-500 text-yellow-700' :
                                                            'border-gray-500 text-gray-700'
                                                        }
                                                    >
                                                        <Calendar className="h-3 w-3 mr-1" />
                                                        {rsvp.eventName}
                                                    </Badge>
                                                ))
                                            ) : (
                                                <span className="text-sm text-muted-foreground">-</span>
                                            )}
                                        </div>
                                    </TableCell>
                                    <TableCell className="text-sm">
                                        {household.email ? (
                                            <div className="flex items-center gap-1 text-muted-foreground">
                                                <Mail className="h-3 w-3" />
                                                {household.email}
                                            </div>
                                        ) : (
                                            <span className="text-muted-foreground">-</span>
                                        )}
                                    </TableCell>
                                    <TableCell>
                                        {household.openedAt ? (
                                            <Badge variant="default" className="bg-green-500">
                                                <Eye className="h-3 w-3 mr-1" />
                                                Opened
                                            </Badge>
                                        ) : household.invitedAt ? (
                                            <Badge variant="secondary">
                                                <Send className="h-3 w-3 mr-1" />
                                                Invited
                                            </Badge>
                                        ) : (
                                            <span className="text-sm text-muted-foreground">Not sent</span>
                                        )}
                                    </TableCell>
                                    <TableCell className="text-right">
                                        <DropdownMenu>
                                            <DropdownMenuTrigger asChild>
                                                <Button variant="ghost" size="sm">
                                                    <MoreVertical className="h-4 w-4" />
                                                </Button>
                                            </DropdownMenuTrigger>
                                            <DropdownMenuContent align="end" className="w-[180px]">
                                                <DropdownMenuLabel>Quick Actions</DropdownMenuLabel>
                                                <DropdownMenuSeparator />
                                                <DropdownMenuItem onClick={() => copyInviteLink(household.token)}>
                                                    <Link className="h-4 w-4 mr-2" />
                                                    Copy Link
                                                </DropdownMenuItem>
                                                <DropdownMenuItem onClick={() => window.open(`/invite/${household.token}`, '_blank')}>
                                                    <Eye className="h-4 w-4 mr-2" />
                                                    Preview
                                                </DropdownMenuItem>
                                                <DropdownMenuItem onClick={() => setQrModalGuest(household)}>
                                                    <QrCode className="h-4 w-4 mr-2" />
                                                    View QR Codes
                                                </DropdownMenuItem>
                                                <DropdownMenuItem onClick={() => window.location.href = '/admin/households'}>
                                                    <Edit className="h-4 w-4 mr-2" />
                                                    Edit
                                                </DropdownMenuItem>
                                                <DropdownMenuSeparator />
                                                <DropdownMenuItem onClick={() => setViewingHousehold(household)}>
                                                    <Users className="h-4 w-4 mr-2" />
                                                    View Guest Details
                                                </DropdownMenuItem>
                                                <DropdownMenuItem onClick={() => regenerateToken(household.id)}>
                                                    <RefreshCw className="h-4 w-4 mr-2" />
                                                    Regen Token
                                                </DropdownMenuItem>
                                            </DropdownMenuContent>
                                        </DropdownMenu>
                                    </TableCell>
                                </TableRow>
                            ))
                        )}
                    </TableBody>
                </Table>

                {/* Pagination */}
                {filteredHouseholds.length > 0 && (
                    <div className="flex items-center justify-between px-6 py-4 border-t">
                        <div className="flex items-center gap-4">
                            <p className="text-sm text-muted-foreground">
                                Showing {startIndex + 1} to {Math.min(endIndex, filteredHouseholds.length)} of {filteredHouseholds.length} households
                            </p>
                            <Select value={itemsPerPage.toString()} onValueChange={(value) => setItemsPerPage(Number(value))}>
                                <SelectTrigger className="w-[100px]">
                                    <SelectValue />
                                </SelectTrigger>
                                <SelectContent>
                                    <SelectItem value="10">10 / page</SelectItem>
                                    <SelectItem value="25">25 / page</SelectItem>
                                    <SelectItem value="50">50 / page</SelectItem>
                                    <SelectItem value="100">100 / page</SelectItem>
                                </SelectContent>
                            </Select>
                        </div>
                        
                        <div className="flex items-center gap-2">
                            <Button
                                variant="outline"
                                size="sm"
                                onClick={() => setCurrentPage(1)}
                                disabled={currentPage === 1}
                            >
                                <ChevronsLeft className="h-4 w-4" />
                            </Button>
                            <Button
                                variant="outline"
                                size="sm"
                                onClick={() => setCurrentPage(currentPage - 1)}
                                disabled={currentPage === 1}
                            >
                                <ChevronLeft className="h-4 w-4" />
                            </Button>
                            <span className="text-sm font-medium px-2">
                                Page {currentPage} of {totalPages}
                            </span>
                            <Button
                                variant="outline"
                                size="sm"
                                onClick={() => setCurrentPage(currentPage + 1)}
                                disabled={currentPage === totalPages}
                            >
                                <ChevronRight className="h-4 w-4" />
                            </Button>
                            <Button
                                variant="outline"
                                size="sm"
                                onClick={() => setCurrentPage(totalPages)}
                                disabled={currentPage === totalPages}
                            >
                                <ChevronsRight className="h-4 w-4" />
                            </Button>
                        </div>
                    </div>
                )}
            </Card>

            {/* Guest Details Dialog */}
            {viewingHousehold && (
                <Dialog open={!!viewingHousehold} onOpenChange={(open) => !open && setViewingHousehold(null)}>
                    <DialogContent className="max-w-3xl max-h-[90vh] overflow-y-auto">
                        <DialogHeader>
                            <DialogTitle>{viewingHousehold.displayName} - Guest Details</DialogTitle>
                            <DialogDescription>
                                Individual guest information, meal choices, and dietary restrictions
                            </DialogDescription>
                        </DialogHeader>
                        
                        <div className="mt-4 space-y-4">
                            {/* Household Info */}
                            <Card>
                                <CardHeader>
                                    <CardTitle className="text-base">Household Information</CardTitle>
                                </CardHeader>
                                <CardContent className="space-y-2 text-sm">
                                    <div className="flex justify-between">
                                        <span className="text-muted-foreground">Total Guests:</span>
                                        <span className="font-medium">{viewingHousehold.guestCount || 0}</span>
                                    </div>
                                    <div className="flex justify-between">
                                        <span className="text-muted-foreground">Contact:</span>
                                        <span>{viewingHousehold.email || 'No email'}</span>
                                    </div>
                                    <div className="flex justify-between">
                                        <span className="text-muted-foreground">Side:</span>
                                        <Badge variant={viewingHousehold.side === 'BRIDE' ? 'default' : 'secondary'}>
                                            {viewingHousehold.side}
                                        </Badge>
                                    </div>
                                </CardContent>
                            </Card>

                            {/* Individual Guests */}
                            {viewingHousehold.guests && viewingHousehold.guests.length > 0 ? (
                                <div className="space-y-3">
                                    <h3 className="font-semibold text-sm">Individual Guests:</h3>
                                    {viewingHousehold.guests.map((guest, idx) => (
                                        <Card key={idx}>
                                            <CardContent className="pt-6">
                                                <div className="space-y-3">
                                                    <div className="flex items-start justify-between">
                                                        <div>
                                                            <h4 className="font-semibold">
                                                                {guest.name || `${guest.firstName || ''} ${guest.lastName || ''}`.trim() || `Guest ${idx + 1}`}
                                                            </h4>
                                                            <Badge variant="outline" className="mt-1">
                                                                {guest.type === 'CHILD' ? 'Child' : 'Adult'}
                                                            </Badge>
                                                        </div>
                                                        {guest.attending !== undefined && (
                                                            <Badge variant={guest.attending ? 'default' : 'secondary'}>
                                                                {guest.attending ? 'Attending' : 'Not Attending'}
                                                            </Badge>
                                                        )}
                                                    </div>
                                                    
                                                    {guest.mealChoice && (
                                                        <div className="flex items-center gap-2 text-sm">
                                                            <Utensils className="h-4 w-4 text-muted-foreground" />
                                                            <span className="text-muted-foreground">Meal:</span>
                                                            <Badge variant="secondary">{guest.mealChoice}</Badge>
                                                        </div>
                                                    )}
                                                    
                                                    {guest.dietary && (
                                                        <div className="flex items-start gap-2 text-sm">
                                                            <AlertCircle className="h-4 w-4 text-amber-600 mt-0.5" />
                                                            <div>
                                                                <span className="text-muted-foreground">Dietary Restrictions:</span>
                                                                <p className="text-sm mt-1 text-amber-800 bg-amber-50 p-2 rounded">
                                                                    {guest.dietary}
                                                                </p>
                                                            </div>
                                                        </div>
                                                    )}
                                                </div>
                                            </CardContent>
                                        </Card>
                                    ))}
                                </div>
                            ) : (
                                <Card>
                                    <CardContent className="py-8 text-center text-muted-foreground">
                                        <Users className="h-12 w-12 mx-auto mb-2 opacity-50" />
                                        <p>No individual guest details available.</p>
                                        <p className="text-xs mt-1">Guests need to complete their RSVP to provide details.</p>
                                    </CardContent>
                                </Card>
                            )}
                        </div>

                        <div className="flex justify-end gap-2 mt-6">
                            <Button variant="outline" onClick={() => setViewingHousehold(null)}>
                                Close
                            </Button>
                        </div>
                    </DialogContent>
                </Dialog>
            )}

            {/* QR Codes Modal */}
            {qrModalGuest && (
                <QrCodesModal
                    household={qrModalGuest}
                    events={events}
                    onClose={() => setQrModalGuest(null)}
                />
            )}
        </div>
    );
}

// QR Codes Modal Component (reused from Households page)
function QrCodesModal({ household, events, onClose }) {
    const copyQrLink = (householdId, eventId) => {
        const url = `${window.location.origin}/q/${householdId}-${eventId}`;
        navigator.clipboard.writeText(url);
        toast.success('QR code link copied!');
    };

    const downloadQrCode = (householdId, eventId, eventName) => {
        const qrUrl = `https://api.qrserver.com/v1/create-qr-code/?size=500x500&data=${encodeURIComponent(window.location.origin + '/q/' + householdId + '-' + eventId)}`;
        const link = document.createElement('a');
        link.href = qrUrl;
        link.download = `qr-${household.displayName.replace(/\s+/g, '-')}-${eventName.replace(/\s+/g, '-')}.png`;
        document.body.appendChild(link);
        link.click();
        document.body.removeChild(link);
        toast.success('QR code downloaded!');
    };

    const openTicketPage = (householdId, eventId) => {
        window.open(`/q/${householdId}-${eventId}`, '_blank');
    };

    return (
        <Dialog open={true} onOpenChange={onClose}>
            <DialogContent className="max-w-3xl max-h-[90vh] overflow-y-auto">
                <DialogHeader>
                    <DialogTitle>QR Code Tickets - {household.displayName}</DialogTitle>
                    <DialogDescription>
                        Generate QR code tickets for check-in at events
                    </DialogDescription>
                </DialogHeader>
                
                <div className="space-y-6 py-4">
                    {events.length === 0 ? (
                        <div className="text-center py-8 text-muted-foreground">
                            <QrCode className="h-12 w-12 mx-auto mb-4 opacity-50" />
                            <p>No events available. Create an event first.</p>
                        </div>
                    ) : (
                        events.map(event => {
                            const qrUrl = `https://api.qrserver.com/v1/create-qr-code/?size=300x300&data=${encodeURIComponent(window.location.origin + '/q/' + household.id + '-' + event.id)}`;
                            
                            return (
                                <Card key={event.id} className="overflow-hidden">
                                    <CardHeader className="bg-muted/30">
                                        <div className="flex items-center justify-between">
                                            <div>
                                                <CardTitle className="text-lg">{event.name}</CardTitle>
                                                {event.venueName && (
                                                    <CardDescription className="mt-1">
                                                        {event.venueName}
                                                        {event.startAt && ` • ${new Date(event.startAt).toLocaleDateString()}`}
                                                    </CardDescription>
                                                )}
                                            </div>
                                            <Badge>{event.isPublic ? 'Public' : 'Private'}</Badge>
                                        </div>
                                    </CardHeader>
                                    <CardContent className="pt-6">
                                        <div className="grid md:grid-cols-2 gap-6">
                                            {/* QR Code Display */}
                                            <div className="flex flex-col items-center">
                                                <div className="bg-white p-4 rounded-lg border-2 border-muted mb-4">
                                                    <img 
                                                        src={qrUrl} 
                                                        alt={`QR Code for ${event.name}`}
                                                        className="w-64 h-64"
                                                    />
                                                </div>
                                                <div className="text-sm text-muted-foreground text-center">
                                                    Scan this code at check-in
                                                </div>
                                            </div>

                                            {/* Actions */}
                                            <div className="flex flex-col justify-center space-y-3">
                                                <div className="space-y-2">
                                                    <label className="text-xs text-muted-foreground">Ticket URL</label>
                                                    <div className="flex gap-2">
                                                        <Input 
                                                            value={`${window.location.origin}/q/${household.id}-${event.id}`}
                                                            readOnly
                                                            className="font-mono text-xs"
                                                        />
                                                        <Button 
                                                            size="sm" 
                                                            variant="outline"
                                                            onClick={() => copyQrLink(household.id, event.id)}
                                                        >
                                                            <Link className="h-4 w-4" />
                                                        </Button>
                                                    </div>
                                                </div>

                                                <Button 
                                                    variant="default" 
                                                    className="w-full"
                                                    onClick={() => openTicketPage(household.id, event.id)}
                                                >
                                                    <Eye className="h-4 w-4 mr-2" />
                                                    View Full Ticket Page
                                                </Button>

                                                <Button 
                                                    variant="outline" 
                                                    className="w-full"
                                                    onClick={() => downloadQrCode(household.id, event.id, event.name)}
                                                >
                                                    <Download className="h-4 w-4 mr-2" />
                                                    Download QR Code
                                                </Button>

                                                <div className="pt-4 border-t space-y-2">
                                                    <div className="text-xs text-muted-foreground">
                                                        <strong>How to use:</strong>
                                                    </div>
                                                    <ul className="text-xs text-muted-foreground space-y-1 list-disc list-inside">
                                                        <li>Send ticket URL to guest via email</li>
                                                        <li>Guest opens on phone</li>
                                                        <li>Staff scans QR at check-in</li>
                                                        <li>Or download & print for physical tickets</li>
                                                    </ul>
                                                </div>
                                            </div>
                                        </div>
                                    </CardContent>
                                </Card>
                            );
                        })
                    )}
                </div>

                <div className="flex justify-end gap-2 pt-4 border-t">
                    <Button variant="outline" onClick={onClose}>
                        Close
                    </Button>
                </div>
            </DialogContent>
        </Dialog>
    );
}
