import React, { useState, useEffect } from 'react';
import { Button } from '@/components/ui/button';
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from '@/components/ui/card';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import { Badge } from '@/components/ui/badge';
import { Dialog, DialogContent, DialogDescription, DialogHeader, DialogTitle } from '@/components/ui/dialog';
import { Skeleton } from '@/components/ui/skeleton';
import { Checkbox } from '@/components/ui/checkbox';
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from '@/components/ui/select';
import { Calendar, MapPin, Users, Utensils, Eye, Edit, Trash2, Plus, X } from 'lucide-react';
import { toast } from 'sonner';
import { apiFetch } from '../../../../utils/api';

export default function Events() {
    const [events, setEvents] = useState([]);
    const [loading, setLoading] = useState(true);
    const [editingEvent, setEditingEvent] = useState(null);
    const [showModal, setShowModal] = useState(false);

    useEffect(() => {
        loadEvents();
    }, []);

    const loadEvents = () => {
        apiFetch('/api/events')
            .then(res => res.json())
            .then(data => {
                setEvents(data.sort((a, b) => a.order - b.order));
                setLoading(false);
            })
            .catch(err => {
                console.error('Failed to load events:', err);
                setLoading(false);
            });
    };

    const openAddModal = () => {
        setEditingEvent({
            name: '',
            venueName: '',
            venueAddress: '',
            startAt: '',
            endAt: '',
            description: '',
            capacity: '',
            isPublic: true,
            mealsEnabled: false,
            mealOptionsJson: [
                { code: 'BEEF', name: 'Beef' },
                { code: 'CHICKEN', name: 'Chicken' },
                { code: 'FISH', name: 'Fish' },
                { code: 'VEG', name: 'Vegetarian' }
            ],
            order: events.length,
            nameFont: 'Bodoni Moda',
            nameOne: '',
            nameTwo: '',
            andText: 'AND'
        });
        setShowModal(true);
    };

    const openEditModal = (event) => {
        setEditingEvent({
            ...event,
            mealOptionsJson: event.mealOptionsJson || []
        });
        setShowModal(true);
    };

    const saveEvent = async () => {
        if (!editingEvent.name) {
            toast.error('Event name is required');
            return;
        }

        const endpoint = editingEvent.id ? `/api/events/${editingEvent.id}` : '/api/events';
        const method = editingEvent.id ? 'PATCH' : 'POST';

        try {
            const response = await apiFetch(endpoint, {
                method,
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify(editingEvent)
            });

            if (response.ok) {
                setShowModal(false);
                setEditingEvent(null);
                loadEvents();
                toast.success('Event saved successfully!');
            } else {
                toast.error('Failed to save event');
            }
        } catch (err) {
            console.error('Save error:', err);
            toast.error('Error saving event');
        }
    };

    const deleteEvent = async (id, name) => {
        if (!confirm(`⚠️ Delete "${name}"? This will remove all associated RSVPs!`)) return;
        
        try {
            await apiFetch(`/api/events/${id}`, { method: 'DELETE' });
            loadEvents();
            toast.success('Event deleted');
        } catch (err) {
            toast.error('Failed to delete event');
        }
    };

    const addMealOption = () => {
        setEditingEvent({
            ...editingEvent,
            mealOptionsJson: [
                ...editingEvent.mealOptionsJson,
                { code: '', name: '' }
            ]
        });
    };

    const updateMealOption = (index, field, value) => {
        const updated = [...editingEvent.mealOptionsJson];
        updated[index][field] = value;
        setEditingEvent({ ...editingEvent, mealOptionsJson: updated });
    };

    const removeMealOption = (index) => {
        const updated = editingEvent.mealOptionsJson.filter((_, i) => i !== index);
        setEditingEvent({ ...editingEvent, mealOptionsJson: updated });
    };

    if (loading) {
        return (
            <div className="space-y-6">
                <Skeleton className="h-10 w-64" />
                <div className="grid gap-4">
                    {[...Array(2)].map((_, i) => (
                        <Card key={i}>
                            <CardHeader>
                                <Skeleton className="h-6 w-48" />
                                <Skeleton className="h-4 w-64 mt-2" />
                            </CardHeader>
                            <CardContent>
                                <Skeleton className="h-20 w-full" />
                            </CardContent>
                        </Card>
                    ))}
                </div>
            </div>
        );
    }

    return (
        <div className="space-y-6">
            {/* Header */}
            <div className="flex flex-col gap-4 sm:flex-row sm:items-center sm:justify-between">
                <div>
                    <h1 className="text-3xl font-bold tracking-tight">Events</h1>
                    <p className="text-muted-foreground">Manage wedding events and receptions</p>
                </div>
                <Button onClick={openAddModal}>
                    <Plus className="h-4 w-4 mr-2" />
                    Add Event
                </Button>
            </div>

            {events.length === 0 ? (
                <Card>
                    <CardContent className="flex flex-col items-center justify-center py-12">
                        <Calendar className="h-12 w-12 text-muted-foreground mb-4" />
                        <p className="text-muted-foreground mb-4">No events created yet</p>
                        <Button onClick={openAddModal}>
                            <Plus className="h-4 w-4 mr-2" />
                            Create Your First Event
                        </Button>
                    </CardContent>
                </Card>
            ) : (
                <div className="grid gap-4">
                    {events.map(event => (
                        <Card key={event.id}>
                            <CardHeader>
                                <div className="flex justify-between items-start">
                                    <div className="flex-1">
                                        <CardTitle className="text-xl">
                                            {event.nameOne && event.nameTwo 
                                                ? `${event.nameOne} ${event.andText || 'AND'} ${event.nameTwo}` 
                                                : event.name}
                                        </CardTitle>
                                        <CardDescription className="mt-2 space-y-1">
                                            {event.venueName && (
                                                <div className="flex items-center gap-2">
                                                    <MapPin className="h-4 w-4" />
                                                    {event.venueName}
                                                </div>
                                            )}
                                            {event.startAt && (
                                                <div className="flex items-center gap-2">
                                                    <Calendar className="h-4 w-4" />
                                                    <span>
                                                        {new Date(event.startAt.replace(' ', 'T')).toLocaleString('en-US', {
                                                            weekday: 'long',
                                                            year: 'numeric',
                                                            month: 'long',
                                                            day: 'numeric',
                                                            hour: 'numeric',
                                                            minute: '2-digit'
                                                        })}
                                                        {event.endAt && (
                                                            <> - {new Date(event.endAt.replace(' ', 'T')).toLocaleTimeString('en-US', {
                                                                hour: 'numeric',
                                                                minute: '2-digit'
                                                            })}</>
                                                        )}
                                                    </span>
                                                </div>
                                            )}
                                        </CardDescription>
                                    </div>
                                    <div className="flex gap-2 flex-wrap justify-end">
                                        <Button 
                                            variant="outline" 
                                            size="sm"
                                            onClick={() => window.open(`/rsvp/${event.slug || event.id}`, '_blank')}
                                        >
                                            <Eye className="h-4 w-4 mr-2" />
                                            View
                                        </Button>
                                        <Button variant="outline" size="sm" onClick={() => openEditModal(event)}>
                                            <Edit className="h-4 w-4 mr-2" />
                                            Edit
                                        </Button>
                                        <Button 
                                            variant="outline" 
                                            size="sm"
                                            onClick={() => deleteEvent(event.id, event.name)}
                                        >
                                            <Trash2 className="h-4 w-4 text-red-600" />
                                        </Button>
                                    </div>
                                </div>
                            </CardHeader>
                            <CardContent>
                                {event.description && (
                                    <p className="text-sm text-muted-foreground mb-3">{event.description}</p>
                                )}
                                <div className="flex gap-2 flex-wrap">
                                    {event.mealsEnabled && (
                                        <Badge variant="secondary">
                                            <Utensils className="h-3 w-3 mr-1" />
                                            Meals Enabled
                                        </Badge>
                                    )}
                                    {event.isPublic && (
                                        <Badge variant="default" className="bg-green-500">
                                            <Eye className="h-3 w-3 mr-1" />
                                            Public
                                        </Badge>
                                    )}
                                    {event.capacity && (
                                        <Badge variant="outline">
                                            <Users className="h-3 w-3 mr-1" />
                                            Capacity: {event.capacity}
                                        </Badge>
                                    )}
                                </div>

                                {event.mealsEnabled && event.mealOptionsJson && event.mealOptionsJson.length > 0 && (
                                    <div className="mt-4 pt-4 border-t">
                                        <h4 className="text-sm font-medium mb-2">Meal Options:</h4>
                                        <div className="flex gap-2 flex-wrap">
                                            {event.mealOptionsJson.map(meal => (
                                                <Badge key={meal.code} variant="outline">
                                                    {meal.name} ({meal.code})
                                                </Badge>
                                            ))}
                                        </div>
                                    </div>
                                )}
                            </CardContent>
                        </Card>
                    ))}
                </div>
            )}

            {/* Event Modal */}
            {showModal && editingEvent && (
                <Dialog open={showModal} onOpenChange={(open) => {
                    if (!open) {
                        setShowModal(false);
                        setEditingEvent(null);
                    }
                }}>
                    <DialogContent className="max-w-2xl max-h-[90vh] overflow-y-auto">
                        <DialogHeader>
                            <DialogTitle>{editingEvent.id ? 'Edit Event' : 'Create Event'}</DialogTitle>
                            <DialogDescription>
                                {editingEvent.id ? 'Update event details and settings' : 'Create a new wedding event'}
                            </DialogDescription>
                        </DialogHeader>
                        
                        <div className="space-y-4 mt-4">
                            <div className="space-y-2">
                                <Label htmlFor="nameFont">Event Name Font</Label>
                                <Select 
                                    value={editingEvent.nameFont || 'Bodoni Moda'} 
                                    onValueChange={(value) => setEditingEvent({...editingEvent, nameFont: value})}
                                >
                                    <SelectTrigger>
                                        <SelectValue />
                                    </SelectTrigger>
                                    <SelectContent>
                                        <SelectItem value="Bodoni Moda">Bodoni Moda (Elegant Serif)</SelectItem>
                                        <SelectItem value="Playfair Display">Playfair Display (Classic Serif)</SelectItem>
                                        <SelectItem value="Cormorant">Cormorant (Refined Serif)</SelectItem>
                                        <SelectItem value="Cinzel">Cinzel (Roman Style)</SelectItem>
                                        <SelectItem value="Great Vibes">Great Vibes (Script)</SelectItem>
                                        <SelectItem value="Inter">Inter (Modern Sans-serif)</SelectItem>
                                    </SelectContent>
                                </Select>
                            </div>
                            
                            <div className="space-y-2">
                                <Label htmlFor="eventName">Event Name *</Label>
                                <Input
                                    id="eventName"
                                    value={editingEvent.name}
                                    onChange={(e) => setEditingEvent({...editingEvent, name: e.target.value})}
                                    placeholder="e.g., Ceremony, Reception, Cocktail Hour"
                                />
                            </div>

                        

                            <div className="space-y-2">
                                <Label className="text-muted-foreground text-xs">Alternative: Couples' Names (overrides Event Name if provided)</Label>
                                <div className="grid grid-cols-2 gap-4">
                                    <div className="space-y-2">
                                        <Label htmlFor="nameOne">Name One</Label>
                                        <Input
                                            id="nameOne"
                                            value={editingEvent.nameOne || ''}
                                            onChange={(e) => setEditingEvent({...editingEvent, nameOne: e.target.value})}
                                            placeholder="e.g., Henrietta"
                                        />
                                    </div>
                                    <div className="space-y-2">
                                        <Label htmlFor="nameTwo">Name Two</Label>
                                        <Input
                                            id="nameTwo"
                                            value={editingEvent.nameTwo || ''}
                                            onChange={(e) => setEditingEvent({...editingEvent, nameTwo: e.target.value})}
                                            placeholder="e.g., Jamie"
                                        />
                                    </div>
                                </div>
                            </div>

                            <div className="space-y-2">
                                <Label htmlFor="andText">Separator Text</Label>
                                <Input
                                    id="andText"
                                    value={editingEvent.andText || 'AND'}
                                    onChange={(e) => setEditingEvent({...editingEvent, andText: e.target.value})}
                                    placeholder="e.g., AND, &, +"
                                    maxLength={10}
                                />
                                <p className="text-xs text-muted-foreground">Text/symbol between names (e.g., "AND", "&", "+")</p>
                            </div>

                            <div className="grid grid-cols-2 gap-4">
                                <div className="space-y-2">
                                    <Label htmlFor="venueName">Venue Name</Label>
                                    <Input
                                        id="venueName"
                                        value={editingEvent.venueName}
                                        onChange={(e) => setEditingEvent({...editingEvent, venueName: e.target.value})}
                                        placeholder="The Grand Hall"
                                    />
                                </div>
                                <div className="space-y-2">
                                    <Label htmlFor="capacity">Capacity</Label>
                                    <Input
                                        id="capacity"
                                        type="number"
                                        value={editingEvent.capacity}
                                        onChange={(e) => setEditingEvent({...editingEvent, capacity: e.target.value})}
                                        placeholder="150"
                                    />
                                </div>
                            </div>

                            <div className="space-y-2">
                                <Label htmlFor="venueAddress">Venue Address</Label>
                                <Input
                                    id="venueAddress"
                                    value={editingEvent.venueAddress}
                                    onChange={(e) => setEditingEvent({...editingEvent, venueAddress: e.target.value})}
                                    placeholder="123 Main Street, City, State ZIP"
                                />
                            </div>

                            <div className="space-y-2">
                                <Label htmlFor="eventDate">Event Date</Label>
                                <Input
                                    id="eventDate"
                                    type="date"
                                    value={editingEvent.startAt ? editingEvent.startAt.replace(' ', 'T').split('T')[0] : ''}
                                    onChange={(e) => {
                                        const date = e.target.value;
                                        const startTime = editingEvent.startAt ? editingEvent.startAt.replace(' ', 'T').split('T')[1]?.substring(0, 5) || '00:00' : '00:00';
                                        const endTime = editingEvent.endAt ? editingEvent.endAt.replace(' ', 'T').split('T')[1]?.substring(0, 5) || '00:00' : '00:00';
                                        setEditingEvent({
                                            ...editingEvent,
                                            startAt: date ? `${date}T${startTime}` : '',
                                            endAt: date ? `${date}T${endTime}` : ''
                                        });
                                    }}
                                />
                            </div>

                            <div className="grid grid-cols-2 gap-4">
                                <div className="space-y-2">
                                    <Label htmlFor="startTime">Start Time</Label>
                                    <Input
                                        id="startTime"
                                        type="time"
                                        value={editingEvent.startAt ? editingEvent.startAt.replace(' ', 'T').split('T')[1]?.substring(0, 5) || '' : ''}
                                        onChange={(e) => {
                                            const time = e.target.value;
                                            const date = editingEvent.startAt ? editingEvent.startAt.replace(' ', 'T').split('T')[0] : new Date().toISOString().split('T')[0];
                                            setEditingEvent({...editingEvent, startAt: `${date}T${time}`});
                                        }}
                                    />
                                </div>
                                <div className="space-y-2">
                                    <Label htmlFor="endTime">End Time (Optional)</Label>
                                    <Input
                                        id="endTime"
                                        type="time"
                                        value={editingEvent.endAt ? editingEvent.endAt.replace(' ', 'T').split('T')[1]?.substring(0, 5) || '' : ''}
                                        onChange={(e) => {
                                            const time = e.target.value;
                                            const date = editingEvent.startAt ? editingEvent.startAt.replace(' ', 'T').split('T')[0] : new Date().toISOString().split('T')[0];
                                            setEditingEvent({...editingEvent, endAt: time ? `${date}T${time}` : ''});
                                        }}
                                    />
                                </div>
                            </div>

                            <div className="space-y-2">
                                <Label htmlFor="description">Description</Label>
                                <textarea
                                    id="description"
                                    value={editingEvent.description}
                                    onChange={(e) => setEditingEvent({...editingEvent, description: e.target.value})}
                                    className="flex min-h-[80px] w-full rounded-md border border-input bg-background px-3 py-2 text-sm ring-offset-background placeholder:text-muted-foreground focus-visible:outline-none focus-visible:ring-2 focus-visible:ring-ring focus-visible:ring-offset-2"
                                    rows="3"
                                    placeholder="Additional details about this event..."
                                />
                            </div>

                            <div className="flex flex-col gap-3">
                                <div className="flex items-center gap-2">
                                    <Checkbox
                                        id="isPublic"
                                        checked={editingEvent.isPublic}
                                        onCheckedChange={(checked) => setEditingEvent({...editingEvent, isPublic: checked})}
                                    />
                                    <Label htmlFor="isPublic" className="text-sm font-normal cursor-pointer">
                                        Visible on RSVP page
                                    </Label>
                                </div>
                                <div className="flex items-center gap-2">
                                    <Checkbox
                                        id="mealsEnabled"
                                        checked={editingEvent.mealsEnabled}
                                        onCheckedChange={(checked) => setEditingEvent({...editingEvent, mealsEnabled: checked})}
                                    />
                                    <Label htmlFor="mealsEnabled" className="text-sm font-normal cursor-pointer">
                                        Enable meal selection
                                    </Label>
                                </div>
                            </div>

                            {editingEvent.mealsEnabled && (
                                <Card>
                                    <CardHeader>
                                        <div className="flex justify-between items-center">
                                            <CardTitle className="text-base">Meal Options</CardTitle>
                                            <Button size="sm" variant="outline" onClick={addMealOption}>
                                                <Plus className="h-4 w-4 mr-2" />
                                                Add Option
                                            </Button>
                                        </div>
                                    </CardHeader>
                                    <CardContent>
                                        <div className="space-y-2">
                                            {editingEvent.mealOptionsJson.map((meal, idx) => (
                                                <div key={idx} className="flex gap-2">
                                                    <Input
                                                        value={meal.code}
                                                        onChange={(e) => updateMealOption(idx, 'code', e.target.value)}
                                                        className="w-24"
                                                        placeholder="CODE"
                                                    />
                                                    <Input
                                                        value={meal.name}
                                                        onChange={(e) => updateMealOption(idx, 'name', e.target.value)}
                                                        className="flex-1"
                                                        placeholder="Meal Name"
                                                    />
                                                    <Button
                                                        type="button"
                                                        variant="ghost"
                                                        size="sm"
                                                        onClick={() => removeMealOption(idx)}
                                                    >
                                                        <X className="h-4 w-4 text-red-600" />
                                                    </Button>
                                                </div>
                                            ))}
                                        </div>
                                    </CardContent>
                                </Card>
                            )}
                        </div>

                        <div className="flex gap-2 mt-6">
                            <Button onClick={saveEvent} className="flex-1">
                                {editingEvent.id ? 'Update' : 'Create'} Event
                            </Button>
                            <Button variant="outline" onClick={() => {
                                setShowModal(false);
                                setEditingEvent(null);
                            }}>
                                Cancel
                            </Button>
                        </div>
                    </DialogContent>
                </Dialog>
            )}
        </div>
    );
}
