import React, { useState, useEffect } from 'react';
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from '@/components/ui/card';
import { Badge } from '@/components/ui/badge';
import { Button } from '@/components/ui/button';
import { Skeleton } from '@/components/ui/skeleton';
import { Users, UserCheck, UserX, Calendar, Mail, Settings } from 'lucide-react';
import { apiFetch } from '../../../../utils/api';

export default function Dashboard() {
    const [stats, setStats] = useState(null);
    const [loading, setLoading] = useState(true);

    useEffect(() => {
        apiFetch('/api/dashboard/stats')
            .then(res => res.json())
            .then(data => {
                setStats(data);
                setLoading(false);
            });
    }, []);

    if (loading) {
        return (
            <div className="space-y-6">
                <Skeleton className="h-10 w-64" />
                <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-4">
                    {[...Array(4)].map((_, i) => (
                        <Card key={i}>
                            <CardHeader>
                                <Skeleton className="h-4 w-20" />
                            </CardHeader>
                            <CardContent>
                                <Skeleton className="h-8 w-16" />
                            </CardContent>
                        </Card>
                    ))}
                </div>
            </div>
        );
    }

    const statCards = [
        {
            title: 'Total Invited',
            value: stats.totalInvited,
            icon: Users,
            description: 'Households invited',
            trend: null,
            variant: 'default'
        },
        {
            title: 'Responded',
            value: stats.responded,
            icon: Mail,
            description: `${stats.responseRate}% response rate`,
            trend: stats.responseRate >= 50 ? 'up' : 'neutral',
            variant: 'default'
        },
        {
            title: 'Attending',
            value: stats.yesCount,
            icon: UserCheck,
            description: 'Confirmed guests',
            trend: 'up',
            variant: 'success'
        },
        {
            title: 'Declined',
            value: stats.noCount,
            icon: UserX,
            description: 'Unable to attend',
            trend: null,
            variant: 'destructive'
        }
    ];

    const quickActions = [
        {
            title: 'Manage Guests',
            description: 'Add, edit, or remove guests from your list',
            href: '/admin/guests',
            icon: Users,
            color: 'text-blue-600'
        },
        {
            title: 'Manage Events',
            description: 'Set up wedding events and schedules',
            href: '/admin/events',
            icon: Calendar,
            color: 'text-purple-600'
        },
        {
            title: 'View RSVPs',
            description: 'See who\'s coming and meal choices',
            href: '/admin/rsvps',
            icon: UserCheck,
            color: 'text-green-600'
        },
        {
            title: 'Settings',
            description: 'Configure your wedding website',
            href: '/admin/settings',
            icon: Settings,
            color: 'text-gray-600'
        }
    ];

    return (
        <div className="space-y-8">
            {/* Header */}
            <div className="flex items-center justify-between">
                <div>
                    <h1 className="text-3xl font-bold tracking-tight">Dashboard</h1>
                    <p className="text-muted-foreground">Welcome back! Here's your wedding overview.</p>
                </div>
                <Badge variant="outline" className="text-sm">
                    Last updated: {new Date().toLocaleTimeString()}
                </Badge>
            </div>

            {/* Stats Grid */}
            <div className="grid gap-4 grid-cols-1 sm:grid-cols-2 lg:grid-cols-4">
                {statCards.map((stat, index) => {
                    const Icon = stat.icon;
                    return (
                        <Card key={index} className="overflow-hidden">
                            <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                                <CardTitle className="text-sm font-medium">
                                    {stat.title}
                                </CardTitle>
                                <Icon className={`h-4 w-4 ${stat.variant === 'success' ? 'text-green-600' : stat.variant === 'destructive' ? 'text-red-600' : 'text-muted-foreground'}`} />
                            </CardHeader>
                            <CardContent>
                                <div className="text-2xl font-bold">{stat.value}</div>
                                <p className="text-xs text-muted-foreground mt-1">
                                    {stat.description}
                                </p>
                            </CardContent>
                        </Card>
                    );
                })}
            </div>

            {/* Quick Actions */}
            <Card>
                <CardHeader>
                    <CardTitle>Quick Actions</CardTitle>
                    <CardDescription>Jump to common tasks</CardDescription>
                </CardHeader>
                <CardContent>
                    <div className="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-4 gap-4">
                        {quickActions.map((action, index) => {
                            const Icon = action.icon;
                            return (
                                <a
                                    key={index}
                                    href={action.href}
                                    className="group relative flex flex-col items-start gap-2 rounded-lg border p-4 hover:shadow-md transition-all"
                                >
                                    <div className={`p-2 rounded-md bg-accent/50 ${action.color}`}>
                                        <Icon className="h-5 w-5" />
                                    </div>
                                    <div>
                                        <h3 className="font-semibold text-sm group-hover:underline">
                                            {action.title}
                                        </h3>
                                        <p className="text-xs text-muted-foreground mt-1">
                                            {action.description}
                                        </p>
                                    </div>
                                </a>
                            );
                        })}
                    </div>
                </CardContent>
            </Card>
        </div>
    );
}
