import React, { useState } from 'react';
import { Routes, Route, Link, useLocation } from 'react-router-dom';
import { Button } from '@/components/ui/button';
import { Separator } from '@/components/ui/separator';
import { Toaster } from '@/components/ui/sonner';
import { cn } from '@/lib/utils';
import { 
    LayoutDashboard, 
    Users, 
    UserCheck, 
    Calendar, 
    Mail, 
    Armchair, 
    FileText, 
    Palette, 
    Settings as SettingsIcon,
    LogOut,
    ChevronLeft,
    ChevronRight,
    Heart,
    QrCode
} from 'lucide-react';
import Dashboard from './pages/Dashboard';
import Guests from './pages/Guests';
import Households from './pages/Households';
import Events from './pages/Events';
import RSVPs from './pages/RSVPs';
import CheckIn from './pages/CheckIn';
import Seating from './pages/Seating';
import Pages from './pages/Pages';
import Theme from './pages/Theme';
import Settings from './pages/Settings';

function App() {
    const location = useLocation();
    const [sidebarOpen, setSidebarOpen] = useState(true);
    
    const navItems = [
        { path: '/', label: 'Dashboard', icon: LayoutDashboard, exact: true },
        { path: '/households', label: 'Households', icon: Users },
        { path: '/guests', label: 'Guests', icon: UserCheck },
        { path: '/events', label: 'Events', icon: Calendar },
        { path: '/rsvps', label: 'RSVPs', icon: Mail },
        { path: '/checkin', label: 'Check-In', icon: QrCode },
        { path: '/seating', label: 'Seating', icon: Armchair },
        // { path: '/pages', label: 'Pages', icon: FileText },
        { path: '/theme', label: 'Theme', icon: Palette },
        { path: '/settings', label: 'Settings', icon: SettingsIcon },
    ];

    const isActive = (item) => {
        if (item.exact) {
            return location.pathname === item.path;
        }
        return location.pathname.startsWith(item.path);
    };

    return (
        <div className="min-h-screen bg-background flex">
            {/* Sidebar */}
            <aside className={cn(
                "bg-card border-r transition-all duration-300 flex flex-col",
                sidebarOpen ? "w-64" : "w-20"
            )}>
                {/* Header */}
                <div className="h-16 flex items-center justify-between px-4 border-b">
                    {sidebarOpen && (
                        <div className="flex items-center gap-2">
                            <Heart className="h-5 w-5 text-primary fill-primary" />
                            <h1 className="text-lg font-bold">Wedding Admin</h1>
                        </div>
                    )}
                    <Button 
                        variant="ghost" 
                        size="icon"
                        onClick={() => setSidebarOpen(!sidebarOpen)}
                        className="ml-auto"
                    >
                        {sidebarOpen ? (
                            <ChevronLeft className="h-4 w-4" />
                        ) : (
                            <ChevronRight className="h-4 w-4" />
                        )}
                    </Button>
                </div>

                {/* Navigation */}
                <nav className="flex-1 p-3 space-y-1 overflow-y-auto">
                    {navItems.map((item) => {
                        const Icon = item.icon;
                        const active = isActive(item);
                        
                        return (
                            <Link
                                key={item.path}
                                to={item.path}
                                className={cn(
                                    "flex items-center gap-3 px-3 py-2 rounded-lg transition-all",
                                    "hover:bg-accent hover:text-accent-foreground",
                                    active && "bg-primary text-primary-foreground shadow-sm",
                                    !sidebarOpen && "justify-center"
                                )}
                                title={!sidebarOpen ? item.label : undefined}
                            >
                                <Icon className={cn("h-5 w-5 flex-shrink-0", active && "")} />
                                {sidebarOpen && (
                                    <span className={cn("text-sm font-medium", active && "font-semibold")}>
                                        {item.label}
                                    </span>
                                )}
                            </Link>
                        );
                    })}
                </nav>

                {/* Footer */}
                <Separator />
                <div className="p-3">
                    <a 
                        href={`${window.__BASE_PATH__ || ''}/logout`}
                        className={cn(
                            "flex items-center gap-3 px-3 py-2 rounded-lg transition-colors",
                            "hover:bg-destructive/10 hover:text-destructive",
                            !sidebarOpen && "justify-center"
                        )}
                        title={!sidebarOpen ? "Logout" : undefined}
                    >
                        <LogOut className="h-5 w-5 flex-shrink-0" />
                        {sidebarOpen && <span className="text-sm font-medium">Logout</span>}
                    </a>
                </div>
            </aside>

            {/* Main Content */}
            <main className="flex-1 overflow-auto bg-muted/40">
                <div className="p-8">
                    <Routes>
                        <Route path="/" element={<Dashboard />} />
                        <Route path="/households" element={<Households />} />
                        <Route path="/guests" element={<Guests />} />
                        <Route path="/events" element={<Events />} />
                        <Route path="/rsvps" element={<RSVPs />} />
                        <Route path="/checkin" element={<CheckIn />} />
                        <Route path="/seating" element={<Seating />} />
                        <Route path="/pages" element={<Pages />} />
                        <Route path="/theme" element={<Theme />} />
                        <Route path="/settings" element={<Settings />} />
                    </Routes>
                </div>
            </main>
            
            {/* Toast Notifications */}
            <Toaster />
        </div>
    );
}

export default App;
