import { Controller } from '@hotwired/stimulus';
import { Html5Qrcode } from 'html5-qrcode';

export default class extends Controller {
    static targets = ['reader', 'manualInput', 'recentList', 'status'];

    connect() {
        this.isScanning = false;
        this.html5QrCode = null;
        this.recentCheckins = [];
        this.loadRecentCheckins();
        this.startScanner();
    }

    disconnect() {
        this.stopScanner();
    }

    async startScanner() {
        try {
            this.html5QrCode = new Html5Qrcode('qr-reader');
            
            await this.html5QrCode.start(
                { facingMode: 'environment' }, // Use back camera
                {
                    fps: 10,
                    qrbox: { width: 250, height: 250 }
                },
                this.onScanSuccess.bind(this),
                this.onScanFailure.bind(this)
            );
            
            this.isScanning = true;
            this.showStatus('Scanner active - point camera at QR code', 'success');
        } catch (err) {
            console.error('Failed to start scanner:', err);
            this.showStatus('Camera not available. Use manual check-in below.', 'warning');
        }
    }

    async stopScanner() {
        if (this.html5QrCode && this.isScanning) {
            try {
                await this.html5QrCode.stop();
                this.isScanning = false;
            } catch (err) {
                console.error('Failed to stop scanner:', err);
            }
        }
    }

    async onScanSuccess(decodedText, decodedResult) {
        // Parse the URL: /q/5-2 or /q/g/23-2
        const url = new URL(decodedText);
        const path = url.pathname;
        
        let householdId = null;
        let guestId = null;
        let eventId = null;
        
        // Match household: /q/{householdId}-{eventId}
        const householdMatch = path.match(/\/q\/(\d+)-(\d+)/);
        if (householdMatch) {
            householdId = parseInt(householdMatch[1]);
            eventId = parseInt(householdMatch[2]);
        }
        
        // Match guest: /q/g/{guestId}-{eventId}
        const guestMatch = path.match(/\/q\/g\/(\d+)-(\d+)/);
        if (guestMatch) {
            guestId = parseInt(guestMatch[1]);
            eventId = parseInt(guestMatch[2]);
        }
        
        if (eventId) {
            await this.submitCheckin({
                eventId,
                householdId,
                guestId,
                method: 'QR_SCAN'
            });
        }
    }

    onScanFailure(error) {
        // Ignore scan failures - they happen constantly while searching
    }

    async manualCheckin(event) {
        event.preventDefault();
        
        const input = this.manualInputTarget.value.trim();
        if (!input) return;
        
        // Parse input: "5-2" for household or "g23-2" for guest
        let householdId = null;
        let guestId = null;
        let eventId = null;
        
        if (input.startsWith('g')) {
            // Guest format: g23-2
            const match = input.match(/g(\d+)-(\d+)/);
            if (match) {
                guestId = parseInt(match[1]);
                eventId = parseInt(match[2]);
            }
        } else {
            // Household format: 5-2
            const match = input.match(/(\d+)-(\d+)/);
            if (match) {
                householdId = parseInt(match[1]);
                eventId = parseInt(match[2]);
            }
        }
        
        if (eventId) {
            await this.submitCheckin({
                eventId,
                householdId,
                guestId,
                method: 'MANUAL'
            });
            this.manualInputTarget.value = '';
        } else {
            this.showStatus('Invalid format. Use: 5-2 or g23-2', 'error');
        }
    }

    async submitCheckin(data) {
        try {
            const response = await fetch('/api/checkin', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                },
                body: JSON.stringify({
                    ...data,
                    countDelta: 1
                })
            });

            const result = await response.json();

            if (result.success) {
                const name = data.guestId ? `Guest #${data.guestId}` : `Household #${data.householdId}`;
                this.showStatus(`✓ Checked in: ${name}`, 'success');
                this.addRecentCheckin(name, data.method);
                
                // Play success sound
                this.playBeep();
            } else {
                this.showStatus(`Error: ${result.error || 'Unknown error'}`, 'error');
            }
        } catch (err) {
            console.error('Check-in failed:', err);
            this.showStatus('Network error - check-in failed', 'error');
        }
    }

    addRecentCheckin(name, method) {
        const now = new Date().toLocaleTimeString();
        this.recentCheckins.unshift({
            name,
            method,
            time: now
        });
        
        // Keep only last 10
        if (this.recentCheckins.length > 10) {
            this.recentCheckins.pop();
        }
        
        this.updateRecentList();
    }

    updateRecentList() {
        if (!this.hasRecentListTarget) return;
        
        if (this.recentCheckins.length === 0) {
            this.recentListTarget.innerHTML = '<p class="text-sm text-muted-foreground">No recent check-ins</p>';
            return;
        }
        
        this.recentListTarget.innerHTML = this.recentCheckins.map(checkin => `
            <div class="flex items-center justify-between p-3 bg-muted/50 rounded">
                <div>
                    <p class="font-medium">${checkin.name}</p>
                    <p class="text-xs text-muted-foreground">${checkin.time} • ${checkin.method}</p>
                </div>
                <svg class="w-5 h-5 text-green-500" fill="currentColor" viewBox="0 0 20 20">
                    <path fill-rule="evenodd" d="M10 18a8 8 0 100-16 8 8 0 000 16zm3.707-9.293a1 1 0 00-1.414-1.414L9 10.586 7.707 9.293a1 1 0 00-1.414 1.414l2 2a1 1 0 001.414 0l4-4z" clip-rule="evenodd"/>
                </svg>
            </div>
        `).join('');
    }

    loadRecentCheckins() {
        // Load from sessionStorage if available
        const stored = sessionStorage.getItem('recentCheckins');
        if (stored) {
            try {
                this.recentCheckins = JSON.parse(stored);
                this.updateRecentList();
            } catch (e) {
                this.recentCheckins = [];
            }
        }
    }

    showStatus(message, type = 'info') {
        if (!this.hasStatusTarget) {
            // Create status element if it doesn't exist
            const statusDiv = document.createElement('div');
            statusDiv.setAttribute('data-checkin-target', 'status');
            statusDiv.className = 'fixed top-4 right-4 z-50 p-4 rounded-lg shadow-lg max-w-md';
            document.body.appendChild(statusDiv);
            this.statusTarget = statusDiv;
        }

        const colors = {
            success: 'bg-green-500 text-white',
            error: 'bg-red-500 text-white',
            warning: 'bg-yellow-500 text-black',
            info: 'bg-blue-500 text-white'
        };

        this.statusTarget.className = `fixed top-4 right-4 z-50 p-4 rounded-lg shadow-lg max-w-md ${colors[type]}`;
        this.statusTarget.textContent = message;
        this.statusTarget.style.display = 'block';

        // Auto-hide after 3 seconds
        setTimeout(() => {
            if (this.hasStatusTarget) {
                this.statusTarget.style.display = 'none';
            }
        }, 3000);
    }

    playBeep() {
        // Simple beep sound using Web Audio API
        try {
            const audioContext = new (window.AudioContext || window.webkitAudioContext)();
            const oscillator = audioContext.createOscillator();
            const gainNode = audioContext.createGain();
            
            oscillator.connect(gainNode);
            gainNode.connect(audioContext.destination);
            
            oscillator.frequency.value = 800;
            oscillator.type = 'sine';
            
            gainNode.gain.setValueAtTime(0.3, audioContext.currentTime);
            gainNode.gain.exponentialRampToValueAtTime(0.01, audioContext.currentTime + 0.1);
            
            oscillator.start(audioContext.currentTime);
            oscillator.stop(audioContext.currentTime + 0.1);
        } catch (e) {
            // Silently fail if audio not supported
        }
    }
}
