import React, { useState, useEffect } from 'react';
import { createRoot } from 'react-dom/client';

function PublicRsvpApp({ eventSlug }) {
    const [loading, setLoading] = useState(true);
    const [submitted, setSubmitted] = useState(false);
    const [settings, setSettings] = useState(null);
    const [events, setEvents] = useState([]);
    const [formData, setFormData] = useState({
        firstName: '',
        lastName: '',
        email: '',
        phone: '',
        numberOfAdults: 1,
        numberOfChildren: 0,
        message: '',
        events: []
    });

    useEffect(() => {
        const url = eventSlug 
            ? `/api/rsvp/public/events/${eventSlug}`
            : '/api/rsvp/public/events';
            
        fetch(url)
            .then(res => {
                if (!res.ok) {
                    throw new Error(`HTTP error! status: ${res.status}`);
                }
                return res.json();
            })
            .then(data => {
                setSettings(data.settings);
                setEvents(data.events);
                // Initialize event selections
                setFormData(prev => ({
                    ...prev,
                    events: data.events.map(e => ({
                        eventId: e.id,
                        status: 'YES',
                        mealChoice: e.mealOptions?.[0]?.code || null,
                        dietary: ''
                    }))
                }));
                setLoading(false);
            })
            .catch(error => {
                console.error('Error fetching events:', error);
                alert('Failed to load event data. Please check the console for details.');
                setLoading(false);
            });
    }, [eventSlug]);

    const handleSubmit = async (e) => {
        e.preventDefault();
        
        if (!formData.firstName || !formData.email) {
            alert('Please fill in your name and email');
            return;
        }

        try {
            const response = await fetch('/api/rsvp/public/submit', {
                method: 'POST',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify({
                    ...formData,
                    name: `${formData.firstName} ${formData.lastName}`.trim()
                })
            });

            const result = await response.json();
            
            if (response.ok) {
                setSubmitted(true);
            } else {
                alert(result.error || 'Failed to submit RSVP');
            }
        } catch (error) {
            alert('Error submitting RSVP. Please try again.');
        }
    };

    const updateEventField = (eventId, field, value) => {
        setFormData(prev => ({
            ...prev,
            events: prev.events.map(e => 
                e.eventId === eventId ? { ...e, [field]: value } : e
            )
        }));
    };

    if (loading || !settings) {
        return (
            <div className="min-h-screen flex items-center justify-center">
                <div className="text-xl">Loading...</div>
            </div>
        );
    }

    if (submitted) {
        return (
            <div className="min-h-screen flex items-center justify-center bg-gradient-to-br from-pink-50 to-purple-50 p-4">
                <div className="max-w-md w-full bg-white rounded-2xl shadow-xl p-8 text-center">
                    <div className="text-6xl mb-4">🎉</div>
                    <h1 className="text-3xl font-bold text-gray-800 mb-4">Thank You!</h1>
                    <p className="text-gray-600 mb-2">Your RSVP has been received.</p>
                    <p className="text-gray-600">We can't wait to celebrate with you!</p>
                    <div className="mt-8">
                        <a href="/" className="text-pink-600 hover:text-pink-700 font-medium">
                            ← Back to Home
                        </a>
                    </div>
                </div>
            </div>
        );
    }

    return (
        <div className="min-h-screen bg-gradient-to-br from-pink-50 to-purple-50 py-12 px-4">
            <div className="max-w-3xl mx-auto">
                {/* Header */}
                <div className="text-center mb-12">
                    <h1 className="text-5xl font-bold text-gray-800 mb-4">{settings.coupleNames}</h1>
                    <p className="text-xl text-gray-600">{settings.locationText}</p>
                    {settings.weddingDate && (
                        <p className="text-lg text-gray-500 mt-2">
                            {new Date(settings.weddingDate).toLocaleDateString('en-US', {
                                weekday: 'long',
                                year: 'numeric',
                                month: 'long',
                                day: 'numeric'
                            })}
                        </p>
                    )}
                </div>

                {/* RSVP Form */}
                <div className="bg-white rounded-2xl shadow-xl p-8">
                    <h2 className="text-3xl font-bold text-gray-800 mb-2 text-center">RSVP</h2>
                    <p className="text-gray-600 mb-8 text-center">Please let us know if you can join us!</p>

                    <form onSubmit={handleSubmit} className="space-y-6">
                        {/* Contact Information */}
                        <div className="space-y-4">
                            <h3 className="text-lg font-semibold text-gray-700 border-b pb-2">Your Information</h3>
                            
                            <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                                <div>
                                    <label className="block text-sm font-medium text-gray-700 mb-1">
                                        First Name *
                                    </label>
                                    <input
                                        type="text"
                                        required
                                        value={formData.firstName}
                                        onChange={(e) => setFormData({...formData, firstName: e.target.value})}
                                        className="w-full px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-pink-500 focus:border-transparent"
                                    />
                                </div>
                                <div>
                                    <label className="block text-sm font-medium text-gray-700 mb-1">
                                        Last Name *
                                    </label>
                                    <input
                                        type="text"
                                        required
                                        value={formData.lastName}
                                        onChange={(e) => setFormData({...formData, lastName: e.target.value})}
                                        className="w-full px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-pink-500 focus:border-transparent"
                                    />
                                </div>
                            </div>

                            <div>
                                <label className="block text-sm font-medium text-gray-700 mb-1">
                                    Email Address *
                                </label>
                                <input
                                    type="email"
                                    required
                                    value={formData.email}
                                    onChange={(e) => setFormData({...formData, email: e.target.value})}
                                    className="w-full px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-pink-500 focus:border-transparent"
                                />
                            </div>

                            <div>
                                <label className="block text-sm font-medium text-gray-700 mb-1">
                                    Phone Number (Optional)
                                </label>
                                <input
                                    type="tel"
                                    value={formData.phone}
                                    onChange={(e) => setFormData({...formData, phone: e.target.value})}
                                    className="w-full px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-pink-500 focus:border-transparent"
                                />
                            </div>

                            <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                                <div>
                                    <label className="block text-sm font-medium text-gray-700 mb-1">
                                        Number of Adults *
                                    </label>
                                    <input
                                        type="number"
                                        min="0"
                                        required
                                        value={formData.numberOfAdults}
                                        onChange={(e) => setFormData({...formData, numberOfAdults: parseInt(e.target.value) || 0})}
                                        className="w-full px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-pink-500 focus:border-transparent"
                                    />
                                </div>
                                <div>
                                    <label className="block text-sm font-medium text-gray-700 mb-1">
                                        Number of Children
                                    </label>
                                    <input
                                        type="number"
                                        min="0"
                                        value={formData.numberOfChildren}
                                        onChange={(e) => setFormData({...formData, numberOfChildren: parseInt(e.target.value) || 0})}
                                        className="w-full px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-pink-500 focus:border-transparent"
                                    />
                                </div>
                            </div>
                        </div>

                        {/* Events */}
                        <div className="space-y-6">
                            <h3 className="text-lg font-semibold text-gray-700 border-b pb-2">Event Details</h3>
                            
                            {events.map((event, idx) => {
                                const eventRsvp = formData.events.find(e => e.eventId === event.id);
                                
                                return (
                                    <div key={event.id} className="bg-gray-50 rounded-lg p-6">
                                        <h4 className="text-xl font-semibold text-gray-800 mb-2">{event.name}</h4>
                                        
                                        {event.description && (
                                            <p className="text-gray-600 text-sm mb-3">{event.description}</p>
                                        )}
                                        
                                        {event.venueName && (
                                            <div className="text-sm text-gray-600 mb-2">
                                                <span className="font-medium">📍 Venue:</span> {event.venueName}
                                            </div>
                                        )}
                                        
                                        {event.startAt && (
                                            <div className="text-sm text-gray-600 mb-4">
                                                <span className="font-medium">🕐 Time:</span> {new Date(event.startAt).toLocaleString('en-US', {
                                                    weekday: 'short',
                                                    month: 'short',
                                                    day: 'numeric',
                                                    hour: 'numeric',
                                                    minute: '2-digit'
                                                })}
                                            </div>
                                        )}

                                        <div className="space-y-4">
                                            <div>
                                                <label className="block text-sm font-medium text-gray-700 mb-2">
                                                    Will you attend?
                                                </label>
                                                <div className="flex gap-4">
                                                    <label className="flex items-center">
                                                        <input
                                                            type="radio"
                                                            checked={eventRsvp?.status === 'YES'}
                                                            onChange={() => updateEventField(event.id, 'status', 'YES')}
                                                            className="mr-2"
                                                        />
                                                        ✓ Yes, I'll be there
                                                    </label>
                                                    <label className="flex items-center">
                                                        <input
                                                            type="radio"
                                                            checked={eventRsvp?.status === 'NO'}
                                                            onChange={() => updateEventField(event.id, 'status', 'NO')}
                                                            className="mr-2"
                                                        />
                                                        ✗ Sorry, can't make it
                                                    </label>
                                                </div>
                                            </div>

                                            {eventRsvp?.status === 'YES' && event.mealsEnabled && event.mealOptions?.length > 0 && (
                                                <div>
                                                    <label className="block text-sm font-medium text-gray-700 mb-2">
                                                        Meal Choice
                                                    </label>
                                                    <select
                                                        value={eventRsvp.mealChoice || ''}
                                                        onChange={(e) => updateEventField(event.id, 'mealChoice', e.target.value)}
                                                        className="w-full px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-pink-500 focus:border-transparent"
                                                    >
                                                        {event.mealOptions.map(meal => (
                                                            <option key={meal.code} value={meal.code}>
                                                                {meal.name}
                                                            </option>
                                                        ))}
                                                    </select>
                                                </div>
                                            )}

                                            {eventRsvp?.status === 'YES' && (
                                                <div>
                                                    <label className="block text-sm font-medium text-gray-700 mb-2">
                                                        Dietary Restrictions (Optional)
                                                    </label>
                                                    <input
                                                        type="text"
                                                        value={eventRsvp.dietary || ''}
                                                        onChange={(e) => updateEventField(event.id, 'dietary', e.target.value)}
                                                        placeholder="e.g., vegetarian, gluten-free, allergies"
                                                        className="w-full px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-pink-500 focus:border-transparent"
                                                    />
                                                </div>
                                            )}
                                        </div>
                                    </div>
                                );
                            })}
                        </div>

                        {/* Message */}
                        <div>
                            <label className="block text-sm font-medium text-gray-700 mb-2">
                                Message to the Couple (Optional)
                            </label>
                            <textarea
                                value={formData.message}
                                onChange={(e) => setFormData({...formData, message: e.target.value})}
                                rows="4"
                                placeholder="Share your well wishes..."
                                className="w-full px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-pink-500 focus:border-transparent"
                            />
                        </div>

                        {/* Submit */}
                        <button
                            type="submit"
                            className="w-full bg-pink-600 hover:bg-pink-700 text-white font-semibold py-3 px-6 rounded-lg transition-colors duration-200"
                        >
                            Submit RSVP
                        </button>

                        {settings.rsvpDeadline && (
                            <p className="text-sm text-gray-500 text-center">
                                Please RSVP by {new Date(settings.rsvpDeadline).toLocaleDateString()}
                            </p>
                        )}
                    </form>
                </div>
            </div>
        </div>
    );
}

// Export for mounting
export function mount(rootElement, eventSlug = '') {
    const root = createRoot(rootElement);
    root.render(<PublicRsvpApp eventSlug={eventSlug || null} />);
}

// Make available globally
window.PublicRsvpApp = { mount };

export default PublicRsvpApp;
