import React, { useState, useEffect } from 'react';
import HouseholdForm from './HouseholdForm';
import PerGuestForm from './PerGuestForm';
import SuccessView from './SuccessView';

export default function InvitePage({ token }) {
    const [data, setData] = useState(null);
    const [loading, setLoading] = useState(true);
    const [submitted, setSubmitted] = useState(false);
    const [error, setError] = useState(null);

    useEffect(() => {
        fetch(`/api/rsvp/${token}`)
            .then(res => res.json())
            .then(data => {
                if (data.error) {
                    setError(data.error);
                } else {
                    setData(data);
                }
                setLoading(false);
            })
            .catch(err => {
                setError('Failed to load invitation');
                setLoading(false);
            });
    }, [token]);

    const handleSubmit = async (formData) => {
        try {
            const res = await fetch(`/api/rsvp/${token}`, {
                method: 'POST',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify(formData)
            });

            const result = await res.json();
            
            if (result.error) {
                alert(result.error);
            } else {
                setSubmitted(true);
            }
        } catch (err) {
            alert('Failed to submit RSVP. Please try again.');
        }
    };

    if (loading) {
        return (
            <div className="min-h-screen flex items-center justify-center">
                <div className="text-lg">Loading...</div>
            </div>
        );
    }

    if (error) {
        return (
            <div className="min-h-screen flex items-center justify-center">
                <div className="text-red-600">{error}</div>
            </div>
        );
    }

    if (submitted) {
        return <SuccessView data={data} />;
    }

    const { mode, household, settings, events } = data;

    return (
        <div className="min-h-screen bg-background py-12 px-4">
            <div className="max-w-3xl mx-auto">
                <div className="text-center mb-8">
                    <h1 className="text-4xl font-bold mb-2">{settings.coupleNames}</h1>
                    {settings.weddingDate && (
                        <p className="text-lg text-muted-foreground">
                            {new Date(settings.weddingDate).toLocaleDateString('en-US', {
                                weekday: 'long',
                                year: 'numeric',
                                month: 'long',
                                day: 'numeric'
                            })}
                        </p>
                    )}
                    {settings.locationText && (
                        <p className="text-muted-foreground mt-2">{settings.locationText}</p>
                    )}
                </div>

                <div className="bg-card rounded-lg shadow-lg p-6 mb-6">
                    <h2 className="text-2xl font-semibold mb-4">
                        Welcome, {household.displayName}!
                    </h2>
                    
                    {settings.rsvpDeadline && new Date() > new Date(settings.rsvpDeadline) ? (
                        <div className="bg-destructive/10 border border-destructive text-destructive rounded-lg p-4">
                            <p className="font-semibold">RSVP Deadline Passed</p>
                            <p>The RSVP deadline has passed. Please contact us directly if you need to make changes.</p>
                        </div>
                    ) : (
                        <>
                            {mode === 'PER_GUEST' ? (
                                <PerGuestForm 
                                    events={events} 
                                    onSubmit={handleSubmit}
                                />
                            ) : (
                                <HouseholdForm 
                                    events={events}
                                    mode={mode}
                                    onSubmit={handleSubmit}
                                />
                            )}
                        </>
                    )}
                </div>
            </div>
        </div>
    );
}
