import React, { useState } from 'react';
import { Button } from '@/components/ui/button';

export default function HouseholdForm({ events, mode, onSubmit }) {
    const [formData, setFormData] = useState({
        events: events.map(event => ({
            eventId: event.id,
            status: event.rsvp?.status || null,
            adults: event.rsvp?.adultsCount || 0,
            children: event.rsvp?.childrenCount || 0,
            meals: event.rsvp?.mealCounts || {}
        })),
        dietary: events[0]?.rsvp?.dietary || '',
        namesText: mode === 'HOUSEHOLD_WITH_OPTIONAL_NAMES' ? (events[0]?.rsvp?.namesText || '') : '',
        message: events[0]?.rsvp?.message || ''
    });

    const updateEvent = (index, updates) => {
        const newEvents = [...formData.events];
        newEvents[index] = { ...newEvents[index], ...updates };
        setFormData({ ...formData, events: newEvents });
    };

    const updateMealCount = (eventIndex, mealCode, count) => {
        const newEvents = [...formData.events];
        const meals = { ...newEvents[eventIndex].meals };
        meals[mealCode] = parseInt(count) || 0;
        newEvents[eventIndex].meals = meals;
        setFormData({ ...formData, events: newEvents });
    };

    const handleSubmit = (e) => {
        e.preventDefault();
        
        // Validation
        for (const eventData of formData.events) {
            const event = events.find(e => e.id === eventData.eventId);
            
            if (eventData.status === 'YES') {
                const total = eventData.adults + eventData.children;
                if (total < 1) {
                    alert('Please specify at least one adult or child for events you\'re attending.');
                    return;
                }

                if (event.mealsEnabled) {
                    const mealTotal = Object.values(eventData.meals).reduce((sum, count) => sum + count, 0);
                    if (mealTotal !== total) {
                        alert(`Meal selections must equal total headcount (${total}) for ${event.name}`);
                        return;
                    }
                }
            }
        }

        onSubmit(formData);
    };

    return (
        <form onSubmit={handleSubmit} className="space-y-6">
            {events.map((event, index) => {
                const eventData = formData.events[index];
                const total = eventData.adults + eventData.children;

                return (
                    <div key={event.id} className="border rounded-lg p-4">
                        <h3 className="text-xl font-semibold mb-2">{event.name}</h3>
                        {event.venueName && (
                            <p className="text-sm text-muted-foreground mb-2">{event.venueName}</p>
                        )}
                        {event.startAt && (
                            <p className="text-sm text-muted-foreground mb-4">
                                {new Date(event.startAt).toLocaleString()}
                            </p>
                        )}

                        <div className="space-y-4">
                            <div>
                                <label className="block text-sm font-medium mb-2">Will you attend?</label>
                                <div className="flex gap-2">
                                    {['YES', 'NO', 'MAYBE'].map(status => (
                                        <button
                                            key={status}
                                            type="button"
                                            onClick={() => updateEvent(index, { status })}
                                            className={`px-4 py-2 rounded border ${
                                                eventData.status === status
                                                    ? 'bg-primary text-primary-foreground'
                                                    : 'bg-background hover:bg-accent'
                                            }`}
                                        >
                                            {status}
                                        </button>
                                    ))}
                                </div>
                            </div>

                            {eventData.status === 'YES' && (
                                <>
                                    <div className="grid grid-cols-2 gap-4">
                                        <div>
                                            <label className="block text-sm font-medium mb-1">Adults</label>
                                            <input
                                                type="number"
                                                min="0"
                                                value={eventData.adults}
                                                onChange={(e) => updateEvent(index, { adults: parseInt(e.target.value) || 0 })}
                                                className="w-full px-3 py-2 border rounded"
                                            />
                                        </div>
                                        <div>
                                            <label className="block text-sm font-medium mb-1">Children</label>
                                            <input
                                                type="number"
                                                min="0"
                                                value={eventData.children}
                                                onChange={(e) => updateEvent(index, { children: parseInt(e.target.value) || 0 })}
                                                className="w-full px-3 py-2 border rounded"
                                            />
                                        </div>
                                    </div>

                                    {event.mealsEnabled && event.mealOptions && (
                                        <div>
                                            <label className="block text-sm font-medium mb-2">
                                                Meal Selections (Total: {total})
                                            </label>
                                            <div className="space-y-2">
                                                {event.mealOptions.map(meal => (
                                                    <div key={meal.code} className="flex items-center gap-2">
                                                        <label className="flex-1">{meal.name}</label>
                                                        <input
                                                            type="number"
                                                            min="0"
                                                            value={eventData.meals[meal.code] || 0}
                                                            onChange={(e) => updateMealCount(index, meal.code, e.target.value)}
                                                            className="w-20 px-3 py-2 border rounded"
                                                        />
                                                    </div>
                                                ))}
                                            </div>
                                        </div>
                                    )}
                                </>
                            )}
                        </div>
                    </div>
                );
            })}

            <div>
                <label className="block text-sm font-medium mb-1">Dietary Restrictions</label>
                <textarea
                    value={formData.dietary}
                    onChange={(e) => setFormData({ ...formData, dietary: e.target.value })}
                    className="w-full px-3 py-2 border rounded"
                    rows="2"
                    placeholder="Any allergies or dietary restrictions?"
                />
            </div>

            {mode === 'HOUSEHOLD_WITH_OPTIONAL_NAMES' && (
                <div>
                    <label className="block text-sm font-medium mb-1">Guest Names (optional)</label>
                    <textarea
                        value={formData.namesText}
                        onChange={(e) => setFormData({ ...formData, namesText: e.target.value })}
                        className="w-full px-3 py-2 border rounded"
                        rows="2"
                        placeholder="Enter names separated by commas"
                    />
                </div>
            )}

            <div>
                <label className="block text-sm font-medium mb-1">Message to Couple (optional)</label>
                <textarea
                    value={formData.message}
                    onChange={(e) => setFormData({ ...formData, message: e.target.value })}
                    className="w-full px-3 py-2 border rounded"
                    rows="3"
                    placeholder="Send a message to the happy couple!"
                />
            </div>

            <Button type="submit" className="w-full">Submit RSVP</Button>
        </form>
    );
}
