import React, { useState, useEffect } from 'react';
import { createRoot } from 'react-dom/client';
import SuccessView from './SuccessView';

// Content configuration
const CONTENT = {
    preheader: 'LOOKING FORWARD TO CELEBRATING WITH YOU',
    rsvpTitle: 'R  S  V  P',
    successTitle: 'THANK YOU FOR YOUR RESPONSE',
    successMessage: "We've saved your RSVP and will be in touch soon.",
};

function PublicRsvpApp({ eventSlug }) {
    const [loading, setLoading] = useState(true);
    const [submitted, setSubmitted] = useState(false);
    const [submittedData, setSubmittedData] = useState(null);
    const [settings, setSettings] = useState(null);
    const [events, setEvents] = useState([]);
    const [attending, setAttending] = useState('YES');
    const [errors, setErrors] = useState({});
    const [formData, setFormData] = useState({
        firstName: '',
        lastName: '',
        email: '',
        phone: '',
        numberOfAdults: 1,
        numberOfChildren: 0,
        message: '',
        adults: [{ name: '', meal: '', dietary: '' }],
        children: [],
        events: []
    });

    // Default styles (fallback if theme not loaded)
    const [styles, setStyles] = useState({
        bg: '#2A0713',
        ink: '#C9B699',
        inkSoft: 'rgba(201,182,153,0.8)',
        fieldBorder: '#7B5F51',
        accent: '#B6A37F',
        accentHover: '#C8B588',
        error: '#E8CBB5'
    });
    const [themeLoaded, setThemeLoaded] = useState(false);

    useEffect(() => {
        // Load theme first
        fetch('/api/rsvp/public/theme')
            .then(res => res.json())
            .then(theme => {
                setStyles(theme);
                setThemeLoaded(true);
            })
            .catch(() => {
                setThemeLoaded(true); // Use default if theme fails
            });
        
        const url = eventSlug 
            ? `/api/rsvp/public/events/${eventSlug}`
            : '/api/rsvp/public/events';
            
        fetch(url)
            .then(res => {
                if (!res.ok) {
                    throw new Error(`HTTP error! status: ${res.status}`);
                }
                return res.json();
            })
            .then(data => {
                setSettings(data.settings);
                setEvents(data.events);
                // Initialize event selections
                setFormData(prev => ({
                    ...prev,
                    events: data.events.map(e => ({
                        eventId: e.id,
                        status: 'YES',
                        mealChoice: e.mealOptions?.[0]?.code || null,
                        dietary: ''
                    }))
                }));
                setLoading(false);
            })
            .catch(error => {
                console.error('Error fetching events:', error);
                alert('Failed to load event data. Please check the console for details.');
                setLoading(false);
            });
    }, [eventSlug]);

    // Update adults/children arrays when counts change
    useEffect(() => {
        const adultsCount = parseInt(formData.numberOfAdults) || 0;
        const childrenCount = parseInt(formData.numberOfChildren) || 0;
        
        setFormData(prev => ({
            ...prev,
            adults: Array.from({ length: adultsCount }, (_, i) => prev.adults[i] || { name: '', meal: '', dietary: '' }),
            children: Array.from({ length: childrenCount }, (_, i) => prev.children[i] || { name: '', meal: '', dietary: '' })
        }));
    }, [formData.numberOfAdults, formData.numberOfChildren]);

    const validateForm = () => {
        const newErrors = {};
        
        if (!formData.firstName.trim()) newErrors.firstName = 'First name is required';
        if (!formData.lastName.trim()) newErrors.lastName = 'Last name is required';
        if (!formData.email.trim()) {
            newErrors.email = 'Email is required';
        } else if (!/^[^\s@]+@[^\s@]+\.[^\s@]+$/.test(formData.email)) {
            newErrors.email = 'Please enter a valid email';
        }
        
        if (attending === 'YES') {
            if (formData.numberOfAdults < 1) {
                newErrors.numberOfAdults = 'Please add at least one adult guest';
            }
        }
        
        setErrors(newErrors);
        return Object.keys(newErrors).length === 0;
    };

    const handleSubmit = async (e) => {
        e.preventDefault();
        
        if (!validateForm()) {
            return;
        }

        try {
            const response = await fetch('/api/rsvp/public/submit', {
                method: 'POST',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify({
                    ...formData,
                    attending,
                    name: `${formData.firstName} ${formData.lastName}`.trim()
                })
            });

            const result = await response.json();
            
            if (response.ok) {
                // Store submitted data for success page display
                // Only include events that were in the submission (formData.events)
                const submittedEventIds = formData.events.map(e => e.eventId);
                const eventsWithRsvp = events
                    .filter(event => submittedEventIds.includes(event.id))
                    .map(event => ({
                        ...event,
                        rsvp: {
                            status: attending,
                            adultsCount: formData.numberOfAdults,
                            childrenCount: formData.numberOfChildren,
                            guestDetails: {
                                adults: formData.adults,
                                children: formData.children
                            }
                        }
                    }));
                
                setSubmittedData({
                    settings: {
                        coupleNames: settings.coupleNames
                    },
                    events: eventsWithRsvp
                });
                setSubmitted(true);
                window.scrollTo(0, 0);
            } else {
                setErrors({ submit: result.error || 'Failed to submit RSVP' });
            }
        } catch (error) {
            setErrors({ submit: 'Error submitting RSVP. Please try again.' });
        }
    };

    const updateAdult = (index, field, value) => {
        setFormData(prev => ({
            ...prev,
            adults: prev.adults.map((adult, i) => i === index ? { ...adult, [field]: value } : adult)
        }));
    };

    const updateChild = (index, field, value) => {
        setFormData(prev => ({
            ...prev,
            children: prev.children.map((child, i) => i === index ? { ...child, [field]: value } : child)
        }));
    };

    const updateEventField = (eventId, field, value) => {
        setFormData(prev => ({
            ...prev,
            events: prev.events.map(e => 
                e.eventId === eventId ? { ...e, [field]: value } : e
            )
        }));
    };

    // Show neutral loading screen while theme loads
    if (!themeLoaded) {
        return (
            <div className="min-h-screen flex items-center justify-center" style={{ background: '#000' }}>
                <div style={{ color: '#fff', fontFamily: 'Inter, sans-serif', letterSpacing: '0.2em', fontSize: '0.875rem' }} className="uppercase">Loading...</div>
            </div>
        );
    }

    if (loading || !settings) {
        return (
            <div style={{ background: styles.bg }} className="min-h-screen flex items-center justify-center">
                <div style={{ color: styles.ink, fontFamily: 'Inter, sans-serif', letterSpacing: '0.2em' }} className="text-sm uppercase">Loading...</div>
            </div>
        );
    }

    if (submitted && submittedData) {
        return <SuccessView data={submittedData} styles={styles} />;
    }

    const mealOptions = events[0]?.mealOptions || [];

    return (
        <div className="min-h-screen py-16 px-5" style={{ background: styles.bg }}>
            <div className="max-w-[880px] mx-auto">
                {/* Header */}
                <div className="text-center mb-10">
                    <p className="font-sans text-xs tracking-wider uppercase mb-5" style={{ color: styles.inkSoft }}>
                        {CONTENT.preheader}
                    </p>
                    
                    <h1 className="font-serif leading-none mb-6 mt-[34px] font-bold uppercase tracking-[0.3em]" 
                        style={{ 
                            fontFamily: "'Bodoni Moda', serif",
                            fontSize: 'clamp(3rem, 10vw, 6rem)',
                            color: styles.ink
                        }}>
                        {CONTENT.rsvpTitle}
                    </h1>
                    
                    {events[0]?.nameOne && events[0]?.nameTwo ? (
                        <div className="grid grid-cols-1 md:grid-cols-[1fr_auto_1fr] items-center mb-3 gap-2 md:gap-6">
                            <div className="text-center md:text-right">
                                <span className="text-[30px] tracking-wider uppercase" 
                                    style={{ 
                                        fontFamily: `'${events[0]?.nameFont || 'Bodoni Moda'}', serif`,
                                        color: styles.ink
                                    }}>
                                    {events[0].nameOne.toUpperCase()}
                                </span>
                            </div>
                            <span className="font-sans text-[20px] tracking-wide whitespace-nowrap text-center" style={{ color: styles.ink }}>
                                {events[0]?.andText || 'AND'}
                            </span>
                            <div className="text-center md:text-left">
                                <span className="text-[30px] tracking-wider uppercase" 
                                    style={{ 
                                        fontFamily: `'${events[0]?.nameFont || 'Bodoni Moda'}', serif`,
                                        color: styles.ink
                                    }}>
                                    {events[0].nameTwo.toUpperCase()}
                                </span>
                            </div>
                        </div>
                    ) : (
                        <p className="text-2xl tracking-wider uppercase mb-3" 
                            style={{ 
                                fontFamily: `'${events[0]?.nameFont || 'Bodoni Moda'}', serif`,
                                color: styles.ink
                            }}>
                            {events[0]?.name?.toUpperCase() || 'WEDDING CEREMONY'}
                        </p>
                    )}
                    
                    {events[0]?.startAt && (
                        <p className="font-sans text-xs tracking-wider uppercase mb-2" style={{ color: styles.inkSoft }}>
                            {new Date(events[0].startAt.replace(' ', 'T')).toLocaleDateString('en-US', {
                                weekday: 'long',
                                year: 'numeric',
                                month: 'long',
                                day: 'numeric'
                            }).toUpperCase()}
                        </p>
                    )}
                    
                    {events[0] && (
                        <p className="font-sans text-xs tracking-wider uppercase text-center" style={{ color: styles.inkSoft }}>
                            {events[0].venueName ? `VENUE: ${events[0].venueName.toUpperCase()}` : ''}{events[0].venueName && events[0].startAt ? '. ' : ''}TIME: {events[0].startAt ? new Date(events[0].startAt.replace(' ', 'T')).toLocaleTimeString('en-US', { hour: 'numeric', minute: '2-digit' }).toUpperCase() : ''}{events[0].endAt ? ` - ${new Date(events[0].endAt.replace(' ', 'T')).toLocaleTimeString('en-US', { hour: 'numeric', minute: '2-digit' }).toUpperCase()}` : ''}
                        </p>
                    )}
                    {events[0]?.description && (
                        <p className="font-sans text-sm text-center mt-4" style={{ color: styles.inkSoft, maxWidth: '600px', margin: '16px auto 0' }}>
                            {events[0].description}
                        </p>
                    )}
                </div>

                {/* Form */}
                <form onSubmit={handleSubmit} className="mt-12">
                    {/* Attendance Choice */}
                    <div className="mb-10 text-center">
                        <p className="font-sans text-xs tracking-widest uppercase mb-4" style={{ color: styles.inkSoft }}>
                            ARE YOU ATTENDING
                        </p>
                        <div className="flex gap-12 justify-center items-center">
                            <label style={{ display: 'flex', alignItems: 'center', gap: '10px', cursor: 'pointer' }}>
                                <input
                                    type="radio"
                                    checked={attending === 'YES'}
                                    onChange={() => setAttending('YES')}
                                    style={{ display: 'none' }}
                                />
                                <div style={{ 
                                    width: '20px', 
                                    height: '20px', 
                                    borderRadius: '50%', 
                                    border: `1px solid ${styles.ink}`,
                                    display: 'flex',
                                    alignItems: 'center',
                                    justifyContent: 'center',
                                    transition: 'border-color 120ms'
                                }}>
                                    {attending === 'YES' && <div style={{ width: '10px', height: '10px', borderRadius: '50%', background: styles.ink }} />}
                                </div>
                                <span style={{ 
                                    fontFamily: 'Inter, sans-serif', 
                                    fontSize: '0.875rem', 
                                    color: attending === 'YES' ? styles.ink : styles.inkSoft,
                                    letterSpacing: '0.05em',
                                    transition: 'color 120ms'
                                }}>YES</span>
                            </label>
                            <label style={{ display: 'flex', alignItems: 'center', gap: '10px', cursor: 'pointer' }}>
                                <input
                                    type="radio"
                                    checked={attending === 'NO'}
                                    onChange={() => setAttending('NO')}
                                    style={{ display: 'none' }}
                                />
                                <div style={{ 
                                    width: '20px', 
                                    height: '20px', 
                                    borderRadius: '50%', 
                                    border: `1px solid ${styles.ink}`,
                                    display: 'flex',
                                    alignItems: 'center',
                                    justifyContent: 'center',
                                    transition: 'border-color 120ms'
                                }}>
                                    {attending === 'NO' && <div style={{ width: '10px', height: '10px', borderRadius: '50%', background: styles.ink }} />}
                                </div>
                                <span style={{ 
                                    fontFamily: 'Inter, sans-serif', 
                                    fontSize: '0.875rem', 
                                    color: attending === 'NO' ? styles.ink : styles.inkSoft,
                                    letterSpacing: '0.05em',
                                    transition: 'color 120ms'
                                }}>NO</span>
                            </label>
                        </div>
                    </div>

                    {/* YOUR DETAILS divider */}
                    <div style={{ display: 'flex', alignItems: 'center', gap: '16px', marginBottom: '32px' }}>
                        <div style={{ flex: 1, height: '1px', background: styles.ink }} />
                        <p style={{ 
                            fontFamily: 'Inter, sans-serif', 
                            fontSize: '0.75rem', 
                            color: styles.inkSoft,
                            letterSpacing: '0.15em',
                            textTransform: 'uppercase'
                        }}>
                            YOUR DETAILS
                        </p>
                        <div style={{ flex: 1, height: '1px', background: styles.ink }} />
                    </div>

                    {/* Form Fields */}
                    <div style={{ display: 'grid', gridTemplateColumns: '1fr', gap: '20px', marginBottom: '28px' }}>
                        <div style={{ display: 'grid', gridTemplateColumns: 'repeat(auto-fit, minmax(280px, 1fr))', gap: '20px' }}>
                            <div>
                                <label style={{ 
                                    display: 'block', 
                                    fontFamily: 'Inter, sans-serif', 
                                    fontSize: '0.75rem', 
                                    color: styles.inkSoft,
                                    letterSpacing: '0.1em',
                                    textTransform: 'uppercase',
                                    marginBottom: '8px'
                                }}>
                                    FIRST NAME *
                                </label>
                                <input
                                    type="text"
                                    required
                                    value={formData.firstName}
                                    onChange={(e) => setFormData({...formData, firstName: e.target.value})}
                                    style={{
                                        width: '100%',
                                        padding: '12px 16px',
                                        background: 'transparent',
                                        border: `1px solid ${styles.fieldBorder}`,
                                        borderRadius: '4px',
                                        color: styles.inkSoft,
                                        fontFamily: 'Inter, sans-serif',
                                        fontSize: '0.9rem',
                                        outline: 'none',
                                        transition: 'border-color 120ms, box-shadow 120ms',
                                        minHeight: '44px'
                                    }}
                                    onFocus={(e) => {
                                        e.target.style.borderColor = styles.ink;
                                        e.target.style.boxShadow = `0 0 0 1px ${styles.ink}`;
                                    }}
                                    onBlur={(e) => {
                                        e.target.style.borderColor = styles.fieldBorder;
                                        e.target.style.boxShadow = 'none';
                                    }}
                                />
                                {errors.firstName && <p style={{ fontSize: '0.75rem', color: styles.error, marginTop: '4px' }}>{errors.firstName}</p>}
                            </div>
                            <div>
                                <label style={{ 
                                    display: 'block', 
                                    fontFamily: 'Inter, sans-serif', 
                                    fontSize: '0.75rem', 
                                    color: styles.inkSoft,
                                    letterSpacing: '0.1em',
                                    textTransform: 'uppercase',
                                    marginBottom: '8px'
                                }}>
                                    LAST NAME *
                                </label>
                                <input
                                    type="text"
                                    required
                                    value={formData.lastName}
                                    onChange={(e) => setFormData({...formData, lastName: e.target.value})}
                                    style={{
                                        width: '100%',
                                        padding: '12px 16px',
                                        background: 'transparent',
                                        border: `1px solid ${styles.fieldBorder}`,
                                        borderRadius: '4px',
                                        color: styles.inkSoft,
                                        fontFamily: 'Inter, sans-serif',
                                        fontSize: '0.9rem',
                                        outline: 'none',
                                        transition: 'border-color 120ms, box-shadow 120ms',
                                        minHeight: '44px'
                                    }}
                                    onFocus={(e) => {
                                        e.target.style.borderColor = styles.ink;
                                        e.target.style.boxShadow = `0 0 0 1px ${styles.ink}`;
                                    }}
                                    onBlur={(e) => {
                                        e.target.style.borderColor = styles.fieldBorder;
                                        e.target.style.boxShadow = 'none';
                                    }}
                                />
                                {errors.lastName && <p style={{ fontSize: '0.75rem', color: styles.error, marginTop: '4px' }}>{errors.lastName}</p>}
                            </div>
                        </div>

                        <div style={{ display: 'grid', gridTemplateColumns: 'repeat(auto-fit, minmax(280px, 1fr))', gap: '20px' }}>
                            <div>
                                <label style={{ 
                                    display: 'block', 
                                    fontFamily: 'Inter, sans-serif', 
                                    fontSize: '0.75rem', 
                                    color: styles.inkSoft,
                                    letterSpacing: '0.1em',
                                    textTransform: 'uppercase',
                                    marginBottom: '8px'
                                }}>
                                    EMAIL *
                                </label>
                                <input
                                    type="email"
                                    required
                                    value={formData.email}
                                    onChange={(e) => setFormData({...formData, email: e.target.value})}
                                    style={{
                                        width: '100%',
                                        padding: '12px 16px',
                                        background: 'transparent',
                                        border: `1px solid ${styles.fieldBorder}`,
                                        borderRadius: '4px',
                                        color: styles.inkSoft,
                                        fontFamily: 'Inter, sans-serif',
                                        fontSize: '0.9rem',
                                        outline: 'none',
                                        transition: 'border-color 120ms, box-shadow 120ms',
                                        minHeight: '44px'
                                    }}
                                    onFocus={(e) => {
                                        e.target.style.borderColor = styles.ink;
                                        e.target.style.boxShadow = `0 0 0 1px ${styles.ink}`;
                                    }}
                                    onBlur={(e) => {
                                        e.target.style.borderColor = styles.fieldBorder;
                                        e.target.style.boxShadow = 'none';
                                    }}
                                />
                                {errors.email && <p style={{ fontSize: '0.75rem', color: styles.error, marginTop: '4px' }}>{errors.email}</p>}
                            </div>
                            <div>
                                <label style={{ 
                                    display: 'block', 
                                    fontFamily: 'Inter, sans-serif', 
                                    fontSize: '0.75rem', 
                                    color: styles.inkSoft,
                                    letterSpacing: '0.1em',
                                    textTransform: 'uppercase',
                                    marginBottom: '8px'
                                }}>
                                    PHONE (OPTIONAL)
                                </label>
                                <input
                                    type="tel"
                                    value={formData.phone}
                                    onChange={(e) => setFormData({...formData, phone: e.target.value})}
                                    style={{
                                        width: '100%',
                                        padding: '12px 16px',
                                        background: 'transparent',
                                        border: `1px solid ${styles.fieldBorder}`,
                                        borderRadius: '4px',
                                        color: styles.inkSoft,
                                        fontFamily: 'Inter, sans-serif',
                                        fontSize: '0.9rem',
                                        outline: 'none',
                                        transition: 'border-color 120ms, box-shadow 120ms',
                                        minHeight: '44px'
                                    }}
                                    onFocus={(e) => {
                                        e.target.style.borderColor = styles.ink;
                                        e.target.style.boxShadow = `0 0 0 1px ${styles.ink}`;
                                    }}
                                    onBlur={(e) => {
                                        e.target.style.borderColor = styles.fieldBorder;
                                        e.target.style.boxShadow = 'none';
                                    }}
                                />
                            </div>
                        </div>

                        {attending === 'YES' && (
                            <div style={{ display: 'grid', gridTemplateColumns: 'repeat(auto-fit, minmax(280px, 1fr))', gap: '20px' }}>
                                <div>
                                    <label style={{ 
                                        display: 'block', 
                                        fontFamily: 'Inter, sans-serif', 
                                        fontSize: '0.75rem', 
                                        color: styles.inkSoft,
                                        letterSpacing: '0.1em',
                                        textTransform: 'uppercase',
                                        marginBottom: '8px'
                                    }}>
                                        NUMBER OF ADULTS *
                                    </label>
                                    <input
                                        type="number"
                                        min="0"
                                        required
                                        value={formData.numberOfAdults}
                                        onChange={(e) => setFormData({...formData, numberOfAdults: parseInt(e.target.value) || 0})}
                                        style={{
                                            width: '100%',
                                            padding: '12px 16px',
                                            background: 'transparent',
                                            border: `1px solid ${styles.fieldBorder}`,
                                            borderRadius: '4px',
                                            color: styles.inkSoft,
                                            fontFamily: 'Inter, sans-serif',
                                            fontSize: '0.9rem',
                                            outline: 'none',
                                            transition: 'border-color 120ms, box-shadow 120ms',
                                            minHeight: '44px'
                                        }}
                                        onFocus={(e) => {
                                            e.target.style.borderColor = styles.ink;
                                            e.target.style.boxShadow = `0 0 0 1px ${styles.ink}`;
                                        }}
                                        onBlur={(e) => {
                                            e.target.style.borderColor = styles.fieldBorder;
                                            e.target.style.boxShadow = 'none';
                                        }}
                                    />
                                    {errors.numberOfAdults && <p style={{ fontSize: '0.75rem', color: styles.error, marginTop: '4px' }}>{errors.numberOfAdults}</p>}
                                </div>
                                <div>
                                    <label style={{ 
                                        display: 'block', 
                                        fontFamily: 'Inter, sans-serif', 
                                        fontSize: '0.75rem', 
                                        color: styles.inkSoft,
                                        letterSpacing: '0.1em',
                                        textTransform: 'uppercase',
                                        marginBottom: '8px'
                                    }}>
                                        NUMBER OF CHILDREN (OPTIONAL)
                                    </label>
                                    <input
                                        type="number"
                                        min="0"
                                        value={formData.numberOfChildren}
                                        onChange={(e) => setFormData({...formData, numberOfChildren: parseInt(e.target.value) || 0})}
                                        style={{
                                            width: '100%',
                                            padding: '12px 16px',
                                            background: 'transparent',
                                            border: `1px solid ${styles.fieldBorder}`,
                                            borderRadius: '4px',
                                            color: styles.inkSoft,
                                            fontFamily: 'Inter, sans-serif',
                                            fontSize: '0.9rem',
                                            outline: 'none',
                                            transition: 'border-color 120ms, box-shadow 120ms',
                                            minHeight: '44px'
                                        }}
                                        onFocus={(e) => {
                                            e.target.style.borderColor = styles.ink;
                                            e.target.style.boxShadow = `0 0 0 1px ${styles.ink}`;
                                        }}
                                        onBlur={(e) => {
                                            e.target.style.borderColor = styles.fieldBorder;
                                            e.target.style.boxShadow = 'none';
                                        }}
                                    />
                                </div>
                            </div>
                        )}
                    </div>

                    {/* Guest Detail Cards */}
                    {attending === 'YES' && (formData.adults.length > 0 || formData.children.length > 0) && (
                        <div style={{ marginTop: '40px', marginBottom: '32px' }}>
                            <div style={{ 
                                display: 'grid',
                                gridTemplateColumns: window.innerWidth >= 768 && formData.children.length > 0 
                                    ? '1fr 1fr' 
                                    : '1fr',
                                gap: '20px'
                            }}>
                                {/* Adult Cards Column */}
                                {formData.adults.length > 0 && (
                                    <div style={{ display: 'flex', flexDirection: 'column', gap: '20px' }}>
                                        {formData.adults.map((adult, index) => (
                                <div key={`adult-${index}`} style={{ 
                                    border: `1px solid ${styles.fieldBorder}`, 
                                    borderRadius: '4px', 
                                    padding: '20px', 
                                    background: 'transparent' 
                                }}>
                                    <p style={{ 
                                        fontFamily: 'Inter, sans-serif', 
                                        fontSize: '0.75rem', 
                                        color: styles.inkSoft,
                                        letterSpacing: '0.15em',
                                        textTransform: 'uppercase',
                                        marginBottom: '16px'
                                    }}>
                                        ADULT {index === 0 ? 'ONE' : index === 1 ? 'TWO' : index + 1}
                                    </p>
                                    
                                    <div style={{ marginBottom: '12px' }}>
                                        <label style={{ 
                                            display: 'block', 
                                            fontFamily: 'Inter, sans-serif', 
                                            fontSize: '0.7rem', 
                                            color: styles.inkSoft,
                                            letterSpacing: '0.1em',
                                            textTransform: 'uppercase',
                                            marginBottom: '6px'
                                        }}>
                                            NAME
                                        </label>
                                        <input
                                            type="text"
                                            value={adult.name}
                                            onChange={(e) => updateAdult(index, 'name', e.target.value)}
                                            style={{
                                                width: '100%',
                                                padding: '10px 12px',
                                                background: 'transparent',
                                                border: `1px solid ${styles.fieldBorder}`,
                                                borderRadius: '4px',
                                                color: styles.inkSoft,
                                                fontFamily: 'Inter, sans-serif',
                                                fontSize: '0.85rem',
                                                outline: 'none',
                                                transition: 'border-color 120ms',
                                                minHeight: '40px'
                                            }}
                                            onFocus={(e) => e.target.style.borderColor = styles.ink}
                                            onBlur={(e) => e.target.style.borderColor = styles.fieldBorder}
                                        />
                                    </div>
                                    
                                    {mealOptions.length > 0 && (
                                        <div style={{ marginBottom: '12px' }}>
                                            <label style={{ 
                                                display: 'block', 
                                                fontFamily: 'Inter, sans-serif', 
                                                fontSize: '0.7rem', 
                                                color: styles.inkSoft,
                                                letterSpacing: '0.1em',
                                                textTransform: 'uppercase',
                                                marginBottom: '6px'
                                            }}>
                                                MEAL
                                            </label>
                                            <select
                                                value={adult.meal}
                                                onChange={(e) => updateAdult(index, 'meal', e.target.value)}
                                                style={{
                                                    width: '100%',
                                                    padding: '10px 12px',
                                                    background: 'transparent',
                                                    border: `1px solid ${styles.fieldBorder}`,
                                                    borderRadius: '4px',
                                                    color: styles.inkSoft,
                                                    fontFamily: 'Inter, sans-serif',
                                                    fontSize: '0.85rem',
                                                    outline: 'none',
                                                    transition: 'border-color 120ms',
                                                    minHeight: '40px',
                                                    cursor: 'pointer',
                                                    appearance: 'none',
                                                    backgroundImage: `url("data:image/svg+xml;charset=UTF-8,%3csvg xmlns='http://www.w3.org/2000/svg' viewBox='0 0 24 24' fill='none' stroke='%23C9B699' stroke-width='2' stroke-linecap='round' stroke-linejoin='round'%3e%3cpolyline points='6 9 12 15 18 9'%3e%3c/polyline%3e%3c/svg%3e")`,
                                                    backgroundRepeat: 'no-repeat',
                                                    backgroundPosition: 'right 12px center',
                                                    backgroundSize: '16px',
                                                    paddingRight: '40px'
                                                }}
                                                onFocus={(e) => e.target.style.borderColor = styles.ink}
                                                onBlur={(e) => e.target.style.borderColor = styles.fieldBorder}
                                            >
                                                <option value="" style={{ background: styles.bg, color: styles.inkSoft }}>Select meal</option>
                                                {mealOptions.map(meal => (
                                                    <option key={meal.code} value={meal.code} style={{ background: styles.bg, color: styles.inkSoft }}>{meal.name}</option>
                                                ))}
                                            </select>
                                        </div>
                                    )}
                                    
                                    <div>
                                        <label style={{ 
                                            display: 'block', 
                                            fontFamily: 'Inter, sans-serif', 
                                            fontSize: '0.7rem', 
                                            color: styles.inkSoft,
                                            letterSpacing: '0.1em',
                                            textTransform: 'uppercase',
                                            marginBottom: '6px'
                                        }}>
                                            DIETARY RESTRICTIONS (OPTIONAL)
                                        </label>
                                        <input
                                            type="text"
                                            value={adult.dietary}
                                            onChange={(e) => updateAdult(index, 'dietary', e.target.value)}
                                            style={{
                                                width: '100%',
                                                padding: '10px 12px',
                                                background: 'transparent',
                                                border: `1px solid ${styles.fieldBorder}`,
                                                borderRadius: '4px',
                                                color: styles.inkSoft,
                                                fontFamily: 'Inter, sans-serif',
                                                fontSize: '0.85rem',
                                                outline: 'none',
                                                transition: 'border-color 120ms',
                                                minHeight: '40px'
                                            }}
                                            onFocus={(e) => e.target.style.borderColor = styles.ink}
                                            onBlur={(e) => e.target.style.borderColor = styles.fieldBorder}
                                        />
                                    </div>
                                </div>
                                        ))}
                                    </div>
                                )}
                            
                                {/* Child Cards Column */}
                                {formData.children.length > 0 && (
                                    <div style={{ display: 'flex', flexDirection: 'column', gap: '20px' }}>
                                        {formData.children.map((child, index) => (
                                            <div key={`child-${index}`} style={{ 
                                                border: `1px solid ${styles.fieldBorder}`, 
                                                borderRadius: '4px', 
                                                padding: '20px', 
                                                background: 'transparent' 
                                            }}>
                                                <p style={{ 
                                        fontFamily: 'Inter, sans-serif', 
                                        fontSize: '0.75rem', 
                                        color: styles.inkSoft,
                                        letterSpacing: '0.15em',
                                        textTransform: 'uppercase',
                                        marginBottom: '16px'
                                    }}>
                                        CHILD {index === 0 ? 'ONE' : index === 1 ? 'TWO' : index + 1}
                                    </p>
                                    
                                    <div style={{ marginBottom: '12px' }}>
                                        <label style={{ 
                                            display: 'block', 
                                            fontFamily: 'Inter, sans-serif', 
                                            fontSize: '0.7rem', 
                                            color: styles.inkSoft,
                                            letterSpacing: '0.1em',
                                            textTransform: 'uppercase',
                                            marginBottom: '6px'
                                        }}>
                                            NAME
                                        </label>
                                        <input
                                            type="text"
                                            value={child.name}
                                            onChange={(e) => updateChild(index, 'name', e.target.value)}
                                            style={{
                                                width: '100%',
                                                padding: '10px 12px',
                                                background: 'transparent',
                                                border: `1px solid ${styles.fieldBorder}`,
                                                borderRadius: '4px',
                                                color: styles.inkSoft,
                                                fontFamily: 'Inter, sans-serif',
                                                fontSize: '0.85rem',
                                                outline: 'none',
                                                transition: 'border-color 120ms',
                                                minHeight: '40px'
                                            }}
                                            onFocus={(e) => e.target.style.borderColor = styles.ink}
                                            onBlur={(e) => e.target.style.borderColor = styles.fieldBorder}
                                        />
                                    </div>
                                    
                                    {mealOptions.length > 0 && (
                                        <div style={{ marginBottom: '12px' }}>
                                            <label style={{ 
                                                display: 'block', 
                                                fontFamily: 'Inter, sans-serif', 
                                                fontSize: '0.7rem', 
                                                color: styles.inkSoft,
                                                letterSpacing: '0.1em',
                                                textTransform: 'uppercase',
                                                marginBottom: '6px'
                                            }}>
                                                MEAL
                                            </label>
                                            <select
                                                value={child.meal}
                                                onChange={(e) => updateChild(index, 'meal', e.target.value)}
                                                style={{
                                                    width: '100%',
                                                    padding: '10px 12px',
                                                    background: 'transparent',
                                                    border: `1px solid ${styles.fieldBorder}`,
                                                    borderRadius: '4px',
                                                    color: styles.inkSoft,
                                                    fontFamily: 'Inter, sans-serif',
                                                    fontSize: '0.85rem',
                                                    outline: 'none',
                                                    transition: 'border-color 120ms',
                                                    minHeight: '40px',
                                                    cursor: 'pointer',
                                                    appearance: 'none',
                                                    backgroundImage: `url("data:image/svg+xml;charset=UTF-8,%3csvg xmlns='http://www.w3.org/2000/svg' viewBox='0 0 24 24' fill='none' stroke='%23C9B699' stroke-width='2' stroke-linecap='round' stroke-linejoin='round'%3e%3cpolyline points='6 9 12 15 18 9'%3e%3c/polyline%3e%3c/svg%3e")`,
                                                    backgroundRepeat: 'no-repeat',
                                                    backgroundPosition: 'right 12px center',
                                                    backgroundSize: '16px',
                                                    paddingRight: '40px'
                                                }}
                                                onFocus={(e) => e.target.style.borderColor = styles.ink}
                                                onBlur={(e) => e.target.style.borderColor = styles.fieldBorder}
                                            >
                                                <option value="" style={{ background: styles.bg, color: styles.inkSoft }}>Select meal</option>
                                                {mealOptions.map(meal => (
                                                    <option key={meal.code} value={meal.code} style={{ background: styles.bg, color: styles.inkSoft }}>{meal.name}</option>
                                                ))}
                                            </select>
                                        </div>
                                    )}
                                    
                                    <div>
                                        <label style={{ 
                                            display: 'block', 
                                            fontFamily: 'Inter, sans-serif', 
                                            fontSize: '0.7rem', 
                                            color: styles.inkSoft,
                                            letterSpacing: '0.1em',
                                            textTransform: 'uppercase',
                                            marginBottom: '6px'
                                        }}>
                                            DIETARY RESTRICTIONS (OPTIONAL)
                                        </label>
                                        <input
                                            type="text"
                                            value={child.dietary}
                                            onChange={(e) => updateChild(index, 'dietary', e.target.value)}
                                            style={{
                                                width: '100%',
                                                padding: '10px 12px',
                                                background: 'transparent',
                                                border: `1px solid ${styles.fieldBorder}`,
                                                borderRadius: '4px',
                                                color: styles.inkSoft,
                                                fontFamily: 'Inter, sans-serif',
                                                fontSize: '0.85rem',
                                                outline: 'none',
                                                transition: 'border-color 120ms',
                                                minHeight: '40px'
                                            }}
                                            onFocus={(e) => e.target.style.borderColor = styles.ink}
                                            onBlur={(e) => e.target.style.borderColor = styles.fieldBorder}
                                        />
                                                </div>
                                            </div>
                                        ))}
                                    </div>
                                )}
                            </div>
                        </div>
                    )}

                    {/* Message */}
                    <div style={{ marginBottom: '36px' }}>
                        <label style={{ 
                            display: 'block', 
                            fontFamily: 'Inter, sans-serif', 
                            fontSize: '0.75rem', 
                            color: styles.inkSoft,
                            letterSpacing: '0.1em',
                            textTransform: 'uppercase',
                            marginBottom: '8px'
                        }}>
                            MESSAGE (OPTIONAL)
                        </label>
                        <textarea
                            value={formData.message}
                            onChange={(e) => setFormData({...formData, message: e.target.value})}
                            rows="6"
                            style={{
                                width: '100%',
                                padding: '12px 16px',
                                background: 'transparent',
                                border: `1px solid ${styles.fieldBorder}`,
                                borderRadius: '4px',
                                color: styles.inkSoft,
                                fontFamily: 'Inter, sans-serif',
                                fontSize: '0.9rem',
                                outline: 'none',
                                transition: 'border-color 120ms, box-shadow 120ms',
                                minHeight: '180px',
                                resize: 'vertical'
                            }}
                            onFocus={(e) => {
                                e.target.style.borderColor = styles.ink;
                                e.target.style.boxShadow = `0 0 0 1px ${styles.ink}`;
                            }}
                            onBlur={(e) => {
                                e.target.style.borderColor = styles.fieldBorder;
                                e.target.style.boxShadow = 'none';
                            }}
                        />
                    </div>

                    {/* Submit Button */}
                    <div style={{ textAlign: 'center', marginTop: '32px' }}>
                        <button
                            type="submit"
                            disabled={Object.keys(errors).length > 0}
                            style={{
                                fontFamily: 'Inter, sans-serif',
                                fontSize: '0.875rem',
                                color: styles.bg,
                                background: styles.accent,
                                border: 'none',
                                borderRadius: '9999px',
                                padding: '14px 48px',
                                letterSpacing: '0.1em',
                                textTransform: 'uppercase',
                                cursor: 'pointer',
                                transition: 'background-color 150ms',
                                fontWeight: '500',
                                minHeight: '44px',
                                opacity: Object.keys(errors).length > 0 ? 0.5 : 1
                            }}
                            onMouseEnter={(e) => e.target.style.background = styles.accentHover}
                            onMouseLeave={(e) => e.target.style.background = styles.accent}
                        >
                            SUBMIT
                        </button>
                        
                        {errors.submit && (
                            <p style={{ 
                                fontSize: '0.875rem', 
                                color: styles.error, 
                                marginTop: '16px',
                                fontFamily: 'Inter, sans-serif'
                            }}>
                                {errors.submit}
                            </p>
                        )}
                    </div>
                </form>
            </div>
        </div>
    );
}

// Export for mounting
export function mount(rootElement, eventSlug = '') {
    const root = createRoot(rootElement);
    root.render(<PublicRsvpApp eventSlug={eventSlug || null} />);
}

// Make available globally
window.PublicRsvpApp = { mount };

export default PublicRsvpApp;
