import React, { useState } from 'react';

export default function LuxuryRsvpForm({ events, mode, onSubmit, styles, household, settings }) {
    // Parse household display name into first/last name
    const getNames = () => {
        if (!household?.displayName) return { firstName: '', lastName: '' };
        const parts = household.displayName.split(' ');
        return {
            firstName: parts[0] || '',
            lastName: parts.slice(1).join(' ') || ''
        };
    };
    
    const { firstName: defaultFirstName, lastName: defaultLastName } = getNames();

    const [formData, setFormData] = useState({
        attending: null, // YES or NO
        firstName: defaultFirstName,
        lastName: defaultLastName,
        email: household?.email || '',
        phone: household?.phone || '',
        numberOfAdults: household?.numberOfAdults || '',
        numberOfChildren: household?.numberOfChildren || '',
        adults: [],
        children: [],
        message: '',
        events: events.map(event => ({
            eventId: event.id,
            status: null,
            adults: 0,
            children: 0,
            dietary: ''
        }))
    });

    // Update adults/children arrays when counts change
    React.useEffect(() => {
        const adultsCount = parseInt(formData.numberOfAdults) || 0;
        const childrenCount = parseInt(formData.numberOfChildren) || 0;
        
        setFormData(prev => ({
            ...prev,
            adults: Array.from({ length: adultsCount }, (_, i) => prev.adults[i] || { name: '', meal: '', dietary: '' }),
            children: Array.from({ length: childrenCount }, (_, i) => prev.children[i] || { name: '', meal: '', dietary: '' })
        }));
    }, [formData.numberOfAdults, formData.numberOfChildren]);

    const [errors, setErrors] = useState({});
    const [touched, setTouched] = useState({});

    const validate = () => {
        const newErrors = {};
        
        if (!formData.attending) {
            newErrors.attending = 'Please select whether you are attending';
        }
        
        if (!formData.firstName.trim()) {
            newErrors.firstName = 'First name is required';
        }
        
        if (!formData.lastName.trim()) {
            newErrors.lastName = 'Last name is required';
        }
        
        if (!formData.email.trim()) {
            newErrors.email = 'Email is required';
        } else if (!/^[^\s@]+@[^\s@]+\.[^\s@]+$/.test(formData.email)) {
            newErrors.email = 'Please enter a valid email address';
        }
        
        if (formData.attending === 'YES') {
            const adults = parseInt(formData.numberOfAdults) || 0;
            if (adults < 1) {
                newErrors.numberOfAdults = 'Please add at least one adult guest';
            }
        }
        
        setErrors(newErrors);
        return Object.keys(newErrors).length === 0;
    };

    const handleBlur = (field) => {
        setTouched({ ...touched, [field]: true });
    };

    const updateAdult = (index, field, value) => {
        setFormData(prev => ({
            ...prev,
            adults: prev.adults.map((adult, i) => i === index ? { ...adult, [field]: value } : adult)
        }));
    };

    const updateChild = (index, field, value) => {
        setFormData(prev => ({
            ...prev,
            children: prev.children.map((child, i) => i === index ? { ...child, [field]: value } : child)
        }));
    };

    const handleSubmit = async (e) => {
        e.preventDefault();
        
        // Mark all fields as touched
        const allFields = ['attending', 'firstName', 'lastName', 'email', 'numberOfAdults'];
        const newTouched = {};
        allFields.forEach(field => newTouched[field] = true);
        setTouched(newTouched);
        
        if (!validate()) {
            return;
        }

        // Build submission data
        const submission = {
            firstName: formData.firstName,
            lastName: formData.lastName,
            email: formData.email,
            phone: formData.phone,
            events: events.map(event => ({
                eventId: event.id,
                status: formData.attending === 'YES' ? 'YES' : 'NO',
                adults: formData.attending === 'YES' ? (parseInt(formData.numberOfAdults) || 0) : 0,
                children: formData.attending === 'YES' ? (parseInt(formData.numberOfChildren) || 0) : 0,
                dietary: formData.message
            })),
            message: formData.message,
            adults: formData.adults,
            children: formData.children
        };

        onSubmit(submission);
    };

    const inputStyle = {
        width: '100%',
        background: 'transparent',
        border: `1px solid ${styles.fieldBorder}`,
        borderRadius: '4px',
        padding: '12px 16px',
        fontFamily: 'Inter, sans-serif',
        fontSize: '0.9375rem',
        color: styles.inkSoft,
        transition: 'all 120ms ease',
        outline: 'none'
    };

    const focusStyle = {
        borderColor: styles.ink,
        boxShadow: `0 0 1px ${styles.ink}`
    };

    const labelStyle = {
        display: 'block',
        fontFamily: 'Inter, sans-serif',
        fontSize: '0.75rem',
        color: styles.inkSoft,
        letterSpacing: '0.1em',
        textTransform: 'uppercase',
        marginBottom: '8px',
        fontWeight: '500'
    };

    const errorStyle = {
        fontFamily: 'Inter, sans-serif',
        fontSize: '0.75rem',
        color: styles.error,
        marginTop: '6px'
    };

    return (
        <form onSubmit={handleSubmit} style={{ fontFamily: 'Inter, sans-serif' }}>
            {/* Attendance Choice */}
            <div style={{ marginBottom: '40px' }}>
                <div style={{ textAlign: 'center', marginBottom: '24px' }}>
                    <label style={{
                        fontFamily: 'Inter, sans-serif',
                        fontSize: '0.75rem',
                        color: styles.inkSoft,
                        letterSpacing: '0.15em',
                        textTransform: 'uppercase',
                        fontWeight: '600'
                    }}>
                        ARE YOU ATTENDING
                    </label>
                </div>
                
                <div style={{ display: 'flex', justifyContent: 'center', gap: '64px' }}>
                    {['YES', 'NO'].map(choice => (
                        <label key={choice} style={{ display: 'flex', alignItems: 'center', gap: '12px', cursor: 'pointer' }}>
                            <input
                                type="radio"
                                name="attending"
                                value={choice}
                                checked={formData.attending === choice}
                                onChange={(e) => setFormData({ ...formData, attending: e.target.value })}
                                onBlur={() => handleBlur('attending')}
                                style={{ display: 'none' }}
                            />
                            <div style={{
                                width: '20px',
                                height: '20px',
                                borderRadius: '50%',
                                border: `2px solid ${formData.attending === choice ? styles.ink : styles.inkSoft}`,
                                display: 'flex',
                                alignItems: 'center',
                                justifyContent: 'center',
                                transition: 'all 120ms ease'
                            }}>
                                {formData.attending === choice && (
                                    <div style={{
                                        width: '10px',
                                        height: '10px',
                                        borderRadius: '50%',
                                        background: styles.ink
                                    }} />
                                )}
                            </div>
                            <span style={{
                                fontFamily: 'Inter, sans-serif',
                                fontSize: '0.9375rem',
                                color: formData.attending === choice ? styles.ink : styles.inkSoft,
                                letterSpacing: '0.1em',
                                transition: 'color 120ms ease'
                            }}>
                                {choice}
                            </span>
                        </label>
                    ))}
                </div>
                
                {touched.attending && errors.attending && (
                    <p style={{ ...errorStyle, textAlign: 'center' }}>{errors.attending}</p>
                )}
            </div>

            {/* Divider */}
            <div style={{ 
                display: 'flex', 
                alignItems: 'center', 
                gap: '16px', 
                marginBottom: '32px' 
            }}>
                <div style={{ flex: 1, height: '1px', background: styles.fieldBorder }} />
                <span style={{
                    fontFamily: 'Inter, sans-serif',
                    fontSize: '0.75rem',
                    color: styles.inkSoft,
                    letterSpacing: '0.15em',
                    textTransform: 'uppercase'
                }}>
                    YOUR DETAILS
                </span>
                <div style={{ flex: 1, height: '1px', background: styles.fieldBorder }} />
            </div>

            {/* Two Column Layout */}
            <div style={{ 
                display: 'grid', 
                gridTemplateColumns: window.innerWidth >= 768 ? '1fr 1fr' : '1fr',
                gap: '20px',
                marginBottom: '20px'
            }}>
                {/* First Name */}
                <div>
                    <label style={labelStyle}>
                        FIRST NAME *
                    </label>
                    <input
                        type="text"
                        value={formData.firstName}
                        onChange={(e) => setFormData({ ...formData, firstName: e.target.value })}
                        onBlur={() => handleBlur('firstName')}
                        onFocus={(e) => Object.assign(e.target.style, focusStyle)}
                        style={inputStyle}
                    />
                    {touched.firstName && errors.firstName && (
                        <p style={errorStyle}>{errors.firstName}</p>
                    )}
                </div>

                {/* Last Name */}
                <div>
                    <label style={labelStyle}>
                        LAST NAME *
                    </label>
                    <input
                        type="text"
                        value={formData.lastName}
                        onChange={(e) => setFormData({ ...formData, lastName: e.target.value })}
                        onBlur={() => handleBlur('lastName')}
                        onFocus={(e) => Object.assign(e.target.style, focusStyle)}
                        style={inputStyle}
                    />
                    {touched.lastName && errors.lastName && (
                        <p style={errorStyle}>{errors.lastName}</p>
                    )}
                </div>

                {/* Email */}
                <div>
                    <label style={labelStyle}>
                        EMAIL *
                    </label>
                    <input
                        type="email"
                        value={formData.email}
                        onChange={(e) => setFormData({ ...formData, email: e.target.value })}
                        onBlur={() => handleBlur('email')}
                        onFocus={(e) => Object.assign(e.target.style, focusStyle)}
                        style={inputStyle}
                    />
                    {touched.email && errors.email && (
                        <p style={errorStyle}>{errors.email}</p>
                    )}
                </div>

                {/* Phone */}
                <div>
                    <label style={labelStyle}>
                        PHONE (OPTIONAL)
                    </label>
                    <input
                        type="tel"
                        value={formData.phone}
                        onChange={(e) => setFormData({ ...formData, phone: e.target.value })}
                        onFocus={(e) => Object.assign(e.target.style, focusStyle)}
                        style={inputStyle}
                    />
                </div>

                {/* Number of Adults */}
                {formData.attending === 'YES' && (
                    <div>
                        <label style={labelStyle}>
                            NUMBER OF ADULTS *
                        </label>
                        <input
                            type="number"
                            min="0"
                            value={formData.numberOfAdults}
                            onChange={(e) => setFormData({ ...formData, numberOfAdults: e.target.value })}
                            onBlur={() => handleBlur('numberOfAdults')}
                            onFocus={(e) => Object.assign(e.target.style, focusStyle)}
                            style={inputStyle}
                        />
                        {touched.numberOfAdults && errors.numberOfAdults && (
                            <p style={errorStyle}>{errors.numberOfAdults}</p>
                        )}
                    </div>
                )}

                {/* Number of Children */}
                {formData.attending === 'YES' && (
                    <div>
                        <label style={labelStyle}>
                            NUMBER OF CHILDREN (OPTIONAL)
                        </label>
                        <input
                            type="number"
                            min="0"
                            value={formData.numberOfChildren}
                            onChange={(e) => setFormData({ ...formData, numberOfChildren: e.target.value })}
                            onFocus={(e) => Object.assign(e.target.style, focusStyle)}
                            style={inputStyle}
                        />
                    </div>
                )}
            </div>

            {/* Guest Detail Cards */}
            {formData.attending === 'YES' && (parseInt(formData.numberOfAdults) > 0 || parseInt(formData.numberOfChildren) > 0) && (
                <div style={{ marginTop: '40px', marginBottom: '32px' }}>
                    <div style={{ 
                        display: 'grid',
                        gridTemplateColumns: window.innerWidth >= 768 && parseInt(formData.numberOfChildren) > 0 
                            ? '1fr 1fr' 
                            : '1fr',
                        gap: '20px'
                    }}>
                        {/* Adult Cards */}
                        {parseInt(formData.numberOfAdults) > 0 && (
                            <div style={{ display: 'flex', flexDirection: 'column', gap: '20px' }}>
                                {Array.from({ length: parseInt(formData.numberOfAdults) }, (_, i) => (
                                    <div 
                                        key={`adult-${i}`}
                                        style={{
                                            border: `1px solid ${styles.fieldBorder}`,
                                            borderRadius: '4px',
                                            padding: '24px',
                                            background: 'transparent'
                                        }}
                                    >
                                        <h3 style={{
                                            fontFamily: 'Inter, sans-serif',
                                            fontSize: '0.75rem',
                                            color: styles.inkSoft,
                                            letterSpacing: '0.1em',
                                            textTransform: 'uppercase',
                                            marginBottom: '20px',
                                            fontWeight: '600'
                                        }}>
                                            ADULT {i === 0 ? 'ONE' : i === 1 ? 'TWO' : i === 2 ? 'THREE' : i === 3 ? 'FOUR' : `${i + 1}`}
                                        </h3>
                                        
                                        <div style={{ marginBottom: '16px' }}>
                                            <label style={labelStyle}>NAME</label>
                                            <input
                                                type="text"
                                                value={formData.adults[i]?.name || ''}
                                                onChange={(e) => updateAdult(i, 'name', e.target.value)}
                                                onFocus={(e) => Object.assign(e.target.style, focusStyle)}
                                                style={inputStyle}
                                                placeholder="Full name"
                                            />
                                        </div>

                                        {events.some(e => e.mealsEnabled) && (
                                            <div style={{ marginBottom: '16px' }}>
                                                <label style={labelStyle}>MEAL</label>
                                                <select
                                                    value={formData.adults[i]?.meal || ''}
                                                    onChange={(e) => updateAdult(i, 'meal', e.target.value)}
                                                    onFocus={(e) => Object.assign(e.target.style, focusStyle)}
                                                    style={{
                                                        ...inputStyle,
                                                        cursor: 'pointer',
                                                        appearance: 'none',
                                                        backgroundImage: `url("data:image/svg+xml;charset=UTF-8,%3csvg xmlns='http://www.w3.org/2000/svg' viewBox='0 0 24 24' fill='none' stroke='%23C9B699' stroke-width='2' stroke-linecap='round' stroke-linejoin='round'%3e%3cpolyline points='6 9 12 15 18 9'%3e%3c/polyline%3e%3c/svg%3e")`,
                                                        backgroundRepeat: 'no-repeat',
                                                        backgroundPosition: 'right 12px center',
                                                        backgroundSize: '16px',
                                                        paddingRight: '40px'
                                                    }}
                                                >
                                                    <option value="" style={{ background: styles.bg, color: styles.inkSoft }}>Select meal preference</option>
                                                    {events[0]?.mealOptions?.map(meal => (
                                                        <option key={meal.code} value={meal.code} style={{ background: styles.bg, color: styles.inkSoft }}>
                                                            {meal.name}
                                                        </option>
                                                    ))}
                                                </select>
                                            </div>
                                        )}

                                        <div>
                                            <label style={labelStyle}>DIETARY RESTRICTIONS (OPTIONAL)</label>
                                            <input
                                                type="text"
                                                value={formData.adults[i]?.dietary || ''}
                                                onChange={(e) => updateAdult(i, 'dietary', e.target.value)}
                                                onFocus={(e) => Object.assign(e.target.style, focusStyle)}
                                                style={inputStyle}
                                                placeholder="Allergies, preferences..."
                                            />
                                        </div>
                                    </div>
                                ))}
                            </div>
                        )}

                        {/* Child Cards */}
                        {parseInt(formData.numberOfChildren) > 0 && (
                            <div style={{ display: 'flex', flexDirection: 'column', gap: '20px' }}>
                                {Array.from({ length: parseInt(formData.numberOfChildren) }, (_, i) => (
                                    <div 
                                        key={`child-${i}`}
                                        style={{
                                            border: `1px solid ${styles.fieldBorder}`,
                                            borderRadius: '4px',
                                            padding: '24px',
                                            background: 'transparent'
                                        }}
                                    >
                                        <h3 style={{
                                            fontFamily: 'Inter, sans-serif',
                                            fontSize: '0.75rem',
                                            color: styles.inkSoft,
                                            letterSpacing: '0.1em',
                                            textTransform: 'uppercase',
                                            marginBottom: '20px',
                                            fontWeight: '600'
                                        }}>
                                            CHILD {i === 0 ? 'ONE' : i === 1 ? 'TWO' : i === 2 ? 'THREE' : i === 3 ? 'FOUR' : `${i + 1}`}
                                        </h3>
                                        
                                        <div style={{ marginBottom: '16px' }}>
                                            <label style={labelStyle}>NAME</label>
                                            <input
                                                type="text"
                                                value={formData.children[i]?.name || ''}
                                                onChange={(e) => updateChild(i, 'name', e.target.value)}
                                                onFocus={(e) => Object.assign(e.target.style, focusStyle)}
                                                style={inputStyle}
                                                placeholder="Full name"
                                            />
                                        </div>

                                        {events.some(e => e.mealsEnabled) && (
                                            <div style={{ marginBottom: '16px' }}>
                                                <label style={labelStyle}>MEAL</label>
                                                <select
                                                    value={formData.children[i]?.meal || ''}
                                                    onChange={(e) => updateChild(i, 'meal', e.target.value)}
                                                    onFocus={(e) => Object.assign(e.target.style, focusStyle)}
                                                    style={{
                                                        ...inputStyle,
                                                        cursor: 'pointer',
                                                        appearance: 'none',
                                                        backgroundImage: `url("data:image/svg+xml;charset=UTF-8,%3csvg xmlns='http://www.w3.org/2000/svg' viewBox='0 0 24 24' fill='none' stroke='%23C9B699' stroke-width='2' stroke-linecap='round' stroke-linejoin='round'%3e%3cpolyline points='6 9 12 15 18 9'%3e%3c/polyline%3e%3c/svg%3e")`,
                                                        backgroundRepeat: 'no-repeat',
                                                        backgroundPosition: 'right 12px center',
                                                        backgroundSize: '16px',
                                                        paddingRight: '40px'
                                                    }}
                                                >
                                                    <option value="" style={{ background: styles.bg, color: styles.inkSoft }}>Select meal preference</option>
                                                    {events[0]?.mealOptions?.map(meal => (
                                                        <option key={meal.code} value={meal.code} style={{ background: styles.bg, color: styles.inkSoft }}>
                                                            {meal.name}
                                                        </option>
                                                    ))}
                                                </select>
                                            </div>
                                        )}

                                        <div>
                                            <label style={labelStyle}>DIETARY RESTRICTIONS (OPTIONAL)</label>
                                            <input
                                                type="text"
                                                value={formData.children[i]?.dietary || ''}
                                                onChange={(e) => updateChild(i, 'dietary', e.target.value)}
                                                onFocus={(e) => Object.assign(e.target.style, focusStyle)}
                                                style={inputStyle}
                                                placeholder="Allergies, preferences..."
                                            />
                                        </div>
                                    </div>
                                ))}
                            </div>
                        )}
                    </div>
                </div>
            )}

            {/* Message */}
            <div style={{ marginTop: '32px', marginBottom: '36px' }}>
                <label style={labelStyle}>
                    MESSAGE (OPTIONAL)
                </label>
                <textarea
                    value={formData.message}
                    onChange={(e) => setFormData({ ...formData, message: e.target.value })}
                    onFocus={(e) => Object.assign(e.target.style, focusStyle)}
                    style={{
                        ...inputStyle,
                        minHeight: '200px',
                        resize: 'vertical',
                        fontFamily: 'Inter, sans-serif'
                    }}
                    placeholder="Share a message with the couple..."
                />
            </div>

            {/* Submit Button */}
            <div style={{ textAlign: 'center' }}>
                <button
                    type="submit"
                    style={{
                        background: styles.accent,
                        color: styles.bg,
                        border: 'none',
                        borderRadius: '9999px',
                        padding: '14px 48px',
                        fontFamily: 'Inter, sans-serif',
                        fontSize: '0.875rem',
                        fontWeight: '600',
                        letterSpacing: '0.15em',
                        textTransform: 'uppercase',
                        cursor: 'pointer',
                        transition: 'all 150ms ease',
                        minHeight: '44px'
                    }}
                    onMouseEnter={(e) => e.target.style.background = styles.accentHover}
                    onMouseLeave={(e) => e.target.style.background = styles.accent}
                >
                    SUBMIT
                </button>
            </div>
        </form>
    );
}
