import React, { useState, useEffect } from 'react';
import LuxuryRsvpForm from './LuxuryRsvpForm';
import SuccessView from './SuccessView';

// Content configuration
const CONTENT = {
    preheader: 'LOOKING FORWARD TO CELEBRATING WITH YOU',
    rsvpTitle: 'R  S  V  P',
};

export default function InvitePage({ token }) {
    // Default styles (fallback if theme not loaded)
    const [styles, setStyles] = useState({
        bg: '#2A0713',
        ink: '#C9B699',
        inkSoft: 'rgba(201,182,153,0.8)',
        fieldBorder: '#7B5F51',
        accent: '#B6A37F',
        accentHover: '#C8B588',
        error: '#E8CBB5'
    });
    const [themeLoaded, setThemeLoaded] = useState(false);
    const [data, setData] = useState(null);
    const [loading, setLoading] = useState(true);
    const [submitted, setSubmitted] = useState(false);
    const [error, setError] = useState(null);
    
    // Get event ID from URL if specified
    const urlParams = new URLSearchParams(window.location.search);
    const specificEventId = urlParams.get('event');

    useEffect(() => {
        // Load theme first
        fetch('/api/rsvp/public/theme')
            .then(res => res.json())
            .then(theme => {
                setStyles(theme);
                setThemeLoaded(true);
            })
            .catch(() => {
                setThemeLoaded(true); // Use default if theme fails
            });
        
        // Load invite data with optional event filter
        const url = specificEventId 
            ? `/api/rsvp/${token}?event=${specificEventId}`
            : `/api/rsvp/${token}`;
            
        fetch(url)
            .then(res => {
                if (!res.ok) {
                    return res.json().then(data => {
                        throw new Error(data.error || `HTTP error! status: ${res.status}`);
                    });
                }
                return res.json();
            })
            .then(data => {
                if (data.error) {
                    setError(data.error);
                } else {
                    setData(data);
                }
                setLoading(false);
            })
            .catch(err => {
                console.error('Failed to load invitation:', err);
                setError(err.message || 'Failed to load invitation');
                setLoading(false);
            });
    }, [token, specificEventId]);

    const handleSubmit = async (formData) => {
        try {
            const res = await fetch(`/api/rsvp/${token}`, {
                method: 'POST',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify(formData)
            });

            const result = await res.json();
            
            if (result.error) {
                alert(result.error);
            } else {
                // Reload invitation data to get updated RSVP details
                const refreshUrl = specificEventId 
                    ? `/api/rsvp/${token}?event=${specificEventId}`
                    : `/api/rsvp/${token}`;
                const refreshRes = await fetch(refreshUrl);
                const refreshedData = await refreshRes.json();
                if (!refreshedData.error) {
                    setData(refreshedData);
                }
                setSubmitted(true);
            }
        } catch (err) {
            alert('Failed to submit RSVP. Please try again.');
        }
    };

    // Show neutral loading screen while theme loads
    if (!themeLoaded) {
        return (
            <div className="min-h-screen flex items-center justify-center" style={{ background: '#000' }}>
                <div style={{ color: '#fff', fontFamily: 'Inter, sans-serif', letterSpacing: '0.2em', fontSize: '0.875rem' }} className="uppercase">Loading...</div>
            </div>
        );
    }

    if (loading) {
        return (
            <div style={{ background: styles.bg }} className="min-h-screen flex items-center justify-center">
                <div style={{ color: styles.ink, fontFamily: 'Inter, sans-serif', letterSpacing: '0.2em' }} className="text-sm uppercase">Loading...</div>
            </div>
        );
    }

    if (error) {
        return (
            <div style={{ background: styles.bg }} className="min-h-screen flex items-center justify-center p-6">
                <div className="text-center" style={{ maxWidth: '560px' }}>
                    <h2 style={{ 
                        fontFamily: "'Playfair Display', serif", 
                        fontSize: 'clamp(1.5rem, 4vw, 2.5rem)', 
                        color: styles.ink,
                        letterSpacing: '0.15em',
                        marginBottom: '24px'
                    }}>
                        OOPS!
                    </h2>
                    <p style={{ 
                        fontFamily: 'Inter, sans-serif', 
                        color: styles.error, 
                        fontSize: '1rem',
                        lineHeight: '1.6'
                    }}>
                        {error}
                    </p>
                </div>
            </div>
        );
    }

    if (submitted) {
        return <SuccessView data={data} styles={styles} />;
    }

    const { mode, household, settings, events } = data;

    return (
        <div className="min-h-screen py-16 px-5" style={{ background: styles.bg }}>
            <div className="max-w-[880px] mx-auto">
                {/* Header */}
                <div className="text-center mb-10">
                    <p className="font-sans text-xs tracking-wider uppercase mb-5" style={{ color: styles.inkSoft }}>
                        {CONTENT.preheader}
                    </p>
                    
                    <h1 className="font-serif leading-none mb-6 mt-[34px] font-bold uppercase tracking-[0.3em]" 
                        style={{ 
                            fontFamily: "'Bodoni Moda', serif",
                            fontSize: 'clamp(3rem, 10vw, 6rem)',
                            color: styles.ink
                        }}>
                        {CONTENT.rsvpTitle}
                    </h1>
                    
                    {events[0]?.nameOne && events[0]?.nameTwo ? (
                        <div className="grid grid-cols-1 md:grid-cols-[1fr_auto_1fr] items-center mb-3 gap-2 md:gap-6">
                            <div className="text-center md:text-right">
                                <span className="text-[30px] tracking-wider uppercase" 
                                    style={{ 
                                        fontFamily: `'${events[0]?.nameFont || 'Bodoni Moda'}', serif`,
                                        color: styles.ink
                                    }}>
                                    {events[0].nameOne.toUpperCase()}
                                </span>
                            </div>
                            <span className="font-sans text-[20px] tracking-wide whitespace-nowrap text-center" style={{ color: styles.ink }}>
                                {events[0]?.andText || 'AND'}
                            </span>
                            <div className="text-center md:text-left">
                                <span className="text-[30px] tracking-wider uppercase" 
                                    style={{ 
                                        fontFamily: `'${events[0]?.nameFont || 'Bodoni Moda'}', serif`,
                                        color: styles.ink
                                    }}>
                                    {events[0].nameTwo.toUpperCase()}
                                </span>
                            </div>
                        </div>
                    ) : (
                        <p className="text-2xl tracking-wider uppercase mb-3" 
                            style={{ 
                                fontFamily: `'${events[0]?.nameFont || 'Bodoni Moda'}', serif`,
                                color: styles.ink
                            }}>
                            {events[0]?.name?.toUpperCase() || 'WEDDING CEREMONY'}
                        </p>
                    )}
                    
                    {events[0]?.startAt && (
                        <p className="font-sans text-xs tracking-wider uppercase mb-2" style={{ color: styles.inkSoft }}>
                            {new Date(events[0].startAt.replace(' ', 'T')).toLocaleDateString('en-US', {
                                weekday: 'long',
                                year: 'numeric',
                                month: 'long',
                                day: 'numeric'
                            }).toUpperCase()}
                        </p>
                    )}
                    
                    {events[0] && (
                        <p className="font-sans text-xs tracking-wider uppercase text-center" style={{ color: styles.inkSoft }}>
                            {events[0].venueName ? `VENUE: ${events[0].venueName.toUpperCase()}` : ''}{events[0].venueName && events[0].startAt ? '. ' : ''}TIME: {events[0].startAt ? new Date(events[0].startAt.replace(' ', 'T')).toLocaleTimeString('en-US', { hour: 'numeric', minute: '2-digit' }).toUpperCase() : ''}{events[0].endAt ? ` - ${new Date(events[0].endAt.replace(' ', 'T')).toLocaleTimeString('en-US', { hour: 'numeric', minute: '2-digit' }).toUpperCase()}` : ''}
                        </p>
                    )}
                    {events[0]?.description && (
                        <p className="font-sans text-sm text-center mt-4" style={{ color: styles.inkSoft, maxWidth: '600px', margin: '16px auto 0' }}>
                            {events[0].description}
                        </p>
                    )}
                </div>

                {/* Welcome Message */}
                <div style={{ 
                    textAlign: 'center', 
                    marginBottom: '40px',
                    padding: '24px',
                    border: `1px solid ${styles.fieldBorder}`,
                    borderRadius: '4px'
                }}>
                    <p style={{ 
                        fontFamily: "'Playfair Display', serif", 
                        fontSize: '1.5rem', 
                        color: styles.ink,
                        letterSpacing: '0.1em'
                    }}>
                        Welcome, {household.displayName}!
                    </p>
                </div>
                
                {settings.rsvpDeadline && new Date() > new Date(settings.rsvpDeadline) ? (
                    <div style={{
                        background: 'rgba(232,203,181,0.1)',
                        border: `1px solid ${styles.error}`,
                        borderRadius: '4px',
                        padding: '24px',
                        textAlign: 'center'
                    }}>
                        <p style={{ 
                            fontFamily: 'Inter, sans-serif',
                            fontSize: '0.875rem',
                            color: styles.error,
                            letterSpacing: '0.1em',
                            textTransform: 'uppercase',
                            marginBottom: '8px',
                            fontWeight: '600'
                        }}>
                            RSVP DEADLINE PASSED
                        </p>
                        <p style={{ 
                            fontFamily: 'Inter, sans-serif',
                            color: styles.inkSoft,
                            fontSize: '0.9rem'
                        }}>
                            The RSVP deadline has passed. Please contact us directly if you need to make changes.
                        </p>
                    </div>
                ) : (
                    <LuxuryRsvpForm 
                        events={events}
                        mode={mode}
                        onSubmit={handleSubmit}
                        styles={styles}
                        household={household}
                        settings={settings}
                    />
                )}
            </div>
        </div>
    );
}
