import React, { useState, useEffect } from 'react';
import { Button } from '@/components/ui/button';
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from '@/components/ui/card';
import { Badge } from '@/components/ui/badge';
import { Palette } from 'lucide-react';
import { toast } from 'sonner';

export default function Theme() {
    const [activeTheme, setActiveTheme] = useState({
        name: 'Current Default',
        primary: '#C9B699',
        accent: '#B6A37F',
        accentHover: '#C8B588',
        bg: '#2A0713',
        text: '#C9B699',
        fieldBorder: '#7B5F51',
        error: '#E8CBB5'
    });
    const [savedThemes, setSavedThemes] = useState([]);
    const [loading, setLoading] = useState(true);

    useEffect(() => {
        // Load saved themes from database
        fetch('/api/themes')
            .then(res => res.json())
            .then(themes => {
                setSavedThemes(themes);
                // Find and set active theme if exists
                const active = themes.find(t => t.isActive);
                if (active) {
                    setActiveTheme({
                        name: active.name,
                        primary: active.primaryColor || active.primary,
                        accent: active.accent,
                        accentHover: active.accentHover || active.accent,
                        bg: active.bg,
                        text: active.text,
                        fieldBorder: active.fieldBorder || '#7B5F51',
                        error: active.error || '#E8CBB5'
                    });
                }
                setLoading(false);
            })
            .catch(err => {
                console.error('Error loading themes:', err);
                setLoading(false);
            });
    }, []);

    const presetThemes = [
        { 
            name: 'Current Default', 
            primary: '#C9B699', 
            accent: '#B6A37F', 
            accentHover: '#C8B588',
            bg: '#2A0713', 
            text: '#C9B699',
            fieldBorder: '#7B5F51',
            error: '#E8CBB5'
        },
        { 
            name: 'Classic Elegance', 
            primary: '#1e40af', 
            accent: '#60a5fa', 
            accentHover: '#7cb5fb',
            bg: '#ffffff', 
            text: '#1f2937',
            fieldBorder: '#d1d5db',
            error: '#ef4444'
        },
        { 
            name: 'Modern Minimalist', 
            primary: '#1f2937', 
            accent: '#9ca3af', 
            accentHover: '#b0b7bf',
            bg: '#f9fafb', 
            text: '#111827',
            fieldBorder: '#e5e7eb',
            error: '#dc2626'
        },
        { 
            name: 'Garden Romance', 
            primary: '#059669', 
            accent: '#6ee7b7', 
            accentHover: '#86efca',
            bg: '#f0fdf4', 
            text: '#064e3b',
            fieldBorder: '#a7f3d0',
            error: '#f87171'
        }
    ];

    const applyTheme = async () => {
        try {
            const response = await fetch('/api/themes', {
                method: 'POST',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify({ ...activeTheme, isActive: true })
            });
            
            if (!response.ok) {
                const error = await response.text();
                throw new Error(`Failed to save theme: ${error}`);
            }
            
            const savedTheme = await response.json();
            console.log('Theme saved:', savedTheme);
            
            // Reload themes to verify it was saved
            const themesResponse = await fetch('/api/themes');
            const themes = await themesResponse.json();
            setSavedThemes(themes);
            console.log('All themes in database:', themes);
            
            toast.success('Theme applied!', {
                description: 'Refresh the public RSVP page to see changes.'
            });
        } catch (error) {
            console.error('Error saving theme:', error);
            toast.error('Error saving theme', {
                description: error.message
            });
        }
    };

    return (
        <div>
            <div className="flex items-center justify-between mb-6">
                <h1 className="text-3xl font-bold">Theme Customization</h1>
                {!loading && (
                    <div className="text-sm text-muted-foreground">
                        {savedThemes.length} theme{savedThemes.length !== 1 ? 's' : ''} saved in database
                    </div>
                )}
            </div>

            <div className="grid grid-cols-2 lg:grid-cols-4 gap-4 mb-8">
                {presetThemes.map(theme => (
                    <div 
                        key={theme.name}
                        className="border rounded-lg p-4 cursor-pointer hover:shadow-lg"
                        style={{ backgroundColor: theme.bg, borderColor: theme.primary }}
                        onClick={() => setActiveTheme(theme)}
                    >
                        <div className="font-semibold mb-2" style={{ color: theme.text }}>{theme.name}</div>
                        <div className="flex gap-2">
                            <div className="w-8 h-8 rounded" style={{ backgroundColor: theme.primary }}></div>
                            <div className="w-8 h-8 rounded" style={{ backgroundColor: theme.accent }}></div>
                        </div>
                    </div>
                ))}
            </div>

            <div className="bg-card rounded-lg border p-6">
                <h2 className="text-xl font-semibold mb-4">Current Theme: {activeTheme.name}</h2>
                <div className="grid grid-cols-2 md:grid-cols-3 gap-4 mb-4">
                    <div>
                        <label className="block text-sm mb-1">Background</label>
                        <input type="color" value={activeTheme.bg} 
                            onChange={(e) => setActiveTheme({...activeTheme, bg: e.target.value})} 
                            className="w-full h-10 rounded"/>
                    </div>
                    <div>
                        <label className="block text-sm mb-1">Text Color</label>
                        <input type="color" value={activeTheme.text}
                            onChange={(e) => setActiveTheme({...activeTheme, text: e.target.value})}
                            className="w-full h-10 rounded"/>
                    </div>
                    <div>
                        <label className="block text-sm mb-1">Accent Color</label>
                        <input type="color" value={activeTheme.accent}
                            onChange={(e) => setActiveTheme({...activeTheme, accent: e.target.value})}
                            className="w-full h-10 rounded"/>
                    </div>
                    <div>
                        <label className="block text-sm mb-1">Accent Hover</label>
                        <input type="color" value={activeTheme.accentHover}
                            onChange={(e) => setActiveTheme({...activeTheme, accentHover: e.target.value})}
                            className="w-full h-10 rounded"/>
                    </div>
                    <div>
                        <label className="block text-sm mb-1">Field Border</label>
                        <input type="color" value={activeTheme.fieldBorder}
                            onChange={(e) => setActiveTheme({...activeTheme, fieldBorder: e.target.value})}
                            className="w-full h-10 rounded"/>
                    </div>
                    <div>
                        <label className="block text-sm mb-1">Error Color</label>
                        <input type="color" value={activeTheme.error}
                            onChange={(e) => setActiveTheme({...activeTheme, error: e.target.value})}
                            className="w-full h-10 rounded"/>
                    </div>
                </div>
                <Button onClick={applyTheme} className="w-full">Apply Theme to Public Pages</Button>
            </div>
        </div>
    );
}
