import React, { useState, useEffect } from 'react';
import { Button } from '@/components/ui/button';
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from '@/components/ui/card';
import { Badge } from '@/components/ui/badge';
import { Table, TableBody, TableCell, TableHead, TableHeader, TableRow } from '@/components/ui/table';
import { Dialog, DialogContent, DialogDescription, DialogHeader, DialogTitle } from '@/components/ui/dialog';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from '@/components/ui/select';
import { Skeleton } from '@/components/ui/skeleton';
import { Users, UserPlus, Download, Plus, Trash2, Edit } from 'lucide-react';
import { toast } from 'sonner';

export default function Seating() {
    const [events, setEvents] = useState([]);
    const [selectedEvent, setSelectedEvent] = useState(null);
    const [guests, setGuests] = useState([]);
    const [seating, setSeating] = useState([]);
    const [tables, setTables] = useState([]);
    const [loading, setLoading] = useState(true);
    const [showAssignDialog, setShowAssignDialog] = useState(false);
    const [editingAssignment, setEditingAssignment] = useState(null);

    useEffect(() => {
        const loadData = async () => {
            try {
                // Load events
                const eventsRes = await fetch('/api/events');
                const eventsData = await eventsRes.json();
                setEvents(eventsData);
                if (eventsData.length > 0) {
                    setSelectedEvent(eventsData[0].id);
                }

                // Load households to get guests
                const householdsRes = await fetch('/api/households');
                const householdsData = await householdsRes.json();
                console.log('Loaded households:', householdsData);
                
                // Extract guests from households
                let guestsData = [];
                householdsData.forEach(household => {
                    const householdGuestCount = household.guestCount || household.guests?.length || 1;
                    
                    // If household has a guests array, use those
                    if (household.guests && Array.isArray(household.guests)) {
                        household.guests.forEach(guest => {
                            guestsData.push({
                                id: guest.id,
                                firstName: guest.firstName || guest.name,
                                lastName: guest.lastName || '',
                                householdName: household.displayName,
                                householdId: household.id,
                                householdGuestCount: householdGuestCount,
                                name: guest.firstName && guest.lastName 
                                    ? `${guest.firstName} ${guest.lastName}` 
                                    : guest.name
                            });
                        });
                    } else {
                        // Otherwise, treat household as a single guest unit
                        guestsData.push({
                            id: `household-${household.id}`,
                            firstName: household.displayName?.split(' ')[0] || household.displayName,
                            lastName: household.displayName?.split(' ').slice(1).join(' ') || '',
                            householdName: household.displayName,
                            householdId: household.id,
                            householdGuestCount: householdGuestCount,
                            name: household.displayName
                        });
                    }
                });
                
                console.log('Processed guests:', guestsData.length, guestsData);
                setGuests(guestsData);
                setLoading(false);
            } catch (error) {
                console.error('Error loading data:', error);
                setLoading(false);
            }
        };
        
        loadData();
    }, []);

    const unassignedGuests = guests.filter(g => 
        !seating.some(s => s.guestId === g.id && s.eventId === selectedEvent)
    );

    const getTableGuests = (tableName) => {
        return seating
            .filter(s => s.tableName === tableName && s.eventId === selectedEvent)
            .map(s => guests.find(g => g.id === s.guestId))
            .filter(Boolean);
    };

    const assignToTable = async (assignment) => {
        try {
            await fetch('/api/seating', {
                method: 'POST',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify({
                    ...assignment,
                    eventId: selectedEvent
                })
            });
            // Reload seating data
            setShowAssignDialog(false);
            setEditingAssignment(null);
            toast.success('Seat assigned successfully!');
        } catch (error) {
            toast.error('Failed to assign seat');
        }
    };

    const removeAssignment = async (assignmentId) => {
        if (!confirm('Remove this seat assignment?')) return;
        try {
            await fetch(`/api/seating/${assignmentId}`, { method: 'DELETE' });
            toast.success('Assignment removed');
        } catch (error) {
            toast.error('Failed to remove assignment');
        }
    };

    const addTable = () => {
        const tableName = prompt('Enter table name (e.g., "Table 1", "Head Table"):');
        if (tableName) {
            setTables([...tables, { name: tableName, capacity: 8 }]);
        }
    };

    if (loading) {
        return (
            <div className="space-y-6">
                <Skeleton className="h-10 w-64" />
                <Skeleton className="h-96 w-full" />
            </div>
        );
    }

    return (
        <div className="space-y-6">
            {/* Header */}
            <div className="flex flex-col gap-4 sm:flex-row sm:items-center sm:justify-between">
                <div>
                    <h1 className="text-3xl font-bold tracking-tight">Seating Arrangements</h1>
                    <p className="text-muted-foreground">Assign guests to tables and seats</p>
                </div>
                <div className="flex gap-2 flex-wrap">
                    <Button variant="outline" size="sm" onClick={addTable}>
                        <Plus className="h-4 w-4 mr-2" />
                        Add Table
                    </Button>
                    <Button variant="outline" size="sm">
                        <Download className="h-4 w-4 mr-2" />
                        Export
                    </Button>
                    <Button size="sm" onClick={() => {
                        setEditingAssignment({ guestId: '', tableName: '', seatNumber: '' });
                        setShowAssignDialog(true);
                    }}>
                        <UserPlus className="h-4 w-4 mr-2" />
                        Assign Seat
                    </Button>
                </div>
            </div>

            {/* Stats Cards */}
            <div className="grid gap-4 grid-cols-1 sm:grid-cols-3">
                <Card>
                    <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                        <CardTitle className="text-sm font-medium">Total Guests</CardTitle>
                        <Users className="h-4 w-4 text-muted-foreground" />
                    </CardHeader>
                    <CardContent>
                        <div className="text-2xl font-bold">{guests.length}</div>
                    </CardContent>
                </Card>
                <Card>
                    <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                        <CardTitle className="text-sm font-medium">Assigned</CardTitle>
                        <UserPlus className="h-4 w-4 text-green-600" />
                    </CardHeader>
                    <CardContent>
                        <div className="text-2xl font-bold text-green-600">{guests.length - unassignedGuests.length}</div>
                        <p className="text-xs text-muted-foreground">Unassigned: {unassignedGuests.length}</p>
                    </CardContent>
                </Card>
                <Card>
                    <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                        <CardTitle className="text-sm font-medium">Tables</CardTitle>
                        <Users className="h-4 w-4 text-blue-600" />
                    </CardHeader>
                    <CardContent>
                        <div className="text-2xl font-bold">{tables.length}</div>
                    </CardContent>
                </Card>
            </div>

            {/* Event Selector */}
            <Card>
                <CardContent className="pt-6">
                    <div className="flex items-center gap-4">
                        <Label>Select Event:</Label>
                        <Select value={selectedEvent?.toString()} onValueChange={(value) => setSelectedEvent(parseInt(value))}>
                            <SelectTrigger className="w-full sm:w-[280px]">
                                <SelectValue />
                            </SelectTrigger>
                            <SelectContent>
                                {events.map(event => (
                                    <SelectItem key={event.id} value={event.id.toString()}>
                                        {event.name}
                                    </SelectItem>
                                ))}
                            </SelectContent>
                        </Select>
                    </div>
                </CardContent>
            </Card>

            {/* Seating Assignments Table */}
            <Card>
                <CardHeader>
                    <CardTitle>Seating Assignments</CardTitle>
                    <CardDescription>All guest seat assignments for the selected event</CardDescription>
                </CardHeader>
                <CardContent>
                    <Table>
                        <TableHeader>
                            <TableRow>
                                <TableHead>Guest Name</TableHead>
                                <TableHead>Household</TableHead>
                                <TableHead>Table</TableHead>
                                <TableHead className="w-[120px]">Seat #</TableHead>
                                <TableHead className="w-[100px] text-right">Actions</TableHead>
                            </TableRow>
                        </TableHeader>
                        <TableBody>
                            {seating.filter(s => s.eventId === selectedEvent).length === 0 ? (
                                <TableRow>
                                    <TableCell colSpan={5} className="h-24 text-center">
                                        <div className="flex flex-col items-center gap-2 text-muted-foreground">
                                            <Users className="h-8 w-8 opacity-50" />
                                            <p>No seat assignments yet.</p>
                                            <Button variant="outline" size="sm" onClick={() => {
                                                setEditingAssignment({ guestId: '', tableName: '', seatNumber: '' });
                                                setShowAssignDialog(true);
                                            }}>
                                                <UserPlus className="h-4 w-4 mr-2" />
                                                Assign First Seat
                                            </Button>
                                        </div>
                                    </TableCell>
                                </TableRow>
                            ) : (
                                seating
                                    .filter(s => s.eventId === selectedEvent)
                                    .map(assignment => {
                                        const guest = guests.find(g => g.id === assignment.guestId);
                                        if (!guest) return null;
                                        
                                        return (
                                            <TableRow key={assignment.id}>
                                                <TableCell className="font-medium">
                                                    {guest.firstName} {guest.lastName}
                                                </TableCell>
                                                <TableCell className="text-sm text-muted-foreground">
                                                    {guest.householdName}
                                                </TableCell>
                                                <TableCell>
                                                    <Badge variant="outline">{assignment.tableName}</Badge>
                                                </TableCell>
                                                <TableCell>
                                                    {assignment.seatNumber ? (
                                                        <Badge>{assignment.seatNumber}</Badge>
                                                    ) : (
                                                        <span className="text-sm text-muted-foreground">-</span>
                                                    )}
                                                </TableCell>
                                                <TableCell className="text-right">
                                                    <div className="flex gap-1 justify-end">
                                                        <Button 
                                                            variant="ghost" 
                                                            size="sm"
                                                            onClick={() => {
                                                                setEditingAssignment(assignment);
                                                                setShowAssignDialog(true);
                                                            }}
                                                        >
                                                            <Edit className="h-4 w-4" />
                                                        </Button>
                                                        <Button 
                                                            variant="ghost" 
                                                            size="sm"
                                                            onClick={() => removeAssignment(assignment.id)}
                                                        >
                                                            <Trash2 className="h-4 w-4 text-red-600" />
                                                        </Button>
                                                    </div>
                                                </TableCell>
                                            </TableRow>
                                        );
                                    })
                            )}
                        </TableBody>
                    </Table>
                </CardContent>
            </Card>

            {/* Unassigned Guests */}
            {unassignedGuests.length > 0 && (
                <Card>
                    <CardHeader>
                        <CardTitle>Unassigned Guests ({unassignedGuests.length})</CardTitle>
                        <CardDescription>Guests without seat assignments</CardDescription>
                    </CardHeader>
                    <CardContent>
                        <div className="grid grid-cols-1 sm:grid-cols-2 md:grid-cols-3 gap-2">
                            {unassignedGuests.map(guest => (
                                <div key={guest.id} className="flex items-center justify-between p-3 border rounded-lg">
                                    <div>
                                        <div className="font-medium text-sm">{guest.firstName} {guest.lastName}</div>
                                        <div className="text-xs text-muted-foreground">{guest.householdName}</div>
                                    </div>
                                    <Button 
                                        variant="outline" 
                                        size="sm"
                                        onClick={() => {
                                            setEditingAssignment({ guestId: guest.id, tableName: '', seatNumber: '' });
                                            setShowAssignDialog(true);
                                        }}
                                    >
                                        <UserPlus className="h-4 w-4" />
                                    </Button>
                                </div>
                            ))}
                        </div>
                    </CardContent>
                </Card>
            )}

            {/* Assign Seat Dialog */}
            {showAssignDialog && editingAssignment && (
                <Dialog open={showAssignDialog} onOpenChange={(open) => {
                    if (!open) {
                        setShowAssignDialog(false);
                        setEditingAssignment(null);
                    }
                }}>
                    <DialogContent>
                        <DialogHeader>
                            <DialogTitle>{editingAssignment.id ? 'Edit' : 'Assign'} Seat</DialogTitle>
                            <DialogDescription>
                                Assign a guest to a table and seat number
                            </DialogDescription>
                        </DialogHeader>
                        <div className="space-y-4 mt-4">
                            <div className="space-y-2">
                                <Label htmlFor="guest">Guest</Label>
                                <Select 
                                    value={editingAssignment.guestId?.toString()} 
                                    onValueChange={(value) => {
                                        // Handle both numeric IDs and string IDs (like "household-123")
                                        const guestId = value.startsWith('household-') ? value : parseInt(value);
                                        const selectedGuest = guests.find(g => g.id.toString() === value);
                                        console.log('Selected guest:', selectedGuest);
                                        console.log('Household guest count:', selectedGuest?.householdGuestCount);
                                        setEditingAssignment({
                                            ...editingAssignment, 
                                            guestId,
                                            _selectedGuestData: selectedGuest // Store for display purposes
                                        });
                                    }}
                                >
                                    <SelectTrigger id="guest">
                                        <SelectValue placeholder="Select guest" />
                                    </SelectTrigger>
                                    <SelectContent>
                                        {guests.length === 0 ? (
                                            <SelectItem value="none" disabled>No guests available</SelectItem>
                                        ) : (
                                            guests.map(guest => {
                                                const displayName = guest.firstName && guest.lastName 
                                                    ? `${guest.firstName} ${guest.lastName}` 
                                                    : guest.name || `Guest ${guest.id}`;
                                                const householdInfo = guest.householdName 
                                                    ? ` (${guest.householdName})` 
                                                    : '';
                                                const guestCountInfo = guest.householdGuestCount > 1 
                                                    ? ` - ${guest.householdGuestCount} guests` 
                                                    : '';
                                                
                                                return (
                                                    <SelectItem key={guest.id} value={guest.id.toString()}>
                                                        {displayName}{householdInfo}{guestCountInfo}
                                                    </SelectItem>
                                                );
                                            })
                                        )}
                                    </SelectContent>
                                </Select>
                                {guests.length === 0 && (
                                    <p className="text-xs text-muted-foreground">
                                        No guests found. Please add households/guests first.
                                    </p>
                                )}
                                {editingAssignment._selectedGuestData && (
                                    <div className={`flex items-center gap-2 p-3 rounded-lg ${
                                        (editingAssignment._selectedGuestData.householdGuestCount || 1) > 1 
                                            ? 'bg-blue-50 border border-blue-200' 
                                            : 'bg-gray-50 border border-gray-200'
                                    }`}>
                                        <Users className={`h-4 w-4 ${
                                            (editingAssignment._selectedGuestData.householdGuestCount || 1) > 1 
                                                ? 'text-blue-600' 
                                                : 'text-gray-600'
                                        }`} />
                                        <div className={`text-sm ${
                                            (editingAssignment._selectedGuestData.householdGuestCount || 1) > 1 
                                                ? 'text-blue-800' 
                                                : 'text-gray-700'
                                        }`}>
                                            <strong>{editingAssignment._selectedGuestData.householdName || 'This household'}</strong> has{' '}
                                            <strong>{editingAssignment._selectedGuestData.householdGuestCount || 1} {(editingAssignment._selectedGuestData.householdGuestCount || 1) === 1 ? 'guest' : 'guests'}</strong> attending.
                                            {(editingAssignment._selectedGuestData.householdGuestCount || 1) > 1 && (
                                                <span> Consider seating them together.</span>
                                            )}
                                        </div>
                                    </div>
                                )}
                            </div>

                            <div className="space-y-2">
                                <Label htmlFor="table">Table Name</Label>
                                <Input
                                    id="table"
                                    value={editingAssignment.tableName}
                                    onChange={(e) => setEditingAssignment({...editingAssignment, tableName: e.target.value})}
                                    placeholder="e.g., Table 1, Head Table"
                                />
                            </div>

                            <div className="space-y-2">
                                <Label htmlFor="seat">Seat Number (Optional)</Label>
                                <Input
                                    id="seat"
                                    type="number"
                                    value={editingAssignment.seatNumber || ''}
                                    onChange={(e) => setEditingAssignment({...editingAssignment, seatNumber: e.target.value})}
                                    placeholder="e.g., 1, 2, 3"
                                />
                            </div>

                            <div className="flex gap-2 pt-4">
                                <Button 
                                    className="flex-1" 
                                    onClick={() => assignToTable(editingAssignment)}
                                    disabled={!editingAssignment.guestId || !editingAssignment.tableName}
                                >
                                    {editingAssignment.id ? 'Update' : 'Assign'} Seat
                                </Button>
                                <Button variant="outline" onClick={() => {
                                    setShowAssignDialog(false);
                                    setEditingAssignment(null);
                                }}>
                                    Cancel
                                </Button>
                            </div>
                        </div>
                    </DialogContent>
                </Dialog>
            )}
        </div>
    );
}
