import React, { useState, useEffect } from 'react';
import { Button } from '@/components/ui/button';
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from '@/components/ui/card';
import { Badge } from '@/components/ui/badge';
import { Table, TableBody, TableCell, TableHead, TableHeader, TableRow } from '@/components/ui/table';
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from '@/components/ui/select';
import { Skeleton } from '@/components/ui/skeleton';
import { Users, UserCheck, UserX, Download, Calendar, Utensils, ChevronLeft, ChevronRight, ChevronsLeft, ChevronsRight, Search, Eye, AlertCircle, Edit, QrCode, Send, RefreshCw } from 'lucide-react';
import { Input } from '@/components/ui/input';
import { Dialog, DialogContent, DialogDescription, DialogHeader, DialogTitle } from '@/components/ui/dialog';
import { Checkbox } from '@/components/ui/checkbox';
import { Label } from '@/components/ui/label';
import { toast } from 'sonner';

export default function RSVPs() {
    const [mode, setMode] = useState('HOUSEHOLD');
    const [rsvps, setRsvps] = useState([]);
    const [events, setEvents] = useState([]);
    const [selectedEvent, setSelectedEvent] = useState(null);
    const [loading, setLoading] = useState(true);
    const [searchTerm, setSearchTerm] = useState('');
    const [currentPage, setCurrentPage] = useState(1);
    const [itemsPerPage, setItemsPerPage] = useState(10);
    const [viewingDetails, setViewingDetails] = useState(null);
    const [editingDetails, setEditingDetails] = useState(false);
    const [editFormData, setEditFormData] = useState({ adults: [], children: [] });
    const [selectedIds, setSelectedIds] = useState([]);
    const [bulkSendModal, setBulkSendModal] = useState(null);
    const [sending, setSending] = useState(false);

    useEffect(() => {
        fetch('/api/settings')
            .then(res => res.json())
            .then(data => {
                setMode(data.publicRsvpMode || 'HOUSEHOLD');
            });

        fetch('/api/events')
            .then(res => res.json())
            .then(data => {
                setEvents(data);
                if (data.length > 0) {
                    setSelectedEvent(data[0].id);
                }
            });
    }, []);

    useEffect(() => {
        if (selectedEvent) {
            loadRsvps();
        } else {
            // No event selected, stop loading
            setLoading(false);
        }
    }, [selectedEvent, mode]);

    const loadRsvps = () => {
        const endpoint = mode === 'PER_GUEST' ? '/api/rsvps/guest' : '/api/rsvps/household';
        fetch(`${endpoint}?eventId=${selectedEvent}`)
            .then(res => res.json())
            .then(data => {
                setRsvps(data);
                setLoading(false);
            });
    };

    const exportCaterer = () => {
        window.location.href = `/api/export/caterer/${selectedEvent}`;
    };

    const exportDietary = () => {
        window.location.href = `/api/export/dietary/${selectedEvent}`;
    };

    // Filter RSVPs based on search
    const filteredRsvps = rsvps.filter(r => {
        const searchLower = searchTerm.toLowerCase();
        const nameMatch = (mode === 'PER_GUEST' ? r.guestName : r.householdName)?.toLowerCase().includes(searchLower);
        return nameMatch;
    });

    // Reset to page 1 when search or event changes
    useEffect(() => {
        setCurrentPage(1);
    }, [searchTerm, selectedEvent]);

    // Calculate pagination
    const totalPages = Math.ceil(filteredRsvps.length / itemsPerPage);
    const startIndex = (currentPage - 1) * itemsPerPage;
    const endIndex = startIndex + itemsPerPage;
    const paginatedRsvps = filteredRsvps.slice(startIndex, endIndex);

    const stats = {
        total: rsvps.length,
        yes: rsvps.filter(r => r.status === 'YES').length,
        no: rsvps.filter(r => r.status === 'NO').length,
        pending: rsvps.filter(r => !r.status || r.status === 'PENDING').length,
    };

    // Selection handlers
    const toggleSelect = (id) => {
        setSelectedIds(prev => 
            prev.includes(id) ? prev.filter(x => x !== id) : [...prev, id]
        );
    };

    const toggleSelectAll = () => {
        if (paginatedRsvps.every(r => selectedIds.includes(r.householdId))) {
            setSelectedIds(prev => prev.filter(id => !paginatedRsvps.some(r => r.householdId === id)));
        } else {
            const newIds = paginatedRsvps.map(r => r.householdId).filter(id => !selectedIds.includes(id));
            setSelectedIds(prev => [...prev, ...newIds]);
        }
    };

    const selectedRsvps = rsvps.filter(r => selectedIds.includes(r.householdId));

    const startEditingDetails = (rsvp) => {
        // Initialize form with existing data or empty arrays
        const adults = rsvp.guestDetails?.adults || Array.from({ length: rsvp.adultsCount || 0 }, () => ({ name: '', meal: '', dietary: '' }));
        const children = rsvp.guestDetails?.children || Array.from({ length: rsvp.childrenCount || 0 }, () => ({ name: '', meal: '', dietary: '' }));
        
        setEditFormData({ adults, children });
        setEditingDetails(true);
    };

    const updateAdult = (index, field, value) => {
        const newAdults = [...editFormData.adults];
        newAdults[index] = { ...newAdults[index], [field]: value };
        setEditFormData({ ...editFormData, adults: newAdults });
    };

    const updateChild = (index, field, value) => {
        const newChildren = [...editFormData.children];
        newChildren[index] = { ...newChildren[index], [field]: value };
        setEditFormData({ ...editFormData, children: newChildren });
    };

    const saveGuestDetails = async () => {
        try {
            const response = await fetch(`/api/admin/rsvps/${viewingDetails.id}/guest-details`, {
                method: 'PUT',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify({
                    adults: editFormData.adults,
                    children: editFormData.children
                })
            });

            if (response.ok) {
                // Reload RSVPs
                loadRsvps();
                // Update viewing details
                setViewingDetails({
                    ...viewingDetails,
                    guestDetails: {
                        adults: editFormData.adults,
                        children: editFormData.children
                    }
                });
                setEditingDetails(false);
            } else {
                alert('Failed to save guest details');
            }
        } catch (error) {
            alert('Error saving guest details');
        }
    };

    if (loading) {
        return (
            <div className="space-y-6">
                <Skeleton className="h-10 w-64" />
                <div className="grid grid-cols-2 md:grid-cols-4 gap-4">
                    {[...Array(4)].map((_, i) => (
                        <Card key={i}>
                            <CardHeader className="pb-2">
                                <Skeleton className="h-4 w-20" />
                            </CardHeader>
                            <CardContent>
                                <Skeleton className="h-8 w-16" />
                            </CardContent>
                        </Card>
                    ))}
                </div>
                <Skeleton className="h-96 w-full" />
            </div>
        );
    }

    const currentEvent = events.find(e => e.id === selectedEvent);

    // Show empty state if no events
    if (events.length === 0) {
        return (
            <div className="space-y-6">
                <div>
                    <h1 className="text-3xl font-bold tracking-tight">RSVPs</h1>
                    <p className="text-muted-foreground">View and manage event responses</p>
                </div>
                <Card>
                    <CardContent className="flex flex-col items-center justify-center py-16">
                        <Calendar className="h-16 w-16 text-muted-foreground mb-4 opacity-50" />
                        <h3 className="text-xl font-semibold mb-2">No Events Created Yet</h3>
                        <p className="text-muted-foreground text-center mb-6">
                            Create an event first to start receiving RSVPs
                        </p>
                        <Button onClick={() => window.location.href = '/admin/events'}>
                            <Calendar className="h-4 w-4 mr-2" />
                            Create Event
                        </Button>
                    </CardContent>
                </Card>
            </div>
        );
    }

    return (
        <div className="space-y-6">
            {/* Header */}
            <div className="flex flex-col gap-4 sm:flex-row sm:items-center sm:justify-between">
                <div>
                    <h1 className="text-3xl font-bold tracking-tight">RSVPs</h1>
                    <p className="text-muted-foreground">View and manage event responses</p>
                </div>
                <div className="flex gap-2 flex-wrap">
                    <Button onClick={exportCaterer} variant="outline" size="sm">
                        <Download className="h-4 w-4 mr-2" />
                        Caterer
                    </Button>
                    <Button onClick={exportDietary} variant="outline" size="sm">
                        <Download className="h-4 w-4 mr-2" />
                        Dietary
                    </Button>
                    <Button onClick={() => window.location.href = '/api/export/attendance'} size="sm">
                        <Download className="h-4 w-4 mr-2" />
                        Attendance
                    </Button>
                </div>
            </div>

            {/* Stats Cards */}
            <div className="grid gap-4 grid-cols-2 sm:grid-cols-4">
                <Card>
                    <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                        <CardTitle className="text-sm font-medium">Total Responses</CardTitle>
                        <Users className="h-4 w-4 text-muted-foreground" />
                    </CardHeader>
                    <CardContent>
                        <div className="text-2xl font-bold">{stats.total}</div>
                        <p className="text-xs text-muted-foreground">{currentEvent?.name}</p>
                    </CardContent>
                </Card>
                <Card>
                    <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                        <CardTitle className="text-sm font-medium">Attending</CardTitle>
                        <UserCheck className="h-4 w-4 text-green-600" />
                    </CardHeader>
                    <CardContent>
                        <div className="text-2xl font-bold text-green-600">{stats.yes}</div>
                        <p className="text-xs text-muted-foreground">
                            {stats.total > 0 ? Math.round((stats.yes / stats.total) * 100) : 0}% of responses
                        </p>
                    </CardContent>
                </Card>
                <Card>
                    <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                        <CardTitle className="text-sm font-medium">Not Attending</CardTitle>
                        <UserX className="h-4 w-4 text-red-600" />
                    </CardHeader>
                    <CardContent>
                        <div className="text-2xl font-bold text-red-600">{stats.no}</div>
                        <p className="text-xs text-muted-foreground">
                            {stats.total > 0 ? Math.round((stats.no / stats.total) * 100) : 0}% of responses
                        </p>
                    </CardContent>
                </Card>
                <Card>
                    <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                        <CardTitle className="text-sm font-medium">Pending</CardTitle>
                        <Calendar className="h-4 w-4 text-yellow-600" />
                    </CardHeader>
                    <CardContent>
                        <div className="text-2xl font-bold text-yellow-600">{stats.pending}</div>
                        <p className="text-xs text-muted-foreground">No response yet</p>
                    </CardContent>
                </Card>
            </div>

            {/* Filters */}
            <Card>
                <CardContent className="pt-6">
                    <div className="flex flex-col sm:flex-row gap-4">
                        <div className="flex items-center gap-4 flex-1">
                            <label className="text-sm font-medium whitespace-nowrap">Select an Event:</label>
                            <Select value={selectedEvent?.toString()} onValueChange={(value) => setSelectedEvent(parseInt(value))}>
                                <SelectTrigger className="w-full sm:w-[280px]">
                                    <SelectValue />
                                </SelectTrigger>
                                <SelectContent>
                                    {events.map(event => (
                                        <SelectItem key={event.id} value={event.id.toString()}>
                                            {event.nameOne && event.nameTwo 
                                                ? `${event.nameOne} ${event.andText || '&'} ${event.nameTwo}`
                                                : event.name
                                            }
                                        </SelectItem>
                                    ))}
                                </SelectContent>
                            </Select>
                        </div>
                        <div className="relative flex-1 sm:max-w-sm">
                            <Search className="absolute left-3 top-1/2 -translate-y-1/2 h-4 w-4 text-muted-foreground" />
                            <Input
                                placeholder="Search by name..."
                                value={searchTerm}
                                onChange={(e) => setSearchTerm(e.target.value)}
                                className="pl-9"
                            />
                        </div>
                    </div>
                </CardContent>
            </Card>

            {/* Bulk Actions - Only show in HOUSEHOLD mode */}
            {mode === 'HOUSEHOLD' && (
                <Card>
                    <CardContent className="pt-6">
                        <div className="flex flex-col sm:flex-row items-center justify-between gap-4">
                            <div className="text-sm text-muted-foreground">
                                {selectedIds.length > 0 ? (
                                    <span><strong>{selectedIds.length}</strong> household{selectedIds.length !== 1 ? 's' : ''} selected</span>
                                ) : (
                                    <span>Select confirmed households to send QR tickets</span>
                                )}
                            </div>
                            <Button 
                                variant="default" 
                                size="sm"
                                disabled={selectedIds.length === 0}
                                onClick={() => {
                                    setBulkSendModal({ recipients: selectedRsvps });
                                }}
                            >
                                <QrCode className="h-4 w-4 mr-2" />
                                Send QR Tickets {selectedIds.length > 0 && `(${selectedIds.length})`}
                            </Button>
                        </div>
                    </CardContent>
                </Card>
            )}

            {/* RSVPs Table */}
            <Card>
                <Table>
                    <TableHeader>
                        <TableRow>
                            {mode === 'HOUSEHOLD' && (
                                <TableHead className="w-[50px]">
                                    <Checkbox
                                        checked={paginatedRsvps.length > 0 && paginatedRsvps.every(r => selectedIds.includes(r.householdId))}
                                        onCheckedChange={toggleSelectAll}
                                    />
                                </TableHead>
                            )}
                            <TableHead>{mode === 'PER_GUEST' ? 'Guest' : 'Household'}</TableHead>
                            <TableHead className="w-[120px]">Status</TableHead>
                            {mode === 'PER_GUEST' ? (
                                <>
                                    <TableHead>Meal</TableHead>
                                    <TableHead>Dietary</TableHead>
                                </>
                            ) : (
                                <>
                                    <TableHead className="w-[120px]">Adults</TableHead>
                                    <TableHead className="w-[120px]">Children</TableHead>
                                    <TableHead className="w-[100px]">Details</TableHead>
                                </>
                            )}
                            <TableHead className="w-[140px]">Responded</TableHead>
                        </TableRow>
                    </TableHeader>
                    <TableBody>
                        {paginatedRsvps.length === 0 ? (
                            <TableRow>
                                <TableCell colSpan={mode === 'PER_GUEST' ? 5 : 7} className="h-24 text-center">
                                    <div className="flex flex-col items-center gap-2 text-muted-foreground">
                                        <Users className="h-8 w-8 opacity-50" />
                                        <p>{filteredRsvps.length === 0 ? 'No RSVPs found.' : 'No results on this page.'}</p>
                                    </div>
                                </TableCell>
                            </TableRow>
                        ) : (
                            paginatedRsvps.map(rsvp => (
                                <TableRow key={rsvp.id}>
                                    {mode === 'HOUSEHOLD' && (
                                        <TableCell>
                                            <Checkbox
                                                checked={selectedIds.includes(rsvp.householdId)}
                                                onCheckedChange={() => toggleSelect(rsvp.householdId)}
                                            />
                                        </TableCell>
                                    )}
                                    <TableCell className="font-medium">
                                        {mode === 'PER_GUEST' ? rsvp.guestName : rsvp.householdName}
                                    </TableCell>
                                    <TableCell>
                                        {rsvp.status === 'YES' ? (
                                            <Badge variant="default" className="bg-green-500">
                                                <UserCheck className="h-3 w-3 mr-1" />
                                                Yes
                                            </Badge>
                                        ) : rsvp.status === 'NO' ? (
                                            <Badge variant="destructive">
                                                <UserX className="h-3 w-3 mr-1" />
                                                No
                                            </Badge>
                                        ) : (
                                            <Badge variant="secondary">Pending</Badge>
                                        )}
                                    </TableCell>
                                    {mode === 'PER_GUEST' ? (
                                        <>
                                            <TableCell>
                                                {rsvp.mealChoice ? (
                                                    <Badge variant="outline">
                                                        <Utensils className="h-3 w-3 mr-1" />
                                                        {rsvp.mealChoice}
                                                    </Badge>
                                                ) : (
                                                    <span className="text-muted-foreground">-</span>
                                                )}
                                            </TableCell>
                                            <TableCell className="text-sm text-muted-foreground">
                                                {rsvp.dietary || '-'}
                                            </TableCell>
                                        </>
                                    ) : (
                                        <>
                                            <TableCell>
                                                {rsvp.adultsCount > 0 ? (
                                                    <div className="flex items-center gap-2">
                                                        <Badge variant="outline">
                                                            <Users className="h-3 w-3 mr-1" />
                                                            {rsvp.adultsCount}
                                                        </Badge>
                                                        {rsvp.guestDetails?.adults?.length > 0 && (
                                                            <Button
                                                                variant="ghost"
                                                                size="sm"
                                                                className="h-7 w-7 p-0"
                                                                onClick={() => setViewingDetails(rsvp)}
                                                            >
                                                                <Eye className="h-4 w-4" />
                                                            </Button>
                                                        )}
                                                    </div>
                                                ) : (
                                                    <Badge variant="outline">0</Badge>
                                                )}
                                            </TableCell>
                                            <TableCell>
                                                {rsvp.childrenCount > 0 ? (
                                                    <div className="flex items-center gap-2">
                                                        <Badge variant="outline">
                                                            {rsvp.childrenCount}
                                                        </Badge>
                                                        {rsvp.guestDetails?.children?.length > 0 && (
                                                            <Button
                                                                variant="ghost"
                                                                size="sm"
                                                                className="h-7 w-7 p-0"
                                                                onClick={() => setViewingDetails(rsvp)}
                                                            >
                                                                <Eye className="h-4 w-4" />
                                                            </Button>
                                                        )}
                                                    </div>
                                                ) : (
                                                    <Badge variant="outline">0</Badge>
                                                )}
                                            </TableCell>
                                            <TableCell>
                                                {(rsvp.adultsCount > 0 || rsvp.childrenCount > 0) ? (
                                                    <Button
                                                        variant="outline"
                                                        size="sm"
                                                        onClick={() => setViewingDetails(rsvp)}
                                                    >
                                                        <Eye className="h-4 w-4 mr-2" />
                                                        View
                                                    </Button>
                                                ) : (
                                                    <span className="text-muted-foreground">-</span>
                                                )}
                                            </TableCell>
                                        </>
                                    )}
                                    <TableCell className="text-sm text-muted-foreground">
                                        {rsvp.respondedAt ? new Date(rsvp.respondedAt).toLocaleDateString() : '-'}
                                    </TableCell>
                                </TableRow>
                            ))
                        )}
                    </TableBody>
                </Table>

                {/* Pagination */}
                {filteredRsvps.length > 0 && (
                    <div className="flex items-center justify-between px-6 py-4 border-t">
                        <div className="flex items-center gap-4">
                            <p className="text-sm text-muted-foreground">
                                Showing {startIndex + 1} to {Math.min(endIndex, filteredRsvps.length)} of {filteredRsvps.length} RSVPs
                            </p>
                            <Select value={itemsPerPage.toString()} onValueChange={(value) => setItemsPerPage(Number(value))}>
                                <SelectTrigger className="w-[100px]">
                                    <SelectValue />
                                </SelectTrigger>
                                <SelectContent>
                                    <SelectItem value="10">10 / page</SelectItem>
                                    <SelectItem value="25">25 / page</SelectItem>
                                    <SelectItem value="50">50 / page</SelectItem>
                                    <SelectItem value="100">100 / page</SelectItem>
                                </SelectContent>
                            </Select>
                        </div>
                        
                        <div className="flex items-center gap-2">
                            <Button
                                variant="outline"
                                size="sm"
                                onClick={() => setCurrentPage(1)}
                                disabled={currentPage === 1}
                            >
                                <ChevronsLeft className="h-4 w-4" />
                            </Button>
                            <Button
                                variant="outline"
                                size="sm"
                                onClick={() => setCurrentPage(currentPage - 1)}
                                disabled={currentPage === 1}
                            >
                                <ChevronLeft className="h-4 w-4" />
                            </Button>
                            <span className="text-sm font-medium px-2">
                                Page {currentPage} of {totalPages}
                            </span>
                            <Button
                                variant="outline"
                                size="sm"
                                onClick={() => setCurrentPage(currentPage + 1)}
                                disabled={currentPage === totalPages}
                            >
                                <ChevronRight className="h-4 w-4" />
                            </Button>
                            <Button
                                variant="outline"
                                size="sm"
                                onClick={() => setCurrentPage(totalPages)}
                                disabled={currentPage === totalPages}
                            >
                                <ChevronsRight className="h-4 w-4" />
                            </Button>
                        </div>
                    </div>
                )}
            </Card>

            {/* Guest Details Modal */}
            {viewingDetails && (
                <Dialog open={!!viewingDetails} onOpenChange={(open) => !open && setViewingDetails(null)}>
                    <DialogContent className="max-w-2xl max-h-[80vh] overflow-y-auto">
                        <DialogHeader>
                            <div className="flex items-center justify-between">
                                <div>
                                    <DialogTitle>{viewingDetails.householdName} - Guest Details</DialogTitle>
                                    <DialogDescription>
                                        Individual guest meal choices and dietary restrictions
                                    </DialogDescription>
                                </div>
                                {!editingDetails && (
                                    <Button
                                        variant="outline"
                                        size="sm"
                                        onClick={() => startEditingDetails(viewingDetails)}
                                    >
                                        <Edit className="h-4 w-4 mr-2" />
                                        Edit Details
                                    </Button>
                                )}
                            </div>
                        </DialogHeader>

                        <div className="space-y-6 mt-4">
                            {/* Edit Mode */}
                            {editingDetails && (
                                <div className="space-y-6">
                                    {/* Edit Adults */}
                                    {editFormData.adults.length > 0 && (
                                        <div>
                                            <h3 className="text-lg font-semibold mb-3 flex items-center gap-2">
                                                <Users className="h-5 w-5" />
                                                Adults ({editFormData.adults.length})
                                            </h3>
                                            <div className="space-y-3">
                                                {editFormData.adults.map((adult, idx) => (
                                                    <div key={idx} className="p-4 border rounded-lg bg-muted/50 space-y-3">
                                                        <p className="font-medium text-sm text-muted-foreground">Adult {idx + 1}</p>
                                                        <input
                                                            type="text"
                                                            placeholder="Full Name"
                                                            value={adult.name}
                                                            onChange={(e) => updateAdult(idx, 'name', e.target.value)}
                                                            className="w-full px-3 py-2 border rounded-md"
                                                        />
                                                        <select
                                                            value={adult.meal}
                                                            onChange={(e) => updateAdult(idx, 'meal', e.target.value)}
                                                            className="w-full px-3 py-2 border rounded-md"
                                                        >
                                                            <option value="">Select Meal</option>
                                                            {currentEvent?.mealOptions?.map(meal => (
                                                                <option key={meal.code} value={meal.code}>
                                                                    {meal.name}
                                                                </option>
                                                            ))}
                                                        </select>
                                                        <input
                                                            type="text"
                                                            placeholder="Dietary Restrictions"
                                                            value={adult.dietary}
                                                            onChange={(e) => updateAdult(idx, 'dietary', e.target.value)}
                                                            className="w-full px-3 py-2 border rounded-md"
                                                        />
                                                    </div>
                                                ))}
                                            </div>
                                        </div>
                                    )}

                                    {/* Edit Children */}
                                    {editFormData.children.length > 0 && (
                                        <div>
                                            <h3 className="text-lg font-semibold mb-3 flex items-center gap-2">
                                                <Users className="h-5 w-5" />
                                                Children ({editFormData.children.length})
                                            </h3>
                                            <div className="space-y-3">
                                                {editFormData.children.map((child, idx) => (
                                                    <div key={idx} className="p-4 border rounded-lg bg-muted/50 space-y-3">
                                                        <p className="font-medium text-sm text-muted-foreground">Child {idx + 1}</p>
                                                        <input
                                                            type="text"
                                                            placeholder="Full Name"
                                                            value={child.name}
                                                            onChange={(e) => updateChild(idx, 'name', e.target.value)}
                                                            className="w-full px-3 py-2 border rounded-md"
                                                        />
                                                        <select
                                                            value={child.meal}
                                                            onChange={(e) => updateChild(idx, 'meal', e.target.value)}
                                                            className="w-full px-3 py-2 border rounded-md"
                                                        >
                                                            <option value="">Select Meal</option>
                                                            {currentEvent?.mealOptions?.map(meal => (
                                                                <option key={meal.code} value={meal.code}>
                                                                    {meal.name}
                                                                </option>
                                                            ))}
                                                        </select>
                                                        <input
                                                            type="text"
                                                            placeholder="Dietary Restrictions"
                                                            value={child.dietary}
                                                            onChange={(e) => updateChild(idx, 'dietary', e.target.value)}
                                                            className="w-full px-3 py-2 border rounded-md"
                                                        />
                                                    </div>
                                                ))}
                                            </div>
                                        </div>
                                    )}

                                    {/* Save/Cancel Buttons */}
                                    <div className="flex gap-2 justify-end pt-4">
                                        <Button
                                            variant="outline"
                                            onClick={() => setEditingDetails(false)}
                                        >
                                            Cancel
                                        </Button>
                                        <Button onClick={saveGuestDetails}>
                                            Save Guest Details
                                        </Button>
                                    </div>
                                </div>
                            )}

                            {/* View Mode - Adults Section */}
                            {!editingDetails && viewingDetails.guestDetails?.adults && viewingDetails.guestDetails.adults.length > 0 && (
                                <div>
                                    <h3 className="text-lg font-semibold mb-3 flex items-center gap-2">
                                        <Users className="h-5 w-5" />
                                        Adults ({viewingDetails.guestDetails.adults.length})
                                    </h3>
                                    <div className="space-y-3">
                                        {viewingDetails.guestDetails.adults.map((adult, idx) => (
                                            <div key={idx} className="p-4 border rounded-lg bg-muted/50">
                                                <div className="flex items-start justify-between gap-4">
                                                    <div className="flex-1">
                                                        <p className="font-medium text-base mb-2">
                                                            {adult.name || `Adult ${idx + 1}`}
                                                        </p>
                                                        <div className="space-y-1 text-sm">
                                                            {adult.meal && (
                                                                <div className="flex items-center gap-2">
                                                                    <Utensils className="h-4 w-4 text-muted-foreground" />
                                                                    <span className="text-muted-foreground">Meal:</span>
                                                                    <Badge variant="secondary">
                                                                        {currentEvent?.mealOptions?.find(m => m.code === adult.meal)?.name || adult.meal}
                                                                    </Badge>
                                                                </div>
                                                            )}
                                                            {adult.dietary && (
                                                                <div className="flex items-start gap-2">
                                                                    <AlertCircle className="h-4 w-4 text-orange-600 mt-0.5" />
                                                                    <span className="text-muted-foreground">Dietary:</span>
                                                                    <span className="font-medium text-orange-600">{adult.dietary}</span>
                                                                </div>
                                                            )}
                                                            {!adult.meal && !adult.dietary && (
                                                                <p className="text-muted-foreground italic">No meal or dietary information</p>
                                                            )}
                                                        </div>
                                                    </div>
                                                </div>
                                            </div>
                                        ))}
                                    </div>
                                </div>
                            )}

                            {/* View Mode - Children Section */}
                            {!editingDetails && viewingDetails.guestDetails?.children && viewingDetails.guestDetails.children.length > 0 && (
                                <div>
                                    <h3 className="text-lg font-semibold mb-3 flex items-center gap-2">
                                        <Users className="h-5 w-5" />
                                        Children ({viewingDetails.guestDetails.children.length})
                                    </h3>
                                    <div className="space-y-3">
                                        {viewingDetails.guestDetails.children.map((child, idx) => (
                                            <div key={idx} className="p-4 border rounded-lg bg-muted/50">
                                                <div className="flex items-start justify-between gap-4">
                                                    <div className="flex-1">
                                                        <p className="font-medium text-base mb-2">
                                                            {child.name || `Child ${idx + 1}`}
                                                        </p>
                                                        <div className="space-y-1 text-sm">
                                                            {child.meal && (
                                                                <div className="flex items-center gap-2">
                                                                    <Utensils className="h-4 w-4 text-muted-foreground" />
                                                                    <span className="text-muted-foreground">Meal:</span>
                                                                    <Badge variant="secondary">
                                                                        {currentEvent?.mealOptions?.find(m => m.code === child.meal)?.name || child.meal}
                                                                    </Badge>
                                                                </div>
                                                            )}
                                                            {child.dietary && (
                                                                <div className="flex items-start gap-2">
                                                                    <AlertCircle className="h-4 w-4 text-orange-600 mt-0.5" />
                                                                    <span className="text-muted-foreground">Dietary:</span>
                                                                    <span className="font-medium text-orange-600">{child.dietary}</span>
                                                                </div>
                                                            )}
                                                            {!child.meal && !child.dietary && (
                                                                <p className="text-muted-foreground italic">No meal or dietary information</p>
                                                            )}
                                                        </div>
                                                    </div>
                                                </div>
                                            </div>
                                        ))}
                                    </div>
                                </div>
                            )}

                            {/* View Mode - Summary for RSVPs without detailed guest info */}
                            {!editingDetails && (!viewingDetails.guestDetails?.adults || viewingDetails.guestDetails.adults.length === 0) &&
                             (!viewingDetails.guestDetails?.children || viewingDetails.guestDetails.children.length === 0) && (
                                <div className="text-center py-8">
                                    <Users className="h-12 w-12 mx-auto mb-3 opacity-50 text-muted-foreground" />
                                    <h3 className="font-semibold text-lg mb-2">Guest Count Summary</h3>
                                    <div className="flex justify-center gap-6 mb-4">
                                        {viewingDetails.adultsCount > 0 && (
                                            <div className="text-center">
                                                <p className="text-2xl font-bold text-primary">{viewingDetails.adultsCount}</p>
                                                <p className="text-sm text-muted-foreground">Adults</p>
                                            </div>
                                        )}
                                        {viewingDetails.childrenCount > 0 && (
                                            <div className="text-center">
                                                <p className="text-2xl font-bold text-primary">{viewingDetails.childrenCount}</p>
                                                <p className="text-sm text-muted-foreground">Children</p>
                                            </div>
                                        )}
                                    </div>
                                    <p className="text-sm text-muted-foreground italic">
                                        Individual guest details not available for this RSVP.<br />
                                        This RSVP was submitted with total counts only.
                                    </p>
                                    {viewingDetails.dietary && (
                                        <div className="mt-4 p-3 border rounded-lg bg-muted/30 text-left">
                                            <p className="text-sm font-medium mb-1">Dietary Restrictions:</p>
                                            <p className="text-sm text-muted-foreground">{viewingDetails.dietary}</p>
                                        </div>
                                    )}
                                </div>
                            )}
                        </div>
                    </DialogContent>
                </Dialog>
            )}

            {/* Bulk Send QR Tickets Modal */}
            {bulkSendModal && (
                <Dialog open={!!bulkSendModal} onOpenChange={() => setBulkSendModal(null)}>
                    <DialogContent className="max-w-2xl max-h-[80vh] overflow-hidden flex flex-col">
                        <DialogHeader>
                            <DialogTitle>Send QR Code Tickets</DialogTitle>
                            <DialogDescription>
                                Send QR code tickets to {bulkSendModal.recipients.length} confirmed household{bulkSendModal.recipients.length !== 1 ? 's' : ''} for {currentEvent?.name}
                            </DialogDescription>
                        </DialogHeader>

                        <div className="flex-1 overflow-y-auto border rounded-lg">
                            <Table>
                                <TableHeader>
                                    <TableRow>
                                        <TableHead>Household</TableHead>
                                        <TableHead>Email</TableHead>
                                        <TableHead>Adults</TableHead>
                                        <TableHead>Children</TableHead>
                                    </TableRow>
                                </TableHeader>
                                <TableBody>
                                    {bulkSendModal.recipients.map((rsvp, idx) => (
                                        <TableRow key={idx}>
                                            <TableCell className="font-medium">{rsvp.householdName}</TableCell>
                                            <TableCell className="text-sm">{rsvp.householdEmail || 'No email'}</TableCell>
                                            <TableCell><Badge variant="outline">{rsvp.adultsCount || 0}</Badge></TableCell>
                                            <TableCell><Badge variant="outline">{rsvp.childrenCount || 0}</Badge></TableCell>
                                        </TableRow>
                                    ))}
                                </TableBody>
                            </Table>
                        </div>

                        <div className="flex flex-col gap-4 pt-4 border-t">
                            <div className="text-sm text-muted-foreground">
                                <strong>{bulkSendModal.recipients.length}</strong> email{bulkSendModal.recipients.length !== 1 ? 's' : ''} will be sent
                            </div>
                            <div className="flex gap-2">
                                <Button 
                                    variant="outline" 
                                    onClick={() => setBulkSendModal(null)}
                                    disabled={sending}
                                >
                                    Cancel
                                </Button>
                                <Button 
                                    onClick={async () => {
                                        if (!selectedEvent) {
                                            toast.error('Please select an event');
                                            return;
                                        }
                                        
                                        // Filter out households without emails
                                        const validRecipients = bulkSendModal.recipients.filter(r => r.householdEmail);
                                        const skipped = bulkSendModal.recipients.length - validRecipients.length;
                                        
                                        if (validRecipients.length === 0) {
                                            toast.error('No valid email addresses found');
                                            return;
                                        }
                                        
                                        if (skipped > 0) {
                                            toast.warning(`${skipped} household${skipped !== 1 ? 's' : ''} will be skipped (no email address)`);
                                        }
                                        
                                        setSending(true);
                                        try {
                                            const response = await fetch('/api/email/send-tickets', {
                                                method: 'POST',
                                                headers: { 'Content-Type': 'application/json' },
                                                body: JSON.stringify({
                                                    recipients: validRecipients.map(r => ({
                                                        id: r.householdId,
                                                        email: r.householdEmail,
                                                        name: r.householdName
                                                    })),
                                                    eventId: selectedEvent
                                                })
                                            });
                                            
                                            const result = await response.json();
                                            
                                            if (response.ok) {
                                                if (result.failed > 0) {
                                                    const errorMsg = result.errors && result.errors.length > 0 
                                                        ? result.errors[0].error 
                                                        : 'Unknown error';
                                                    toast.error(`Sent ${result.sent} tickets, but ${result.failed} failed: ${errorMsg}`);
                                                } else {
                                                    toast.success(`Successfully sent ${result.sent} QR tickets!`);
                                                }
                                                if (result.sent > 0) {
                                                    setBulkSendModal(null);
                                                    setSelectedIds([]); // Clear selection
                                                }
                                            } else {
                                                toast.error(result.message || 'Failed to send tickets. Please try again.');
                                            }
                                        } catch (error) {
                                            toast.error('Failed to send tickets: ' + error.message);
                                        }
                                        setSending(false);
                                    }}
                                    disabled={sending}
                                >
                                    {sending ? (
                                        <>
                                            <RefreshCw className="h-4 w-4 mr-2 animate-spin" />
                                            Sending...
                                        </>
                                    ) : (
                                        <>
                                            <Send className="h-4 w-4 mr-2" />
                                            Send Tickets
                                        </>
                                    )}
                                </Button>
                            </div>
                        </div>
                    </DialogContent>
                </Dialog>
            )}
        </div>
    );
}
