import React, { useState, useEffect } from 'react';
import { Button } from '@/components/ui/button';
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from '@/components/ui/card';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import { Badge } from '@/components/ui/badge';
import { Dialog, DialogContent, DialogDescription, DialogHeader, DialogTitle } from '@/components/ui/dialog';
import { Checkbox } from '@/components/ui/checkbox';
import { FileText, Eye, Edit, Plus } from 'lucide-react';

export default function Pages() {
    const [pages, setPages] = useState([]);
    const [editingPage, setEditingPage] = useState(null);
    const [loading, setLoading] = useState(true);

    useEffect(() => {
        loadPages();
    }, []);

    const loadPages = () => {
        fetch('/api/pages')
            .then(res => res.json())
            .then(data => {
                setPages(data);
                setLoading(false);
            });
    };

    const savePage = async (page) => {
        const endpoint = page.id ? `/api/pages/${page.id}` : '/api/pages';
        const method = page.id ? 'PATCH' : 'POST';

        await fetch(endpoint, {
            method,
            headers: { 'Content-Type': 'application/json' },
            body: JSON.stringify(page)
        });

        loadPages();
        setEditingPage(null);
    };

    const defaultPages = [
        { slug: 'home', title: 'Home', description: 'Main landing page' },
        { slug: 'schedule', title: 'Schedule', description: 'Event timeline and schedule' },
        { slug: 'travel', title: 'Travel', description: 'Accommodation and directions' },
        { slug: 'faq', title: 'FAQ', description: 'Frequently asked questions' },
        { slug: 'our-story', title: 'Our Story', description: 'How you met' },
    ];

    if (loading) return <div className="flex items-center justify-center h-64">Loading...</div>;

    return (
        <div className="space-y-6">
            <div className="flex flex-col gap-4 sm:flex-row sm:items-center sm:justify-between">
                <div>
                    <h1 className="text-3xl font-bold tracking-tight">Website Pages</h1>
                    <p className="text-muted-foreground">Manage public-facing content</p>
                </div>
                <Button onClick={() => setEditingPage({ slug: '', title: '', body: '', published: true })}>
                    <Plus className="h-4 w-4 mr-2" />
                    Add Page
                </Button>
            </div>

            <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-4">
                {defaultPages.map(template => {
                    const existingPage = pages.find(p => p.slug === template.slug);
                    const page = existingPage || { ...template, body: '', published: false };

                    return (
                        <Card key={template.slug}>
                            <CardHeader>
                                <div className="flex justify-between items-start">
                                    <FileText className="h-5 w-5 text-muted-foreground" />
                                    {page.published && (
                                        <Badge variant="default" className="bg-green-500">Published</Badge>
                                    )}
                                </div>
                                <CardTitle>{template.title}</CardTitle>
                                <CardDescription>{template.description}</CardDescription>
                            </CardHeader>
                            <CardContent>
                                <div className="flex gap-2">
                                    <Button 
                                        variant="outline" 
                                        size="sm"
                                        className="flex-1"
                                        onClick={() => setEditingPage(page)}
                                    >
                                        <Edit className="h-4 w-4 mr-2" />
                                        Edit
                                    </Button>
                                    <Button 
                                        variant="outline" 
                                        size="sm"
                                        onClick={() => window.open(`/${template.slug === 'home' ? '' : template.slug}`, '_blank')}
                                    >
                                        <Eye className="h-4 w-4" />
                                    </Button>
                                </div>
                            </CardContent>
                        </Card>
                    );
                })}
            </div>

            {/* Edit Modal */}
            {editingPage && (
                <Dialog open={!!editingPage} onOpenChange={(open) => !open && setEditingPage(null)}>
                    <DialogContent className="max-w-4xl max-h-[90vh] overflow-y-auto">
                        <DialogHeader>
                            <DialogTitle>Edit Page</DialogTitle>
                            <DialogDescription>Update page content and settings</DialogDescription>
                        </DialogHeader>
                        
                        <div className="space-y-4 mt-4">
                            <div className="space-y-2">
                                <Label htmlFor="slug">Slug</Label>
                                <Input
                                    id="slug"
                                    value={editingPage.slug}
                                    onChange={(e) => setEditingPage({...editingPage, slug: e.target.value})}
                                    placeholder="page-url-slug"
                                />
                            </div>

                            <div className="space-y-2">
                                <Label htmlFor="title">Title</Label>
                                <Input
                                    id="title"
                                    value={editingPage.title}
                                    onChange={(e) => setEditingPage({...editingPage, title: e.target.value})}
                                    placeholder="Page Title"
                                />
                            </div>

                            <div className="space-y-2">
                                <Label htmlFor="body">Content</Label>
                                <textarea
                                    id="body"
                                    value={editingPage.body}
                                    onChange={(e) => setEditingPage({...editingPage, body: e.target.value})}
                                    className="flex min-h-[300px] w-full rounded-md border border-input bg-background px-3 py-2 text-sm ring-offset-background placeholder:text-muted-foreground focus-visible:outline-none focus-visible:ring-2 focus-visible:ring-ring font-mono"
                                    rows="15"
                                    placeholder="Write your content here. Supports plain text and basic HTML."
                                />
                                <p className="text-xs text-muted-foreground">
                                    Tip: Use line breaks for paragraphs. HTML tags are supported.
                                </p>
                            </div>

                            <div className="flex items-center gap-2">
                                <Checkbox
                                    id="published"
                                    checked={editingPage.published}
                                    onCheckedChange={(checked) => setEditingPage({...editingPage, published: checked})}
                                />
                                <Label htmlFor="published" className="text-sm font-normal cursor-pointer">
                                    Published (visible to guests)
                                </Label>
                            </div>
                        </div>

                        <div className="flex gap-2 mt-6">
                            <Button onClick={() => savePage(editingPage)} className="flex-1">
                                Save Page
                            </Button>
                            <Button variant="outline" onClick={() => setEditingPage(null)}>
                                Cancel
                            </Button>
                        </div>
                    </DialogContent>
                </Dialog>
            )}
        </div>
    );
}
