import React, { useState, useEffect } from 'react';
import { Button } from '@/components/ui/button';
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from '@/components/ui/card';
import { Input } from '@/components/ui/input';
import { Badge } from '@/components/ui/badge';
import { Table, TableBody, TableCell, TableHead, TableHeader, TableRow } from '@/components/ui/table';
import { DropdownMenu, DropdownMenuContent, DropdownMenuItem, DropdownMenuLabel, DropdownMenuSeparator, DropdownMenuTrigger } from '@/components/ui/dropdown-menu';
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from '@/components/ui/select';
import { Dialog, DialogContent, DialogDescription, DialogHeader, DialogTitle } from '@/components/ui/dialog';
import { Label } from '@/components/ui/label';
import { Skeleton } from '@/components/ui/skeleton';
import { Checkbox } from '@/components/ui/checkbox';
import { Users, Mail, Eye, Link, MoreVertical, Plus, Upload, Download, RefreshCw, Edit, Trash2, Send, ChevronLeft, ChevronRight, ChevronsLeft, ChevronsRight, QrCode } from 'lucide-react';
import { toast } from 'sonner';

export default function Households() {
    const [households, setHouseholds] = useState([]);
    const [events, setEvents] = useState([]);
    const [loading, setLoading] = useState(true);
    const [searchTerm, setSearchTerm] = useState('');
    const [filterSide, setFilterSide] = useState('all');
    const [showAddModal, setShowAddModal] = useState(false);
    const [showImportModal, setShowImportModal] = useState(false);
    const [editingHousehold, setEditingHousehold] = useState(null);
    const [bulkSendModal, setBulkSendModal] = useState(null); // { type: 'invite' | 'reminder', recipients: [], eventId: null }
    const [sending, setSending] = useState(false);
    const [selectedIds, setSelectedIds] = useState([]);
    const [currentPage, setCurrentPage] = useState(1);
    const [itemsPerPage, setItemsPerPage] = useState(10);
    const [qrModalHousehold, setQrModalHousehold] = useState(null);

    useEffect(() => {
        loadHouseholds();
        loadEvents();
    }, []);

    const loadHouseholds = () => {
        fetch('/api/households')
            .then(res => res.json())
            .then(data => {
                setHouseholds(data);
                setLoading(false);
            });
    };

    const loadEvents = () => {
        fetch('/api/events')
            .then(res => res.json())
            .then(data => {
                setEvents(data);
            });
    };

    const filteredHouseholds = households.filter(h => {
        const matchesSearch = h.displayName.toLowerCase().includes(searchTerm.toLowerCase()) ||
                            h.email?.toLowerCase().includes(searchTerm.toLowerCase());
        const matchesSide = filterSide === 'all' || h.side === filterSide;
        return matchesSearch && matchesSide;
    });

    // Reset to page 1 when search or filter changes
    useEffect(() => {
        setCurrentPage(1);
    }, [searchTerm, filterSide]);

    // Calculate pagination
    const totalPages = Math.ceil(filteredHouseholds.length / itemsPerPage);
    const startIndex = (currentPage - 1) * itemsPerPage;
    const endIndex = startIndex + itemsPerPage;
    const paginatedHouseholds = filteredHouseholds.slice(startIndex, endIndex);

    const copyInviteLink = (token) => {
        const url = `${window.location.origin}/invite/${token}`;
        navigator.clipboard.writeText(url);
        toast.success('Invite link copied to clipboard!');
    };

    const sendInviteEmail = async (householdId) => {
        // This would integrate with your email service
        toast.info('Email sending feature - integrate with your SMTP service');
    };

    const regenerateToken = async (id) => {
        if (!confirm('⚠️ Regenerate invite token? The old link will stop working.')) return;
        
        await fetch(`/api/households/${id}/regen-token`, { method: 'POST' });
        loadHouseholds();
        toast.success('Token regenerated successfully!');
    };

    const deleteHousehold = async (id, name) => {
        if (!confirm(`⚠️ Delete "${name}"? This will remove all associated guests and RSVPs!`)) return;
        
        await fetch(`/api/households/${id}`, { method: 'DELETE' });
        loadHouseholds();
    };

    const toggleSelectAll = () => {
        const pageIds = paginatedHouseholds.map(h => h.id);
        const allPageSelected = pageIds.every(id => selectedIds.includes(id));
        
        if (allPageSelected) {
            // Deselect all on current page
            setSelectedIds(selectedIds.filter(id => !pageIds.includes(id)));
        } else {
            // Select all on current page
            setSelectedIds([...new Set([...selectedIds, ...pageIds])]);
        }
    };

    const toggleSelect = (id) => {
        if (selectedIds.includes(id)) {
            setSelectedIds(selectedIds.filter(selectedId => selectedId !== id));
        } else {
            setSelectedIds([...selectedIds, id]);
        }
    };

    const selectedHouseholds = households.filter(h => selectedIds.includes(h.id));

    const stats = {
        total: households.length,
        invited: households.filter(h => h.invitedAt).length,
        opened: households.filter(h => h.openedAt).length,
        bride: households.filter(h => h.side === 'BRIDE').length,
        groom: households.filter(h => h.side === 'GROOM').length,
    };

    if (loading) {
        return (
            <div className="space-y-6">
                <Skeleton className="h-10 w-64" />
                <div className="grid grid-cols-2 md:grid-cols-5 gap-4">
                    {[...Array(5)].map((_, i) => (
                        <Card key={i}>
                            <CardHeader className="pb-2">
                                <Skeleton className="h-4 w-20" />
                            </CardHeader>
                            <CardContent>
                                <Skeleton className="h-8 w-16" />
                            </CardContent>
                        </Card>
                    ))}
                </div>
                <Skeleton className="h-96 w-full" />
            </div>
        );
    }

    return (
        <div className="space-y-6">
            {/* Header */}
            <div className="flex flex-col gap-4 sm:flex-row sm:items-center sm:justify-between">
                <div>
                    <h1 className="text-3xl font-bold tracking-tight">Households</h1>
                    <p className="text-muted-foreground">Manage guest households and invite links</p>
                </div>
                <div className="flex gap-2 flex-wrap">
                    <Button variant="outline" size="sm" onClick={() => setShowImportModal(true)}>
                        <Upload className="h-4 w-4 mr-2" />
                        Import
                    </Button>
                    <Button variant="outline" size="sm" onClick={() => window.location.href = '/api/export/households'}>
                        <Download className="h-4 w-4 mr-2" />
                        Export
                    </Button>
                    <Button size="sm" onClick={() => setShowAddModal(true)}>
                        <Plus className="h-4 w-4 mr-2" />
                        Add Household
                    </Button>
                </div>
            </div>

            {/* Stats Cards */}
            <div className="grid gap-4 grid-cols-2 sm:grid-cols-3 lg:grid-cols-5">
                <Card>
                    <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                        <CardTitle className="text-sm font-medium">Total</CardTitle>
                        <Users className="h-4 w-4 text-muted-foreground" />
                    </CardHeader>
                    <CardContent>
                        <div className="text-2xl font-bold">{stats.total}</div>
                    </CardContent>
                </Card>
                <Card>
                    <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                        <CardTitle className="text-sm font-medium">Invited</CardTitle>
                        <Send className="h-4 w-4 text-blue-600" />
                    </CardHeader>
                    <CardContent>
                        <div className="text-2xl font-bold text-blue-600">{stats.invited}</div>
                    </CardContent>
                </Card>
                <Card>
                    <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                        <CardTitle className="text-sm font-medium">Opened</CardTitle>
                        <Eye className="h-4 w-4 text-green-600" />
                    </CardHeader>
                    <CardContent>
                        <div className="text-2xl font-bold text-green-600">{stats.opened}</div>
                    </CardContent>
                </Card>
                <Card>
                    <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                        <CardTitle className="text-sm font-medium">Bride's Side</CardTitle>
                    </CardHeader>
                    <CardContent>
                        <div className="text-2xl font-bold">{stats.bride}</div>
                    </CardContent>
                </Card>
                <Card>
                    <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                        <CardTitle className="text-sm font-medium">Groom's Side</CardTitle>
                    </CardHeader>
                    <CardContent>
                        <div className="text-2xl font-bold">{stats.groom}</div>
                    </CardContent>
                </Card>
            </div>

            {/* Filters */}
            <Card>
                <CardContent className="pt-6">
                    <div className="flex flex-col sm:flex-row gap-4">
                        <div className="flex-1">
                            <Input
                                placeholder="Search households..."
                                value={searchTerm}
                                onChange={(e) => setSearchTerm(e.target.value)}
                                className="max-w-sm"
                            />
                        </div>
                        <Select value={filterSide} onValueChange={setFilterSide}>
                            <SelectTrigger className="w-full sm:w-[180px]">
                                <SelectValue />
                            </SelectTrigger>
                            <SelectContent>
                                <SelectItem value="all">All Sides</SelectItem>
                                <SelectItem value="BRIDE">Bride's Side</SelectItem>
                                <SelectItem value="GROOM">Groom's Side</SelectItem>
                            </SelectContent>
                        </Select>
                    </div>
                </CardContent>
            </Card>

            {/* Bulk Actions */}
            <Card className={selectedIds.length > 0 ? "border-primary" : ""}>
                <CardContent className="pt-6">
                    <div className="flex flex-col sm:flex-row items-start sm:items-center justify-between gap-4">
                        <div className="flex items-center gap-2">
                            {selectedIds.length > 0 ? (
                                <>
                                    <Badge variant="secondary" className="text-sm">
                                        {selectedIds.length} selected
                                    </Badge>
                                    <Button 
                                        variant="ghost" 
                                        size="sm"
                                        onClick={() => setSelectedIds([])}
                                    >
                                        Clear selection
                                    </Button>
                                </>
                            ) : (
                                <p className="text-sm text-muted-foreground">
                                    Select households to send invites or reminders
                                </p>
                            )}
                        </div>
                        <div className="flex gap-2 flex-wrap">
                            <Button 
                                variant="default" 
                                size="sm"
                                disabled={selectedIds.length === 0}
                                onClick={() => {
                                    setBulkSendModal({ type: 'invite', recipients: selectedHouseholds, eventId: events[0]?.id || null });
                                }}
                            >
                                <Send className="h-4 w-4 mr-2" />
                                Send Invites {selectedIds.length > 0 && `(${selectedIds.length})`}
                            </Button>
                            <Button 
                                variant="outline" 
                                size="sm"
                                disabled={selectedIds.length === 0}
                                onClick={() => {
                                    setBulkSendModal({ type: 'reminder', recipients: selectedHouseholds, eventId: events[0]?.id || null });
                                }}
                            >
                                <Mail className="h-4 w-4 mr-2" />
                                Send Reminders {selectedIds.length > 0 && `(${selectedIds.length})`}
                            </Button>
                        </div>
                    </div>
                </CardContent>
            </Card>

            {/* Households Table */}
            <Card>
                <Table>
                    <TableHeader>
                        <TableRow>
                            <TableHead className="w-[50px]">
                                <Checkbox
                                    checked={paginatedHouseholds.length > 0 && paginatedHouseholds.every(h => selectedIds.includes(h.id))}
                                    onCheckedChange={toggleSelectAll}
                                />
                            </TableHead>
                            <TableHead>Household</TableHead>
                            <TableHead>Contact</TableHead>
                            <TableHead className="w-[80px]">Adults</TableHead>
                            <TableHead className="w-[80px]">Children</TableHead>
                            <TableHead className="w-[120px]">Side</TableHead>
                            <TableHead className="w-[120px]">Status</TableHead>
                            <TableHead className="w-[80px] text-right">Actions</TableHead>
                        </TableRow>
                    </TableHeader>
                    <TableBody>
                        {paginatedHouseholds.length === 0 ? (
                            <TableRow>
                                <TableCell colSpan={8} className="h-24 text-center">
                                    <div className="flex flex-col items-center gap-2 text-muted-foreground">
                                        <Users className="h-8 w-8 opacity-50" />
                                        <p>{filteredHouseholds.length === 0 ? 'No households found.' : 'No results on this page.'}</p>
                                        {filteredHouseholds.length === 0 && (
                                            <Button variant="outline" size="sm" onClick={() => setShowAddModal(true)}>
                                                <Plus className="h-4 w-4 mr-2" />
                                                Add Household
                                            </Button>
                                        )}
                                    </div>
                                </TableCell>
                            </TableRow>
                        ) : (
                            paginatedHouseholds.map(household => (
                                <TableRow key={household.id}>
                                    <TableCell>
                                        <Checkbox
                                            checked={selectedIds.includes(household.id)}
                                            onCheckedChange={() => toggleSelect(household.id)}
                                        />
                                    </TableCell>
                                    <TableCell>
                                        <div className="font-medium">{household.displayName}</div>
                                        {household.tags && (
                                            <div className="flex gap-1 mt-1 flex-wrap">
                                                {household.tags.split(',').map(tag => (
                                                    <Badge key={tag} variant="secondary" className="text-xs">
                                                        {tag.trim()}
                                                    </Badge>
                                                ))}
                                            </div>
                                        )}
                                    </TableCell>
                                    <TableCell className="text-sm">
                                        {household.email && (
                                            <div className="flex items-center gap-1 text-muted-foreground">
                                                <Mail className="h-3 w-3" />
                                                {household.email}
                                            </div>
                                        )}
                                        {household.phone && (
                                            <div className="text-muted-foreground mt-1">{household.phone}</div>
                                        )}
                                    </TableCell>
                                    <TableCell>
                                        <Badge variant="outline">{household.adultsCount || 0}</Badge>
                                    </TableCell>
                                    <TableCell>
                                        <Badge variant="outline">{household.childrenCount || 0}</Badge>
                                    </TableCell>
                                    <TableCell>
                                        <Badge variant={household.side === 'BRIDE' ? 'default' : 'secondary'}>
                                            {household.side}
                                        </Badge>
                                    </TableCell>
                                    <TableCell>
                                        {household.openedAt ? (
                                            <Badge variant="default" className="bg-green-500">
                                                <Eye className="h-3 w-3 mr-1" />
                                                Opened
                                            </Badge>
                                        ) : household.invitedAt ? (
                                            <Badge variant="secondary">
                                                <Send className="h-3 w-3 mr-1" />
                                                Invited
                                            </Badge>
                                        ) : (
                                            <span className="text-sm text-muted-foreground">Not sent</span>
                                        )}
                                    </TableCell>
                                    <TableCell className="text-right">
                                        <DropdownMenu>
                                            <DropdownMenuTrigger asChild>
                                                <Button variant="ghost" size="sm">
                                                    <MoreVertical className="h-4 w-4" />
                                                </Button>
                                            </DropdownMenuTrigger>
                                            <DropdownMenuContent align="end" className="w-[200px]">
                                                <DropdownMenuLabel>Actions</DropdownMenuLabel>
                                                <DropdownMenuSeparator />
                                                <DropdownMenuItem onClick={() => copyInviteLink(household.token)}>
                                                    <Link className="h-4 w-4 mr-2" />
                                                    Copy Invite Link
                                                </DropdownMenuItem>
                                                <DropdownMenuItem onClick={() => window.open(`/invite/${household.token}`, '_blank')}>
                                                    <Eye className="h-4 w-4 mr-2" />
                                                    Preview Invite
                                                </DropdownMenuItem>
                                                <DropdownMenuItem onClick={() => setQrModalHousehold(household)}>
                                                    <QrCode className="h-4 w-4 mr-2" />
                                                    View QR Codes
                                                </DropdownMenuItem>
                                                <DropdownMenuItem 
                                                    onSelect={() => {
                                                        // Use requestAnimationFrame + setTimeout for smooth transition
                                                        requestAnimationFrame(() => {
                                                            setTimeout(() => {
                                                                setEditingHousehold(household);
                                                            }, 100);
                                                        });
                                                    }}
                                                >
                                                    <Edit className="h-4 w-4 mr-2" />
                                                    Edit
                                                </DropdownMenuItem>
                                                <DropdownMenuSeparator />
                                                <DropdownMenuItem onClick={() => regenerateToken(household.id)}>
                                                    <RefreshCw className="h-4 w-4 mr-2" />
                                                    Regenerate Token
                                                </DropdownMenuItem>
                                                <DropdownMenuItem 
                                                    onClick={() => deleteHousehold(household.id, household.displayName)}
                                                    className="text-red-600"
                                                >
                                                    <Trash2 className="h-4 w-4 mr-2" />
                                                    Delete
                                                </DropdownMenuItem>
                                            </DropdownMenuContent>
                                        </DropdownMenu>
                                    </TableCell>
                                </TableRow>
                            ))
                        )}
                    </TableBody>
                </Table>

                {/* Pagination */}
                {filteredHouseholds.length > 0 && (
                    <div className="flex items-center justify-between px-6 py-4 border-t">
                        <div className="flex items-center gap-4">
                            <p className="text-sm text-muted-foreground">
                                Showing {startIndex + 1} to {Math.min(endIndex, filteredHouseholds.length)} of {filteredHouseholds.length} households
                            </p>
                            <Select value={itemsPerPage.toString()} onValueChange={(value) => setItemsPerPage(Number(value))}>
                                <SelectTrigger className="w-[100px]">
                                    <SelectValue />
                                </SelectTrigger>
                                <SelectContent>
                                    <SelectItem value="10">10 / page</SelectItem>
                                    <SelectItem value="25">25 / page</SelectItem>
                                    <SelectItem value="50">50 / page</SelectItem>
                                    <SelectItem value="100">100 / page</SelectItem>
                                </SelectContent>
                            </Select>
                        </div>
                        
                        <div className="flex items-center gap-2">
                            <Button
                                variant="outline"
                                size="sm"
                                onClick={() => setCurrentPage(1)}
                                disabled={currentPage === 1}
                            >
                                <ChevronsLeft className="h-4 w-4" />
                            </Button>
                            <Button
                                variant="outline"
                                size="sm"
                                onClick={() => setCurrentPage(currentPage - 1)}
                                disabled={currentPage === 1}
                            >
                                <ChevronLeft className="h-4 w-4" />
                            </Button>
                            <span className="text-sm font-medium px-2">
                                Page {currentPage} of {totalPages}
                            </span>
                            <Button
                                variant="outline"
                                size="sm"
                                onClick={() => setCurrentPage(currentPage + 1)}
                                disabled={currentPage === totalPages}
                            >
                                <ChevronRight className="h-4 w-4" />
                            </Button>
                            <Button
                                variant="outline"
                                size="sm"
                                onClick={() => setCurrentPage(totalPages)}
                                disabled={currentPage === totalPages}
                            >
                                <ChevronsRight className="h-4 w-4" />
                            </Button>
                        </div>
                    </div>
                )}
            </Card>

            {/* Add/Edit Modal */}
            {(showAddModal || editingHousehold) && (
                <HouseholdModal
                    key={editingHousehold?.id || 'new'}
                    household={editingHousehold}
                    onClose={() => {
                        setShowAddModal(false);
                        setEditingHousehold(null);
                    }}
                    onSave={() => {
                        loadHouseholds();
                        setShowAddModal(false);
                        setEditingHousehold(null);
                    }}
                />
            )}

            {/* Import Modal */}
            {showImportModal && (
                <ImportModal
                    onClose={() => setShowImportModal(false)}
                    onImport={() => {
                        // Handle CSV import
                        setShowImportModal(false);
                        loadHouseholds();
                    }}
                />
            )}

            {/* QR Codes Modal */}
            {qrModalHousehold && (
                <QrCodesModal
                    household={qrModalHousehold}
                    events={events}
                    onClose={() => setQrModalHousehold(null)}
                />
            )}

            {/* Bulk Send Modal */}
            {bulkSendModal && (
                <Dialog open={!!bulkSendModal} onOpenChange={() => setBulkSendModal(null)}>
                    <DialogContent className="max-w-2xl max-h-[80vh] overflow-hidden flex flex-col">
                        <DialogHeader>
                            <DialogTitle>
                                {bulkSendModal.type === 'invite' ? 'Send Invitations' : 'Send Reminders'}
                            </DialogTitle>
                            <DialogDescription>
                                {bulkSendModal.type === 'invite' 
                                    ? `Send invitation emails to ${bulkSendModal.recipients.length} household${bulkSendModal.recipients.length !== 1 ? 's' : ''}`
                                    : `Send reminder emails to ${bulkSendModal.recipients.length} household${bulkSendModal.recipients.length !== 1 ? 's' : ''} that haven't responded`
                                }
                            </DialogDescription>
                        </DialogHeader>

                        <div className="space-y-2 py-4">
                            <Label htmlFor="eventSelect">Select Event *</Label>
                            <Select 
                                value={bulkSendModal.eventId?.toString() || ''} 
                                onValueChange={(value) => setBulkSendModal({...bulkSendModal, eventId: parseInt(value)})}
                            >
                                <SelectTrigger>
                                    <SelectValue placeholder="Select an event" />
                                </SelectTrigger>
                                <SelectContent>
                                    {events.map(event => {
                                        // Display couples' names if both are provided, otherwise use event name
                                        const displayName = event.nameOne && event.nameTwo 
                                            ? `${event.nameOne} ${event.andText || 'AND'} ${event.nameTwo}`
                                            : event.name;
                                        
                                        return (
                                            <SelectItem key={event.id} value={event.id.toString()}>
                                                {displayName}
                                            </SelectItem>
                                        );
                                    })}
                                </SelectContent>
                            </Select>
                        </div>

                        <div className="flex-1 overflow-y-auto border rounded-lg">
                            <Table>
                                <TableHeader>
                                    <TableRow>
                                        <TableHead>Household</TableHead>
                                        <TableHead>Email</TableHead>
                                        <TableHead>Adults</TableHead>
                                        <TableHead>Children</TableHead>
                                        <TableHead>Side</TableHead>
                                    </TableRow>
                                </TableHeader>
                                <TableBody>
                                    {bulkSendModal.recipients.map(household => (
                                        <TableRow key={household.id}>
                                            <TableCell className="font-medium">{household.displayName}</TableCell>
                                            <TableCell className="text-sm text-muted-foreground">
                                                {household.email || 'No email'}
                                            </TableCell>
                                            <TableCell>
                                                <Badge variant="outline">{household.adultsCount || 0}</Badge>
                                            </TableCell>
                                            <TableCell>
                                                <Badge variant="outline">{household.childrenCount || 0}</Badge>
                                            </TableCell>
                                            <TableCell>
                                                <Badge variant={household.side === 'BRIDE' ? 'default' : 'secondary'}>
                                                    {household.side}
                                                </Badge>
                                            </TableCell>
                                        </TableRow>
                                    ))}
                                </TableBody>
                            </Table>
                        </div>

                        <div className="flex items-center justify-between pt-4 border-t">
                            <div className="text-sm text-muted-foreground">
                                <strong>{bulkSendModal.recipients.length}</strong> email{bulkSendModal.recipients.length !== 1 ? 's' : ''} will be sent
                            </div>
                            <div className="flex gap-2">
                                <Button 
                                    variant="outline" 
                                    onClick={() => setBulkSendModal(null)}
                                    disabled={sending}
                                >
                                    Cancel
                                </Button>
                                <Button 
                                    onClick={async () => {
                                        if (!bulkSendModal.eventId) {
                                            toast.error('Please select an event');
                                            return;
                                        }
                                        setSending(true);
                                        try {
                                            const response = await fetch('/api/email/send-bulk', {
                                                method: 'POST',
                                                headers: { 'Content-Type': 'application/json' },
                                                body: JSON.stringify({
                                                    recipients: bulkSendModal.recipients.map(h => ({
                                                        id: h.id,
                                                        email: h.email,
                                                        name: h.displayName,
                                                        rsvpUrl: bulkSendModal.eventId 
                                                            ? `${window.location.origin}/invite/${h.token}?event=${bulkSendModal.eventId}`
                                                            : `${window.location.origin}/invite/${h.token}`
                                                    })),
                                                    isReminder: bulkSendModal.type === 'reminder',
                                                    eventId: bulkSendModal.eventId
                                                })
                                            });
                                            
                                            const result = await response.json();
                                            
                                            if (response.ok) {
                                                if (result.failed > 0) {
                                                    const errorMsg = result.errors && result.errors.length > 0 
                                                        ? result.errors[0].error 
                                                        : 'Unknown error';
                                                    toast.error(`Sent ${result.sent} emails, but ${result.failed} failed: ${errorMsg}`);
                                                } else {
                                                    toast.success(`Successfully sent ${result.sent} emails!`);
                                                }
                                                if (result.sent > 0) {
                                                    setBulkSendModal(null);
                                                    setSelectedIds([]); // Clear selection
                                                    loadHouseholds(); // Reload to update status
                                                }
                                            } else {
                                                toast.error(result.message || 'Failed to send emails. Please try again.');
                                            }
                                        } catch (error) {
                                            toast.error('Failed to send emails: ' + error.message);
                                        }
                                        setSending(false);
                                    }}
                                    disabled={sending || !bulkSendModal.eventId}
                                >
                                    {sending ? (
                                        <>
                                            <RefreshCw className="h-4 w-4 mr-2 animate-spin" />
                                            Sending...
                                        </>
                                    ) : (
                                        <>
                                            <Send className="h-4 w-4 mr-2" />
                                            Send {bulkSendModal.type === 'invite' ? 'Invitations' : 'Reminders'}
                                        </>
                                    )}
                                </Button>
                            </div>
                        </div>
                    </DialogContent>
                </Dialog>
            )}
        </div>
    );
}

// Household Add/Edit Modal Component
function HouseholdModal({ household, onClose, onSave }) {
    const [formData, setFormData] = useState({
        displayName: household?.displayName || '',
        email: household?.email || '',
        phone: household?.phone || '',
        address: household?.address || '',
        side: household?.side || 'OTHER',
        locale: household?.locale || 'en',
        adultsCount: household?.adultsCount || 1,
        childrenCount: household?.childrenCount || 0,
        notes: household?.notes || ''
    });
    const [saving, setSaving] = useState(false);

    const handleSubmit = async (e) => {
        e.preventDefault();
        setSaving(true);

        try {
            const url = household ? `/api/households/${household.id}` : '/api/households';
            const method = household ? 'PATCH' : 'POST';

            const response = await fetch(url, {
                method,
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify(formData)
            });

            if (response.ok) {
                onSave();
                toast.success(household ? 'Household updated!' : 'Household created!');
            } else {
                toast.error('Error saving household');
            }
        } catch (error) {
            toast.error('Error: ' + error.message);
        } finally {
            setSaving(false);
        }
    };

    return (
        <Dialog open={true} onOpenChange={(open) => { if (!open) onClose(); }} modal={true}>
            <DialogContent className="max-w-2xl max-h-[90vh] overflow-y-auto" onOpenAutoFocus={(e) => e.preventDefault()}>
                <DialogHeader>
                    <DialogTitle>{household ? 'Edit Household' : 'Add Household'}</DialogTitle>
                    <DialogDescription>
                        {household ? 'Update household information' : 'Create a new household and generate an invite link'}
                    </DialogDescription>
                </DialogHeader>

                <form onSubmit={handleSubmit} className="space-y-4">
                    <div className="space-y-2">
                        <Label htmlFor="displayName">Display Name *</Label>
                        <Input
                            id="displayName"
                            required
                            value={formData.displayName}
                            onChange={(e) => setFormData({...formData, displayName: e.target.value})}
                            placeholder="Smith Family"
                        />
                    </div>

                    <div className="grid grid-cols-2 gap-4">
                        <div className="space-y-2">
                            <Label htmlFor="email">Email</Label>
                            <Input
                                id="email"
                                type="email"
                                value={formData.email}
                                onChange={(e) => setFormData({...formData, email: e.target.value})}
                            />
                        </div>
                        <div className="space-y-2">
                            <Label htmlFor="phone">Phone</Label>
                            <Input
                                id="phone"
                                type="tel"
                                value={formData.phone}
                                onChange={(e) => setFormData({...formData, phone: e.target.value})}
                            />
                        </div>
                    </div>

                    <div className="space-y-2">
                        <Label htmlFor="address">Address</Label>
                        <textarea
                            id="address"
                            value={formData.address}
                            onChange={(e) => setFormData({...formData, address: e.target.value})}
                            className="flex min-h-[60px] w-full rounded-md border border-input bg-background px-3 py-2 text-sm ring-offset-background placeholder:text-muted-foreground focus-visible:outline-none focus-visible:ring-2 focus-visible:ring-ring focus-visible:ring-offset-2"
                            rows="2"
                        />
                    </div>

                    <div className="grid grid-cols-3 gap-4">
                        <div className="space-y-2">
                            <Label htmlFor="side">Side</Label>
                            <Select value={formData.side} onValueChange={(value) => setFormData({...formData, side: value})}>
                                <SelectTrigger id="side">
                                    <SelectValue />
                                </SelectTrigger>
                                <SelectContent>
                                    <SelectItem value="BRIDE">Bride</SelectItem>
                                    <SelectItem value="GROOM">Groom</SelectItem>
                                    <SelectItem value="OTHER">Other</SelectItem>
                                </SelectContent>
                            </Select>
                        </div>
                        <div className="space-y-2">
                            <Label htmlFor="adults">Adults</Label>
                            <Input
                                id="adults"
                                type="number"
                                min="0"
                                value={formData.adultsCount}
                                onChange={(e) => setFormData({...formData, adultsCount: parseInt(e.target.value) || 0})}
                            />
                        </div>
                        <div className="space-y-2">
                            <Label htmlFor="children">Children</Label>
                            <Input
                                id="children"
                                type="number"
                                min="0"
                                value={formData.childrenCount}
                                onChange={(e) => setFormData({...formData, childrenCount: parseInt(e.target.value) || 0})}
                            />
                        </div>
                    </div>

                    <div className="space-y-2">
                        <Label htmlFor="notes">Notes</Label>
                        <textarea
                            id="notes"
                            value={formData.notes}
                            onChange={(e) => setFormData({...formData, notes: e.target.value})}
                            className="flex min-h-[80px] w-full rounded-md border border-input bg-background px-3 py-2 text-sm ring-offset-background placeholder:text-muted-foreground focus-visible:outline-none focus-visible:ring-2 focus-visible:ring-ring focus-visible:ring-offset-2"
                            rows="3"
                            placeholder="Internal notes..."
                        />
                    </div>

                    <div className="flex justify-end gap-2 pt-4">
                        <Button type="button" variant="outline" onClick={onClose}>
                            Cancel
                        </Button>
                        <Button type="submit" disabled={saving}>
                            {saving ? 'Saving...' : (household ? 'Update' : 'Create')}
                        </Button>
                    </div>
                </form>
            </DialogContent>
        </Dialog>
    );
}

// Import Modal Component  
function ImportModal({ onClose, onImport }) {
    return (
        <Dialog open={true} onOpenChange={onClose}>
            <DialogContent className="max-w-md">
                <DialogHeader>
                    <DialogTitle>Import Households</DialogTitle>
                    <DialogDescription>
                        Upload a CSV file with household data to bulk import
                    </DialogDescription>
                </DialogHeader>
                <div className="space-y-4">
                    <div>
                        <Label htmlFor="csvFile">CSV File</Label>
                        <Input id="csvFile" type="file" accept=".csv" className="mt-2" />
                        <p className="text-xs text-muted-foreground mt-2">
                            Expected columns: displayName, email, phone, side, numberOfAdults, numberOfChildren
                        </p>
                    </div>
                    <div className="flex justify-end gap-2">
                        <Button variant="outline" onClick={onClose}>Cancel</Button>
                        <Button onClick={onImport}>
                            <Upload className="h-4 w-4 mr-2" />
                            Import
                        </Button>
                    </div>
                </div>
            </DialogContent>
        </Dialog>
    );
}

// QR Codes Modal Component
function QrCodesModal({ household, events, onClose }) {
    const copyQrLink = (householdId, eventId) => {
        const url = `${window.location.origin}/q/${householdId}-${eventId}`;
        navigator.clipboard.writeText(url);
        toast.success('QR code link copied!');
    };

    const downloadQrCode = (householdId, eventId, eventName) => {
        const qrUrl = `https://api.qrserver.com/v1/create-qr-code/?size=500x500&data=${encodeURIComponent(window.location.origin + '/q/' + householdId + '-' + eventId)}`;
        const link = document.createElement('a');
        link.href = qrUrl;
        link.download = `qr-${household.displayName.replace(/\s+/g, '-')}-${eventName.replace(/\s+/g, '-')}.png`;
        document.body.appendChild(link);
        link.click();
        document.body.removeChild(link);
        toast.success('QR code downloaded!');
    };

    const openTicketPage = (householdId, eventId) => {
        window.open(`/q/${householdId}-${eventId}`, '_blank');
    };

    return (
        <Dialog open={true} onOpenChange={onClose}>
            <DialogContent className="max-w-3xl max-h-[90vh] overflow-y-auto">
                <DialogHeader>
                    <DialogTitle>QR Code Tickets - {household.displayName}</DialogTitle>
                    <DialogDescription>
                        Generate QR code tickets for check-in at events
                    </DialogDescription>
                </DialogHeader>
                
                <div className="space-y-6 py-4">
                    {events.length === 0 ? (
                        <div className="text-center py-8 text-muted-foreground">
                            <QrCode className="h-12 w-12 mx-auto mb-4 opacity-50" />
                            <p>No events available. Create an event first.</p>
                        </div>
                    ) : (
                        events.map(event => {
                            const qrUrl = `https://api.qrserver.com/v1/create-qr-code/?size=300x300&data=${encodeURIComponent(window.location.origin + '/q/' + household.id + '-' + event.id)}`;
                            
                            return (
                                <Card key={event.id} className="overflow-hidden">
                                    <CardHeader className="bg-muted/30">
                                        <div className="flex items-center justify-between">
                                            <div>
                                                <CardTitle className="text-lg">{event.name}</CardTitle>
                                                {event.venueName && (
                                                    <CardDescription className="mt-1">
                                                        {event.venueName}
                                                        {event.startAt && ` • ${new Date(event.startAt).toLocaleDateString()}`}
                                                    </CardDescription>
                                                )}
                                            </div>
                                            <Badge>{event.isPublic ? 'Public' : 'Private'}</Badge>
                                        </div>
                                    </CardHeader>
                                    <CardContent className="pt-6">
                                        <div className="grid md:grid-cols-2 gap-6">
                                            {/* QR Code Display */}
                                            <div className="flex flex-col items-center">
                                                <div className="bg-white p-4 rounded-lg border-2 border-muted mb-4">
                                                    <img 
                                                        src={qrUrl} 
                                                        alt={`QR Code for ${event.name}`}
                                                        className="w-64 h-64"
                                                    />
                                                </div>
                                                <div className="text-sm text-muted-foreground text-center">
                                                    Scan this code at check-in
                                                </div>
                                            </div>

                                            {/* Actions */}
                                            <div className="flex flex-col justify-center space-y-3">
                                                <div className="space-y-2">
                                                    <Label className="text-xs text-muted-foreground">Ticket URL</Label>
                                                    <div className="flex gap-2">
                                                        <Input 
                                                            value={`${window.location.origin}/q/${household.id}-${event.id}`}
                                                            readOnly
                                                            className="font-mono text-xs"
                                                        />
                                                        <Button 
                                                            size="sm" 
                                                            variant="outline"
                                                            onClick={() => copyQrLink(household.id, event.id)}
                                                        >
                                                            <Link className="h-4 w-4" />
                                                        </Button>
                                                    </div>
                                                </div>

                                                <Button 
                                                    variant="default" 
                                                    className="w-full"
                                                    onClick={() => openTicketPage(household.id, event.id)}
                                                >
                                                    <Eye className="h-4 w-4 mr-2" />
                                                    View Full Ticket Page
                                                </Button>

                                                <Button 
                                                    variant="outline" 
                                                    className="w-full"
                                                    onClick={() => downloadQrCode(household.id, event.id, event.name)}
                                                >
                                                    <Download className="h-4 w-4 mr-2" />
                                                    Download QR Code
                                                </Button>

                                                <div className="pt-4 border-t space-y-2">
                                                    <div className="text-xs text-muted-foreground">
                                                        <strong>How to use:</strong>
                                                    </div>
                                                    <ul className="text-xs text-muted-foreground space-y-1 list-disc list-inside">
                                                        <li>Send ticket URL to guest via email</li>
                                                        <li>Guest opens on phone</li>
                                                        <li>Staff scans QR at check-in</li>
                                                        <li>Or download & print for physical tickets</li>
                                                    </ul>
                                                </div>
                                            </div>
                                        </div>
                                    </CardContent>
                                </Card>
                            );
                        })
                    )}
                </div>

                <div className="flex justify-end gap-2 pt-4 border-t">
                    <Button variant="outline" onClick={onClose}>
                        Close
                    </Button>
                </div>
            </DialogContent>
        </Dialog>
    );
}
