import { useState, useEffect, useRef } from 'react';
import { Html5Qrcode } from 'html5-qrcode';
import { Card, CardContent, CardHeader, CardTitle, CardDescription } from '../../components/ui/card';
import { Button } from '../../components/ui/button';
import { CheckCircle, XCircle, Camera, CameraOff } from 'lucide-react';

export default function CheckIn() {
    const [events, setEvents] = useState([]);
    const [selectedEvent, setSelectedEvent] = useState(null);
    const [scanning, setScanning] = useState(false);
    const [result, setResult] = useState(null);
    const [scanner, setScanner] = useState(null);
    const [stats, setStats] = useState({ total: 0, checkedIn: 0, pending: 0 });
    const [attendingGuests, setAttendingGuests] = useState([]);
    const [searchTerm, setSearchTerm] = useState('');
    const scannerRef = useRef(null);

    useEffect(() => {
        // Load events
        fetch('/api/events')
            .then(res => res.json())
            .then(data => {
                setEvents(data);
                if (data.length > 0) {
                    setSelectedEvent(data[0].id);
                }
            });
    }, []);

    useEffect(() => {
        if (selectedEvent) {
            loadStats();
        }
    }, [selectedEvent]);

    useEffect(() => {
        return () => {
            if (scanner) {
                scanner.stop().catch(err => console.error(err));
            }
        };
    }, [scanner]);

    const loadStats = async () => {
        const res = await fetch(`/api/rsvps/event/${selectedEvent}`);
        const rsvps = await res.json();
        const attending = rsvps.filter(r => r.status === 'attending');
        const checkedIn = attending.filter(r => r.checkedIn);
        setStats({
            total: attending.length,
            checkedIn: checkedIn.length,
            pending: attending.length - checkedIn.length
        });
        setAttendingGuests(attending);
    };

    const startScanner = async () => {
        try {
            const html5QrCode = new Html5Qrcode("qr-reader");
            setScanner(html5QrCode);

            await html5QrCode.start(
                { facingMode: "environment" },
                {
                    fps: 10,
                    qrbox: { width: 250, height: 250 }
                },
                (decodedText) => {
                    // Extract IDs from URL: /q/11-2
                    const match = decodedText.match(/\/q\/(\d+)-(\d+)/);
                    if (match) {
                        const householdId = match[1];
                        const eventId = match[2];
                        
                        // Check if scanned QR is for selected event
                        if (parseInt(eventId) === selectedEvent) {
                            checkIn(householdId, eventId);
                        } else {
                            setResult({
                                success: false,
                                error: 'This QR code is for a different event!'
                            });
                            setTimeout(() => setResult(null), 3000);
                        }
                    } else {
                        setResult({
                            success: false,
                            error: 'Invalid QR code format'
                        });
                        setTimeout(() => setResult(null), 3000);
                    }
                },
                (errorMessage) => {
                    // Scanning errors - can be ignored
                }
            );

            setScanning(true);
        } catch (err) {
            console.error('Failed to start scanner:', err);
            alert('Failed to start camera. Please check camera permissions.');
        }
    };

    const stopScanner = async () => {
        if (scanner) {
            await scanner.stop();
            setScanner(null);
            setScanning(false);
        }
    };

    const checkIn = async (householdId, eventId) => {
        try {
            const res = await fetch(`/api/qr-checkin/${householdId}/${eventId}`, {
                method: 'POST'
            });

            const data = await res.json();
            
            if (res.ok) {
                setResult(data);
                loadStats(); // Refresh stats
                
                // Auto-clear after 3 seconds
                setTimeout(() => setResult(null), 3000);
                
                // Play success sound
                playSound();
            } else {
                setResult({
                    success: false,
                    error: data.error || 'Check-in failed'
                });
                setTimeout(() => setResult(null), 3000);
            }
        } catch (err) {
            console.error('Check-in error:', err);
            setResult({
                success: false,
                error: 'Network error. Please try again.'
            });
            setTimeout(() => setResult(null), 3000);
        }
    };

    const playSound = () => {
        // Create a simple beep sound
        const audioContext = new (window.AudioContext || window.webkitAudioContext)();
        const oscillator = audioContext.createOscillator();
        const gainNode = audioContext.createGain();
        
        oscillator.connect(gainNode);
        gainNode.connect(audioContext.destination);
        
        oscillator.frequency.value = 800;
        oscillator.type = 'sine';
        gainNode.gain.value = 0.3;
        
        oscillator.start(audioContext.currentTime);
        oscillator.stop(audioContext.currentTime + 0.1);
    };

    const toggleManualCheckIn = async (householdId, currentlyCheckedIn) => {
        try {
            const endpoint = currentlyCheckedIn 
                ? `/api/qr-checkin/undo/${householdId}/${selectedEvent}`
                : `/api/qr-checkin/${householdId}/${selectedEvent}`;
            
            const res = await fetch(endpoint, { method: 'POST' });
            const data = await res.json();
            
            if (res.ok) {
                loadStats(); // Refresh the list
                playSound();
            }
        } catch (err) {
            console.error('Manual check-in error:', err);
        }
    };

    const filteredGuests = attendingGuests.filter(guest => {
        if (!searchTerm) return true;
        const search = searchTerm.toLowerCase();
        return guest.householdName?.toLowerCase().includes(search) ||
               guest.householdEmail?.toLowerCase().includes(search);
    });

    return (
        <div className="p-8 max-w-4xl mx-auto">
            <div className="mb-6">
                <h1 className="text-3xl font-bold mb-2">Event Check-In</h1>
                <p className="text-muted-foreground">Scan QR code tickets to check in guests</p>
            </div>

            {/* Event Selection */}
            <Card className="mb-6">
                <CardHeader>
                    <CardTitle>Select Event</CardTitle>
                </CardHeader>
                <CardContent>
                    <select
                        value={selectedEvent || ''}
                        onChange={(e) => setSelectedEvent(parseInt(e.target.value))}
                        className="w-full p-2 border rounded"
                        disabled={scanning}
                    >
                        {events.map(event => {
                            const eventName = event.nameOne && event.nameTwo 
                                ? `${event.nameOne} ${event.andText || 'AND'} ${event.nameTwo}`
                                : event.name;
                            return (
                                <option key={event.id} value={event.id}>
                                    {eventName} - {new Date(event.startAt).toLocaleDateString()}
                                </option>
                            );
                        })}
                    </select>
                </CardContent>
            </Card>

            {/* Stats */}
            {selectedEvent && (
                <div className="grid grid-cols-3 gap-4 mb-6">
                    <Card>
                        <CardContent className="pt-6">
                            <div className="text-center">
                                <div className="text-3xl font-bold">{stats.total}</div>
                                <div className="text-sm text-muted-foreground">Total Expected</div>
                            </div>
                        </CardContent>
                    </Card>
                    <Card>
                        <CardContent className="pt-6">
                            <div className="text-center">
                                <div className="text-3xl font-bold text-green-600">{stats.checkedIn}</div>
                                <div className="text-sm text-muted-foreground">Checked In</div>
                            </div>
                        </CardContent>
                    </Card>
                    <Card>
                        <CardContent className="pt-6">
                            <div className="text-center">
                                <div className="text-3xl font-bold text-orange-600">{stats.pending}</div>
                                <div className="text-sm text-muted-foreground">Pending</div>
                            </div>
                        </CardContent>
                    </Card>
                </div>
            )}

            {/* Scanner */}
            <Card className="mb-6">
                <CardHeader>
                    <CardTitle>QR Code Scanner</CardTitle>
                    <CardDescription>
                        Position the QR code within the frame to scan
                    </CardDescription>
                </CardHeader>
                <CardContent>
                    <div id="qr-reader" className="w-full mx-auto mb-4" style={{ maxWidth: '500px' }}></div>
                    
                    <div className="flex justify-center">
                        {!scanning ? (
                            <Button 
                                onClick={startScanner}
                                disabled={!selectedEvent}
                                size="lg"
                            >
                                <Camera className="mr-2 h-5 w-5" />
                                Start Scanning
                            </Button>
                        ) : (
                            <Button 
                                onClick={stopScanner}
                                variant="destructive"
                                size="lg"
                            >
                                <CameraOff className="mr-2 h-5 w-5" />
                                Stop Scanning
                            </Button>
                        )}
                    </div>
                </CardContent>
            </Card>

            {/* Result Display */}
            {result && (
                <div className={`border rounded-lg p-6 ${result.success !== false ? 'bg-green-50 border-green-200' : 'bg-red-50 border-red-200'}`}>
                    <div className="flex items-start gap-4">
                        {result.success !== false ? (
                            <CheckCircle className="h-8 w-8 text-green-600 flex-shrink-0" />
                        ) : (
                            <XCircle className="h-8 w-8 text-red-600 flex-shrink-0" />
                        )}
                        <div className="flex-1">
                            {result.success !== false ? (
                                <>
                                    <h3 className="text-2xl font-bold text-green-800 mb-2">
                                        {result.message || '✅ Welcome!'}
                                    </h3>
                                    <p className="text-xl font-semibold mb-1">
                                        {result.household.name}
                                    </p>
                                    <p className="text-sm text-muted-foreground">
                                        {result.household.guests} guest(s) • {result.household.email}
                                    </p>
                                    <p className="text-xs text-muted-foreground mt-1">
                                        Checked in at: {new Date(result.checked_in_at).toLocaleString()}
                                    </p>
                                </>
                            ) : (
                                <>
                                    <h3 className="text-xl font-bold text-red-800 mb-1">Error</h3>
                                    <p className="text-red-700">
                                        {result.error}
                                    </p>
                                </>
                            )}
                        </div>
                    </div>
                </div>
            )}

            {/* Manual Check-In Section */}
            {selectedEvent && (
                <Card className="mt-6">
                    <CardHeader>
                        <CardTitle>Manual Check-In</CardTitle>
                        <CardDescription>
                            Search and manually check in guests without QR code scanning
                        </CardDescription>
                    </CardHeader>
                    <CardContent>
                        {/* Search Box */}
                        <div className="mb-4">
                            <input
                                type="text"
                                placeholder="Search by name or email..."
                                value={searchTerm}
                                onChange={(e) => setSearchTerm(e.target.value)}
                                className="w-full p-3 border rounded-lg"
                            />
                        </div>

                        {/* Guest List */}
                        <div className="space-y-2 max-h-96 overflow-y-auto">
                            {filteredGuests.length === 0 ? (
                                <p className="text-center text-muted-foreground py-8">
                                    {searchTerm ? 'No guests found matching your search' : 'No attending guests for this event'}
                                </p>
                            ) : (
                                filteredGuests.map(guest => (
                                    <div
                                        key={`${guest.householdId}-${guest.eventId}`}
                                        className={`flex items-center justify-between p-3 border rounded-lg transition-colors ${
                                            guest.checkedIn ? 'bg-green-50 border-green-200' : 'bg-white hover:bg-gray-50'
                                        }`}
                                    >
                                        <div className="flex-1">
                                            <div className="flex items-center gap-2">
                                                <p className="font-semibold">{guest.householdName}</p>
                                                {guest.checkedIn && (
                                                    <CheckCircle className="h-4 w-4 text-green-600" />
                                                )}
                                            </div>
                                            <p className="text-sm text-muted-foreground">
                                                {guest.adultsCount + guest.childrenCount} guest(s)
                                                {guest.householdEmail && ` • ${guest.householdEmail}`}
                                            </p>
                                            {guest.checkedIn && guest.checkedInAt && (
                                                <p className="text-xs text-muted-foreground mt-1">
                                                    Checked in: {new Date(guest.checkedInAt).toLocaleString()}
                                                </p>
                                            )}
                                        </div>
                                        <Button
                                            onClick={() => toggleManualCheckIn(guest.householdId, guest.checkedIn)}
                                            variant={guest.checkedIn ? "outline" : "default"}
                                            size="sm"
                                        >
                                            {guest.checkedIn ? 'Undo' : 'Check In'}
                                        </Button>
                                    </div>
                                ))
                            )}
                        </div>
                    </CardContent>
                </Card>
            )}
        </div>
    );
}
