# Production Security Guide

## 🔐 Critical Security Issues & Solutions

### 1. SMTP Password Encryption

**Current Issue:** SMTP passwords are stored in plain text in the `user` table.

**Solutions:**

#### Option A: Use Symfony Secrets (Recommended)
```bash
# Initialize secrets for production
php bin/console secrets:set SMTP_PASSWORD

# Store in secrets vault instead of database
```

Then modify `EmailService.php` to read from secrets:
```php
use Symfony\Component\DependencyInjection\Attribute\Autowire;

public function __construct(
    #[Autowire('%env(SMTP_PASSWORD)%')] 
    private string $smtpPassword,
    // ... other dependencies
) {}
```

#### Option B: Encrypt at Application Level
Add encryption service:
```php
namespace App\Service;

class EncryptionService
{
    private string $encryptionKey;
    
    public function __construct(string $appSecret)
    {
        $this->encryptionKey = hash('sha256', $appSecret);
    }
    
    public function encrypt(string $data): string
    {
        $iv = openssl_random_pseudo_bytes(16);
        $encrypted = openssl_encrypt($data, 'AES-256-CBC', $this->encryptionKey, 0, $iv);
        return base64_encode($iv . $encrypted);
    }
    
    public function decrypt(string $data): string
    {
        $data = base64_decode($data);
        $iv = substr($data, 0, 16);
        $encrypted = substr($data, 16);
        return openssl_decrypt($encrypted, 'AES-256-CBC', $this->encryptionKey, 0, $iv);
    }
}
```

Update `User` entity getters/setters:
```php
public function setSmtpPassword(?string $smtpPassword): static
{
    $this->smtpPassword = $smtpPassword ? $this->encryptionService->encrypt($smtpPassword) : null;
    return $this;
}

public function getSmtpPassword(): ?string
{
    return $this->smtpPassword ? $this->encryptionService->decrypt($this->smtpPassword) : null;
}
```

#### Option C: Use Environment Variables Per User
Store SMTP credentials in environment variables instead of database:
```env
SMTP_HOST=smtp.gmail.com
SMTP_PORT=587
SMTP_USERNAME=your-email@gmail.com
SMTP_PASSWORD=your-app-password
```

**Recommended:** Use Option A (Symfony Secrets) for production.

---

### 2. Rate Limiting Configuration

Rate limiting has been added to prevent abuse:

**Configured Limits:**
- **RSVP Submissions:** 5 per 15 minutes per IP
- **API Requests:** 100 limit, 20 per minute refill
- **Email Sending:** 50 per hour
- **Login Attempts:** 5 per 15 minutes

**To adjust limits**, edit `config/packages/rate_limiter.yaml`

**Monitor rate limit hits** in logs:
```bash
tail -f var/log/prod.log | grep "rate_limit"
```

---

### 3. Additional Security Headers

Add to your web server configuration:

**Nginx:**
```nginx
add_header X-Frame-Options "SAMEORIGIN" always;
add_header X-Content-Type-Options "nosniff" always;
add_header X-XSS-Protection "1; mode=block" always;
add_header Referrer-Policy "strict-origin-when-cross-origin" always;
add_header Content-Security-Policy "default-src 'self'; script-src 'self' 'unsafe-inline'; style-src 'self' 'unsafe-inline';" always;
add_header Strict-Transport-Security "max-age=31536000; includeSubDomains" always;
```

**Apache (.htaccess):**
```apache
Header always set X-Frame-Options "SAMEORIGIN"
Header always set X-Content-Type-Options "nosniff"
Header always set X-XSS-Protection "1; mode=block"
Header always set Referrer-Policy "strict-origin-when-cross-origin"
Header always set Strict-Transport-Security "max-age=31536000; includeSubDomains"
```

---

### 4. Database Security

**Recommendations:**
1. Use separate database users for different operations:
   - `app_read`: Read-only for reporting
   - `app_write`: Full access for application
   - `app_admin`: Schema migrations only

2. Limit connections:
```sql
CREATE USER 'app_write'@'localhost' IDENTIFIED BY 'strong_password';
GRANT SELECT, INSERT, UPDATE, DELETE ON wedding_prod.* TO 'app_write'@'localhost';
```

3. Enable MySQL SSL:
```env
DATABASE_URL="mysql://user:pass@localhost:3306/db?serverVersion=8.0&sslmode=require"
```

4. Regular backups:
```bash
# Daily automated backup
0 2 * * * mysqldump -u backup_user -p wedding_prod > /backups/wedding_$(date +\%Y\%m\%d).sql
```

---

### 5. File Permissions

Set correct permissions on production:
```bash
# Application files (read-only)
chown -R www-data:www-data /var/www/wedding-rsvp
find /var/www/wedding-rsvp -type f -exec chmod 644 {} \;
find /var/www/wedding-rsvp -type d -exec chmod 755 {} \;

# Writable directories
chmod -R 775 /var/www/wedding-rsvp/var
chmod -R 775 /var/www/wedding-rsvp/public/uploads
```

---

### 6. Session Security

Add to `config/packages/framework.yaml`:
```yaml
framework:
    session:
        cookie_secure: true       # HTTPS only
        cookie_httponly: true     # No JavaScript access
        cookie_samesite: lax      # CSRF protection
        handler_id: null
        cookie_lifetime: 3600     # 1 hour
        gc_maxlifetime: 3600
```

---

### 7. Input Validation

Add validation constraints to API endpoints (see VALIDATION.md for details).

---

### 8. Monitoring & Alerts

**Recommended Tools:**
- **Sentry** - Error tracking and monitoring
- **New Relic** - Application performance monitoring
- **Uptime Robot** - Uptime monitoring
- **Log rotation** - Prevent disk space issues

**Basic log monitoring:**
```bash
# Set up logrotate
cat > /etc/logrotate.d/wedding-rsvp << EOF
/var/www/wedding-rsvp/var/log/*.log {
    daily
    missingok
    rotate 14
    compress
    delaycompress
    notifempty
    create 0640 www-data www-data
}
EOF
```

---

### 9. SSL/TLS Configuration

**Let's Encrypt (Free):**
```bash
certbot --nginx -d yourdomain.com -d www.yourdomain.com
```

**Force HTTPS in Nginx:**
```nginx
server {
    listen 80;
    server_name yourdomain.com;
    return 301 https://$server_name$request_uri;
}
```

---

### 10. Security Checklist

Before launching to production:

- [ ] `APP_SECRET` set to strong random value (64+ characters)
- [ ] `APP_ENV=prod` and `APP_DEBUG=0`
- [ ] SMTP passwords encrypted or in secrets vault
- [ ] Database uses strong passwords
- [ ] HTTPS/SSL certificate installed and forced
- [ ] Rate limiting enabled and tested
- [ ] Security headers configured in web server
- [ ] File permissions set correctly
- [ ] Session security configured
- [ ] Regular backups scheduled
- [ ] Error monitoring configured (Sentry/etc)
- [ ] Logs are being rotated
- [ ] `.env.prod.local` never committed to git
- [ ] All dependencies updated (`composer update`)
- [ ] Security audit run (`composer audit`)

**Run security audit:**
```bash
composer audit
symfony security:check
```

---

## 🚨 Emergency Response

If you detect a security breach:

1. **Immediately** change `APP_SECRET` and all passwords
2. Invalidate all active sessions: `php bin/console cache:clear`
3. Review access logs for suspicious activity
4. Check database for unauthorized changes
5. Notify users if data was compromised
6. Document the incident for future prevention

---

## 📞 Security Resources

- Symfony Security Best Practices: https://symfony.com/doc/current/security.html
- OWASP Top 10: https://owasp.org/www-project-top-ten/
- PHP Security Cheat Sheet: https://cheatsheetseries.owasp.org/cheatsheets/PHP_Configuration_Cheat_Sheet.html
