# Wedding RSVP Application

A comprehensive self-hosted wedding RSVP system built with Symfony 7.3 and React 19.

## Features

### For Guests
- **Three RSVP Modes:**
  - **HOUSEHOLD**: Group RSVP with guest counts and meal totals
  - **HOUSEHOLD_WITH_OPTIONAL_NAMES**: Group RSVP with optional name entry
  - **PER_GUEST**: Individual RSVP for each guest with personal meal choices

- **Public Features:**
  - Tokenized invite links (`/rsvp/{token}`)
  - Event schedule, FAQ, and travel information pages
  - ICS calendar file downloads for events
  - Mobile-responsive design with TailwindCSS
  - Deadline enforcement

### For Admins
- **Dashboard:** Quick stats (invited, responded, yes/no counts)
- **Guest Management:** CRUD operations for households and guests, invite link generation
- **Event Management:** Multiple events with meal options, capacity tracking
- **RSVP Tracking:** View all responses, filter by event/status
- **Data Conversion:** Switch between RSVP modes with automatic data migration
- **Exports:**
  - CSV: Caterer meal counts, dietary restrictions, attendance, guest lists
  - PDF: Place cards, table lists (requires DOMPDF)
- **Settings:** Configure RSVP mode, deadlines, couple info, staff PIN

### Additional Features
- **QR Check-in:** Staff PIN-protected scanner for event check-in
- **Theming:** Three presets (Blush, Classic, Dark) + custom colors
- **Security:** Session-based auth (ROLE_ADMIN, ROLE_STAFF), CSRF protection

## Tech Stack

**Backend:**
- Symfony 7.3 (PHP 8.2+)
- Doctrine ORM (MySQL/MariaDB)
- Twig templating

**Frontend:**
- React 19.2
- React Router v7
- TanStack React Query
- TailwindCSS 4.1
- shadcn/ui components
- Webpack Encore

## Architecture

The app follows Symfony UX Stimulus pattern:
1. Symfony Controller renders Twig template
2. Twig template mounts Stimulus controller
3. Stimulus controller bootstraps React app
4. React app makes API calls to Symfony backend

### Directory Structure

```
src/
├── Controller/
│   ├── WeddingController.php        # Public pages
│   ├── InviteController.php         # RSVP submission
│   ├── IcsController.php            # Calendar downloads
│   ├── QrController.php             # QR tickets
│   ├── CheckinController.php        # Event check-in
│   └── Api/AdminApiController.php   # Admin API
├── Entity/
│   ├── User.php
│   ├── Setting.php
│   ├── Event.php
│   ├── Household.php
│   ├── Guest.php
│   ├── HouseholdEventRsvp.php
│   ├── Rsvp.php
│   ├── Seating.php
│   ├── Checkin.php
│   ├── ThemeSetting.php
│   └── PageContent.php
├── Repository/
├── Service/
│   ├── RsvpConverterService.php     # Mode switching
│   ├── ExportService.php            # CSV generation
│   ├── IcsService.php               # Calendar files
│   └── QrService.php                # QR code generation

assets/controllers/react/
├── admin/
│   ├── App.js                       # Admin SPA
│   └── pages/
│       ├── Dashboard.js
│       ├── Guests.js
│       ├── Events.js
│       ├── RSVPs.js
│       └── Settings.js
└── rsvp/
    ├── InvitePage.js                # RSVP entry point
    ├── HouseholdForm.js             # Group RSVP form
    ├── PerGuestForm.js              # Individual RSVP form
    └── SuccessView.js               # Confirmation page
```

## Installation

### Requirements
- PHP 8.2+
- Composer
- Node.js 18+
- MySQL/MariaDB

### Quick Start

1. **Install Dependencies**
```bash
composer install
npm install
```

2. **Configure Database**
Edit `.env.local`:
```
DATABASE_URL="mysql://user:pass@localhost:3306/wedding_rsvp"
```

3. **Create Database & Run Migrations**
```bash
php bin/console doctrine:database:create
php bin/console doctrine:migrations:migrate
```

4. **Build Assets**
```bash
npm run build
```

5. **Start Server**
```bash
symfony server:start
# or
php -S localhost:8000 -t public
```

6. **Create Admin User**
```bash
php bin/console app:create-user admin@example.com password ROLE_ADMIN
```

7. **Access Application**
- Public: `http://localhost:8000`
- Admin: `http://localhost:8000/admin`

## Configuration

### RSVP Modes

**HOUSEHOLD Mode:**
- Guests enter total adults/children counts
- Meal selections aggregate by type
- Use: Small weddings, casual events

**HOUSEHOLD_WITH_OPTIONAL_NAMES Mode:**
- Same as HOUSEHOLD + optional name textarea
- Use: When you want guest names but not full individual tracking

**PER_GUEST Mode:**
- Each guest has individual status and meal choice
- Requires Guest entities to be created
- Use: Formal weddings, detailed seating arrangements

**Switching Modes:**
Use converter endpoints in Admin API:
- `/api/convert/household-to-guests` - Creates Guest entities from household RSVPs
- `/api/convert/guests-to-household` - Aggregates guest RSVPs into household totals

### Events Setup

1. Navigate to Admin > Events
2. Create events with:
   - Name, venue, date/time
   - Capacity (optional)
   - Meal options (if enabled)
3. Mark as "Public" to show on invite pages

### Guest Import

Prepare CSV with columns:
```
display_name,email,phone,side,tags
Smith Family,smith@example.com,555-1234,BRIDE,"vip,out-of-town"
```

Upload via Admin > Guests > Import

## API Endpoints

### Public
- `GET /` - Home page
- `GET /rsvp/{token}` - RSVP form
- `POST /api/rsvp/{token}` - Submit RSVP
- `GET /ics/event/{id}` - Download ICS file

### Admin (Auth Required)
- `GET/PATCH /api/settings` - Settings CRUD
- `GET/POST/PATCH/DELETE /api/events` - Events CRUD
- `GET/POST/PATCH/DELETE /api/households` - Households CRUD
- `GET /api/rsvps/household` - Household RSVPs
- `GET /api/rsvps/guest` - Per-guest RSVPs
- `GET /api/dashboard/stats` - Dashboard metrics
- `GET /api/export/{type}` - CSV exports
- `POST /api/convert/household-to-guests` - Mode converter
- `POST /api/checkin` - Record check-in

## Exports

### CSV Types
- **Caterer:** Meal code, name, total count (per event)
- **Dietary:** Household, dietary notes (per event)
- **Attendance:** Full RSVP details (all events)
- **Guests:** Guest list with contact info
- **Households:** Household list with invite tokens

Access via Admin > RSVPs > Export buttons

## Security

- **Authentication:** Session-based (Symfony Security)
- **Roles:** ROLE_ADMIN (full access), ROLE_STAFF (check-in only)
- **CSRF:** Enabled on forms
- **Rate Limiting:** Recommended via Symfony Rate Limiter bundle
- **Tokenized URLs:** Household invite tokens are 24-character random strings

## Customization

### Themes
Create theme presets in Admin > Theme with:
- Primary, accent, background, text colors
- Heading and body fonts

### Page Content
Edit public pages (Home, Schedule, FAQ, Travel, Our Story) in Admin > Pages
- Markdown support
- Publish toggle

## Troubleshooting

**RSVP not saving:**
- Check database connection
- Verify CSRF token in forms
- Check Symfony profiler for errors

**Assets not loading:**
- Run `npm run build`
- Clear cache: `php bin/console cache:clear`
- Check `public/build/` directory exists

**TypeScript errors in IDE:**
- These are benign compilation warnings
- JavaScript files in `assets/controllers/react/` are valid
- Can be ignored or fixed by updating `tsconfig.json`

## Production Deployment

1. **Environment:**
```bash
APP_ENV=prod
APP_DEBUG=0
```

2. **Optimize:**
```bash
composer install --no-dev --optimize-autoloader
npm run build
php bin/console cache:warmup
```

3. **Security:**
- Use HTTPS
- Set strong `APP_SECRET`
- Configure firewall rules
- Enable rate limiting

4. **Database:**
- Regular backups
- Optimize indexes

## License

Proprietary - For Etsy distribution

## Support

For issues and questions:
- Check Symfony documentation: https://symfony.com/doc
- React documentation: https://react.dev
- TailwindCSS: https://tailwindcss.com

---

Built with ❤️ for couples planning their special day
