# Production Deployment Guide

Complete guide for deploying the Wedding RSVP application to production.

## 📋 Pre-Deployment Checklist

### Requirements
- PHP 8.2 or higher
- Composer 2.x
- Node.js 18+ and npm
- MySQL 8.0 or MariaDB 10.5+
- Web server (Nginx or Apache)
- SSL certificate
- 512MB RAM minimum (1GB+ recommended)

### Prepare Your Server
```bash
# Update system
sudo apt update && sudo apt upgrade -y

# Install PHP and extensions
sudo apt install php8.2-fpm php8.2-cli php8.2-mysql php8.2-xml \
  php8.2-mbstring php8.2-curl php8.2-intl php8.2-zip

# Install Composer
curl -sS https://getcomposer.org/installer | php
sudo mv composer.phar /usr/local/bin/composer

# Install Node.js
curl -fsSL https://deb.nodesource.com/setup_18.x | sudo -E bash -
sudo apt install -y nodejs

# Install MySQL
sudo apt install mysql-server
sudo mysql_secure_installation
```

---

## 🚀 Deployment Steps

### 1. Clone Repository
```bash
cd /var/www
sudo git clone https://github.com/your-repo/wedding-rsvp.git
cd wedding-rsvp
```

### 2. Configure Environment
```bash
# Copy production environment template
cp .env.prod.local.example .env.prod.local

# Generate APP_SECRET
php -r "echo bin2hex(random_bytes(32));"

# Edit .env.prod.local with your values
nano .env.prod.local
```

**Required settings in `.env.prod.local`:**
```env
APP_ENV=prod
APP_DEBUG=0
APP_SECRET=<your-generated-secret>
DATABASE_URL="mysql://db_user:db_pass@localhost:3306/wedding_prod?serverVersion=8.0"
DEFAULT_URI=https://yourdomain.com
```

### 3. Install Dependencies
```bash
# Install PHP dependencies (production only)
composer install --no-dev --optimize-autoloader

# Install Node dependencies
npm install

# Build frontend assets
npm run build
```

### 4. Create Database
```bash
# Create production database
mysql -u root -p << EOF
CREATE DATABASE wedding_prod CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci;
CREATE USER 'wedding_user'@'localhost' IDENTIFIED BY 'STRONG_PASSWORD_HERE';
GRANT SELECT, INSERT, UPDATE, DELETE ON wedding_prod.* TO 'wedding_user'@'localhost';
FLUSH PRIVILEGES;
EOF

# Run migrations
php bin/console doctrine:migrations:migrate --no-interaction --env=prod
```

### 5. Create Admin User
```bash
# Create admin account
php bin/console app:create-user admin@yourdomain.com SecurePassword123! ROLE_ADMIN --env=prod
```

### 6. Set Permissions
```bash
# Set ownership
sudo chown -R www-data:www-data /var/www/wedding-rsvp

# Set directory permissions
find /var/www/wedding-rsvp -type d -exec chmod 755 {} \;
find /var/www/wedding-rsvp -type f -exec chmod 644 {} \;

# Set writable directories
chmod -R 775 /var/www/wedding-rsvp/var
chmod -R 775 /var/www/wedding-rsvp/public/uploads

# Set console executable
chmod +x /var/www/wedding-rsvp/bin/console
```

### 7. Clear and Warm Cache
```bash
php bin/console cache:clear --env=prod
php bin/console cache:warmup --env=prod
```

---

## 🌐 Web Server Configuration

### Nginx Configuration

Create `/etc/nginx/sites-available/wedding-rsvp`:

```nginx
server {
    listen 80;
    listen [::]:80;
    server_name yourdomain.com www.yourdomain.com;
    return 301 https://$server_name$request_uri;
}

server {
    listen 443 ssl http2;
    listen [::]:443 ssl http2;
    server_name yourdomain.com www.yourdomain.com;
    
    root /var/www/wedding-rsvp/public;
    index index.php;
    
    # SSL Configuration
    ssl_certificate /etc/letsencrypt/live/yourdomain.com/fullchain.pem;
    ssl_certificate_key /etc/letsencrypt/live/yourdomain.com/privkey.pem;
    ssl_protocols TLSv1.2 TLSv1.3;
    ssl_ciphers HIGH:!aNULL:!MD5;
    ssl_prefer_server_ciphers on;
    
    # Security Headers
    add_header X-Frame-Options "SAMEORIGIN" always;
    add_header X-Content-Type-Options "nosniff" always;
    add_header X-XSS-Protection "1; mode=block" always;
    add_header Referrer-Policy "strict-origin-when-cross-origin" always;
    add_header Strict-Transport-Security "max-age=31536000; includeSubDomains" always;
    
    # Logging
    access_log /var/log/nginx/wedding-rsvp-access.log;
    error_log /var/log/nginx/wedding-rsvp-error.log;
    
    # Client body size (for uploads)
    client_max_body_size 10M;
    
    location / {
        try_files $uri /index.php$is_args$args;
    }
    
    location ~ ^/index\.php(/|$) {
        fastcgi_pass unix:/var/run/php/php8.2-fpm.sock;
        fastcgi_split_path_info ^(.+\.php)(/.*)$;
        include fastcgi_params;
        fastcgi_param SCRIPT_FILENAME $realpath_root$fastcgi_script_name;
        fastcgi_param DOCUMENT_ROOT $realpath_root;
        internal;
    }
    
    location ~ \.php$ {
        return 404;
    }
    
    # Static files caching
    location ~* \.(js|css|png|jpg|jpeg|gif|ico|svg|woff|woff2|ttf)$ {
        expires 1y;
        add_header Cache-Control "public, immutable";
    }
}
```

Enable site:
```bash
sudo ln -s /etc/nginx/sites-available/wedding-rsvp /etc/nginx/sites-enabled/
sudo nginx -t
sudo systemctl reload nginx
```

### Apache Configuration

Create `/etc/apache2/sites-available/wedding-rsvp.conf`:

```apache
<VirtualHost *:80>
    ServerName yourdomain.com
    ServerAlias www.yourdomain.com
    Redirect permanent / https://yourdomain.com/
</VirtualHost>

<VirtualHost *:443>
    ServerName yourdomain.com
    ServerAlias www.yourdomain.com
    
    DocumentRoot /var/www/wedding-rsvp/public
    DirectoryIndex index.php
    
    # SSL Configuration
    SSLEngine on
    SSLCertificateFile /etc/letsencrypt/live/yourdomain.com/fullchain.pem
    SSLCertificateKeyFile /etc/letsencrypt/live/yourdomain.com/privkey.pem
    
    <Directory /var/www/wedding-rsvp/public>
        AllowOverride All
        Require all granted
        FallbackResource /index.php
    </Directory>
    
    # Security Headers
    Header always set X-Frame-Options "SAMEORIGIN"
    Header always set X-Content-Type-Options "nosniff"
    Header always set X-XSS-Protection "1; mode=block"
    Header always set Referrer-Policy "strict-origin-when-cross-origin"
    Header always set Strict-Transport-Security "max-age=31536000; includeSubDomains"
    
    # Logging
    ErrorLog ${APACHE_LOG_DIR}/wedding-rsvp-error.log
    CustomLog ${APACHE_LOG_DIR}/wedding-rsvp-access.log combined
</VirtualHost>
```

Enable site and modules:
```bash
sudo a2enmod rewrite ssl headers
sudo a2ensite wedding-rsvp
sudo apache2ctl configtest
sudo systemctl reload apache2
```

---

## 🔒 SSL Certificate (Let's Encrypt)

```bash
# Install Certbot
sudo apt install certbot python3-certbot-nginx

# Obtain certificate (Nginx)
sudo certbot --nginx -d yourdomain.com -d www.yourdomain.com

# OR for Apache
sudo apt install python3-certbot-apache
sudo certbot --apache -d yourdomain.com -d www.yourdomain.com

# Auto-renewal is set up automatically
# Test renewal:
sudo certbot renew --dry-run
```

---

## 💾 Database Backups

### Automated Daily Backups

Create `/usr/local/bin/wedding-backup.sh`:

```bash
#!/bin/bash

BACKUP_DIR="/backups/wedding-rsvp"
DB_NAME="wedding_prod"
DB_USER="wedding_user"
DB_PASS="your_password"
DATE=$(date +%Y%m%d_%H%M%S)

# Create backup directory
mkdir -p $BACKUP_DIR

# Dump database
mysqldump -u $DB_USER -p$DB_PASS $DB_NAME | gzip > $BACKUP_DIR/wedding_${DATE}.sql.gz

# Keep only last 30 days
find $BACKUP_DIR -name "wedding_*.sql.gz" -mtime +30 -delete

# Backup uploaded files
tar -czf $BACKUP_DIR/uploads_${DATE}.tar.gz /var/www/wedding-rsvp/public/uploads

echo "Backup completed: $DATE"
```

Make executable and schedule:
```bash
chmod +x /usr/local/bin/wedding-backup.sh

# Add to crontab (daily at 2 AM)
sudo crontab -e
0 2 * * * /usr/local/bin/wedding-backup.sh >> /var/log/wedding-backup.log 2>&1
```

---

## 📊 Monitoring & Maintenance

### Log Rotation

Create `/etc/logrotate.d/wedding-rsvp`:

```
/var/www/wedding-rsvp/var/log/*.log {
    daily
    missingok
    rotate 14
    compress
    delaycompress
    notifempty
    create 0640 www-data www-data
    sharedscripts
    postrotate
        systemctl reload php8.2-fpm
    endscript
}
```

### Health Check Endpoint

Monitor application health:
```bash
# Check application status
curl https://yourdomain.com/

# Monitor logs
tail -f /var/www/wedding-rsvp/var/log/prod.log
tail -f /var/log/nginx/wedding-rsvp-error.log
```

### Performance Monitoring

Install OPcache for PHP:
```bash
sudo apt install php8.2-opcache

# Edit /etc/php/8.2/fpm/conf.d/10-opcache.ini
opcache.enable=1
opcache.memory_consumption=256
opcache.max_accelerated_files=20000
opcache.validate_timestamps=0

sudo systemctl restart php8.2-fpm
```

---

## 🔄 Updates & Maintenance

### Deploying Updates

```bash
cd /var/www/wedding-rsvp

# Backup database before update
mysqldump -u wedding_user -p wedding_prod > /tmp/backup_before_update.sql

# Pull latest changes
git pull origin main

# Install dependencies
composer install --no-dev --optimize-autoloader
npm install
npm run build

# Run migrations
php bin/console doctrine:migrations:migrate --no-interaction --env=prod

# Clear cache
php bin/console cache:clear --env=prod
php bin/console cache:warmup --env=prod

# Test
curl https://yourdomain.com/
```

### Zero-Downtime Deployment

For critical production systems:

1. Set up second server or use blue-green deployment
2. Update second server
3. Test thoroughly
4. Switch traffic to new server
5. Monitor for issues

---

## 🐛 Troubleshooting

### Common Issues

**500 Internal Server Error:**
```bash
# Check PHP-FPM logs
tail -f /var/log/php8.2-fpm.log

# Check application logs
tail -f /var/www/wedding-rsvp/var/log/prod.log

# Check permissions
ls -la /var/www/wedding-rsvp/var
```

**Database Connection Errors:**
```bash
# Test database connection
php bin/console doctrine:query:sql "SELECT 1" --env=prod

# Check MySQL is running
sudo systemctl status mysql
```

**Assets Not Loading:**
```bash
# Rebuild assets
npm run build

# Check public directory
ls -la /var/www/wedding-rsvp/public/build
```

---

## 📝 Post-Deployment Tasks

1. ✅ Test all critical functionality:
   - User login
   - RSVP submission
   - Admin dashboard
   - Email sending
   - Data exports

2. ✅ Configure monitoring:
   - Set up Uptime monitoring
   - Configure error alerting
   - Set up performance monitoring

3. ✅ Document:
   - Admin credentials (secure storage)
   - Backup procedures
   - Rollback procedures
   - Emergency contacts

4. ✅ Security scan:
```bash
composer audit
symfony security:check
```

---

## 🆘 Emergency Procedures

### Rollback

```bash
# Restore database
mysql -u wedding_user -p wedding_prod < /backups/wedding-rsvp/wedding_YYYYMMDD.sql

# Revert code
git reset --hard <previous-commit-hash>
composer install --no-dev --optimize-autoloader
php bin/console cache:clear --env=prod
```

### Emergency Maintenance Mode

Create `public/maintenance.html` and redirect all traffic:

**Nginx:**
```nginx
if (-f $document_root/maintenance.html) {
    return 503;
}
error_page 503 @maintenance;
location @maintenance {
    rewrite ^(.*)$ /maintenance.html break;
}
```

---

## 📞 Support Resources

- **Symfony Documentation:** https://symfony.com/doc/current/deployment.html
- **DigitalOcean Tutorials:** https://www.digitalocean.com/community/tags/symfony
- **Stack Overflow:** Tag questions with `symfony` and `deployment`

---

**Deployment completed? Update this checklist:**

- [ ] Application accessible via HTTPS
- [ ] Admin login works
- [ ] RSVP submission works
- [ ] Email sending configured and tested
- [ ] Backups configured and tested
- [ ] Monitoring set up
- [ ] Security headers verified
- [ ] SSL certificate verified (A+ rating on ssllabs.com)
- [ ] Performance tested
- [ ] Documentation updated
