# Production Readiness Checklist

## ✅ COMPLETED ITEMS

### Security
- ✅ Password hashing configured (auto algorithm)
- ✅ CSRF protection enabled on forms
- ✅ Session-based authentication
- ✅ Role-based access control (ROLE_ADMIN, ROLE_STAFF)
- ✅ Tokenized RSVP URLs (24-char random strings)
- ✅ Password reset functionality with expiring tokens
- ✅ Public routes properly configured in security.yaml

### Database
- ✅ Doctrine ORM configured
- ✅ Migrations created and organized
- ✅ Entities properly mapped with relationships

### Email System
- ✅ SMTP configuration support
- ✅ Invite email templates
- ✅ Reminder email templates
- ✅ Password reset email templates
- ✅ Email service with error handling

### Frontend
- ✅ React 19 with modern hooks
- ✅ TailwindCSS 4.1 for styling
- ✅ shadcn/ui components
- ✅ Responsive mobile design
- ✅ Beautiful luxury-themed UI

### Features
- ✅ Complete RSVP system (3 modes)
- ✅ Admin dashboard with statistics
- ✅ Guest & household management
- ✅ Event management
- ✅ Seating arrangements
- ✅ QR code check-in system
- ✅ CSV exports
- ✅ Theme customization
- ✅ Password reset system

## ⚠️ CRITICAL - MUST DO BEFORE PRODUCTION

### 1. Environment Configuration
**Status:** ❌ NOT CONFIGURED

Create `.env.local` (NOT committed to git) with:

```bash
# PRODUCTION ENVIRONMENT
APP_ENV=prod
APP_DEBUG=0

# GENERATE A NEW SECRET (64+ characters)
APP_SECRET=CHANGE_ME_TO_A_RANDOM_64_CHAR_STRING

# Production Database
DATABASE_URL="mysql://prod_user:STRONG_PASSWORD@localhost:3306/wedding_prod?serverVersion=8.0"

# Base URL for emails
DEFAULT_URI=https://your-domain.com
```

### 2. Run Database Migration
**Status:** ⚠️ PENDING

Run the password reset migration:

```bash
php bin/console doctrine:migrations:migrate --no-interaction
```

### 3. Security Hardening
**Status:** ⚠️ NEEDS ATTENTION

#### A. Create .env.local
```bash
# DO NOT commit this file to git
echo ".env.local" >> .gitignore
```

#### B. Generate Strong APP_SECRET
```bash
# Generate 64-character random string
php -r "echo bin2hex(random_bytes(32)) . PHP_EOL;"
```

#### C. Configure HTTPS
- ✅ SSL certificate installed
- ✅ Force HTTPS redirect in web server config
- ✅ Set secure cookies (automatic in prod mode)

### 4. Web Server Configuration
**Status:** ⚠️ NEEDS SETUP

#### Apache (.htaccess already in /public)
Ensure `AllowOverride All` is set for the document root

#### Nginx (recommended config)
```nginx
server {
    listen 80;
    server_name your-domain.com;
    return 301 https://$server_name$request_uri;
}

server {
    listen 443 ssl http2;
    server_name your-domain.com;
    root /path/to/project/public;

    ssl_certificate /path/to/cert.pem;
    ssl_certificate_key /path/to/key.pem;

    location / {
        try_files $uri /index.php$is_args$args;
    }

    location ~ ^/index\.php(/|$) {
        fastcgi_pass unix:/var/run/php/php8.2-fpm.sock;
        fastcgi_split_path_info ^(.+\.php)(/.*)$;
        include fastcgi_params;
        fastcgi_param SCRIPT_FILENAME $realpath_root$fastcgi_script_name;
        fastcgi_param DOCUMENT_ROOT $realpath_root;
        internal;
    }

    location ~ \.php$ {
        return 404;
    }
}
```

### 5. Optimize for Production
**Status:** ⚠️ PENDING

```bash
# Install production dependencies only
composer install --no-dev --optimize-autoloader

# Build production assets
npm run build

# Warm up cache
php bin/console cache:warmup --env=prod

# Set correct permissions
chmod -R 755 var/
chmod -R 755 public/build/
```

### 6. Database Backups
**Status:** ⚠️ NEEDS SETUP

Set up automated daily backups:

```bash
# Example cron job (daily at 2 AM)
0 2 * * * mysqldump -u USER -pPASS wedding_prod > /backups/wedding_$(date +\%Y\%m\%d).sql
```

### 7. SMTP Configuration
**Status:** ⚠️ NEEDS USER INPUT

Configure SMTP in Admin panel:
1. Login to `/admin`
2. Go to Settings
3. Configure SMTP credentials:
   - Host (e.g., smtp.gmail.com, smtp.mailgun.org)
   - Port (587 for TLS, 465 for SSL)
   - Username & Password
   - From Email & Name

**Test before sending invites!**

### 8. Rate Limiting (Recommended)
**Status:** ⚠️ OPTIONAL BUT RECOMMENDED

Protect against brute force:

```yaml
# config/packages/rate_limiter.yaml
framework:
    rate_limiter:
        login:
            policy: 'sliding_window'
            limit: 5
            interval: '15 minutes'
```

Update SecurityController to use rate limiter.

### 9. Error Monitoring (Recommended)
**Status:** ⚠️ OPTIONAL

Consider adding:
- Sentry.io for error tracking
- Monolog for structured logging

### 10. Performance Optimization
**Status:** ⚠️ OPTIONAL

```bash
# Enable OPcache in php.ini
opcache.enable=1
opcache.memory_consumption=256
opcache.max_accelerated_files=20000

# Install APCu for better caching
apt-get install php8.2-apcu
```

## 🔍 PRE-LAUNCH TESTING

### Manual Testing Checklist
- [ ] Login page works correctly
- [ ] Password reset flow works (email sent & received)
- [ ] Admin dashboard loads
- [ ] Create household and generate invite link
- [ ] Test RSVP submission as guest
- [ ] Verify RSVP appears in admin panel
- [ ] Test CSV exports
- [ ] Test QR check-in system
- [ ] Mobile responsiveness on phone/tablet
- [ ] Email delivery works (test with real SMTP)

### Security Testing
- [ ] Cannot access /admin without login
- [ ] Cannot access API endpoints without auth
- [ ] CSRF tokens working on forms
- [ ] Password reset tokens expire after 1 hour
- [ ] Invalid tokens show proper error messages

## 📋 LAUNCH DAY

### Final Steps
1. [ ] Run migrations on production database
2. [ ] Create admin user account
3. [ ] Test login
4. [ ] Configure SMTP and send test email
5. [ ] Import guest list
6. [ ] Create events
7. [ ] Test at least one full RSVP flow
8. [ ] Monitor error logs for first few hours

## 📝 MAINTENANCE

### Regular Tasks
- **Daily:** Monitor error logs
- **Weekly:** Check disk space, verify backups
- **Monthly:** Review RSVP statistics
- **Before event:** Export final guest list and meal counts

### Database Cleanup (Post-Wedding)
```sql
-- Archive data, don't delete immediately
-- Keep for 6-12 months for records
```

## 🚨 KNOWN LIMITATIONS

1. **Single Tenant:** App designed for one wedding
2. **SMTP Required:** Password reset requires SMTP configuration
3. **No built-in email queue:** Large invite batches sent synchronously
4. **File uploads:** Not currently supported (add if needed)
5. **Timezone:** Uses server timezone, not configurable per-event

## 📞 EMERGENCY CONTACTS

- **Hosting Provider:** _____________
- **Domain Registrar:** _____________
- **Email Provider:** _____________
- **Database Backup Location:** _____________

## ✨ CURRENT STATUS SUMMARY

### Ready ✅
- Core application functionality
- Security foundation
- Beautiful UI/UX
- Password reset system
- Email templates

### Needs Configuration ⚠️
- Production environment variables
- SMTP credentials
- Web server setup
- SSL certificate
- Database migration (password reset fields)

### Production Risk Assessment

**Overall: 🟡 ALMOST READY**

The application is functionally complete and secure, but requires standard production configuration before launch. Follow the checklist above to ensure a smooth deployment.

### Estimated Time to Production Ready
- **Basic setup:** 2-4 hours
- **With testing:** 4-8 hours
- **With optimization:** 1-2 days

---

**Last Updated:** October 31, 2025
**Version:** 1.0
