# Individual Guest Details - Database & API Update Guide

## Problem
The frontend RSVP form collects detailed individual guest information (names, meal choices, dietary restrictions for each adult/child), but the backend database and API don't properly store or retrieve this data.

## Solution Overview
1. ✅ Database migration created
2. ⚠️ Entity updates needed
3. ⚠️ API endpoints need updating
4. ✅ Admin UI updated to display data

---

## Step 1: Run Database Migration

The migration file has been created: `migrations/Version20251029170500.php`

**Run this command to apply it:**
```bash
php bin/console doctrine:migrations:migrate
```

**This adds the following columns to the `guest` table:**
- `type` - VARCHAR(20) - "ADULT" or "CHILD"
- `meal_choice` - VARCHAR(100) - Guest's meal selection
- `dietary_restrictions` - TEXT - Dietary restrictions/allergies  
- `attending` - TINYINT(1) - Whether guest is attending
- `notes` - TEXT - Additional notes from RSVP

---

## Step 2: Update Guest Entity

**File:** `src/Entity/Guest.php`

Add these properties and methods:

```php
#[ORM\Column(type: 'string', length: 20, nullable: true)]
private ?string $type = 'ADULT';

#[ORM\Column(type: 'string', length: 100, nullable: true)]
private ?string $mealChoice = null;

#[ORM\Column(type: 'text', nullable: true)]
private ?string $dietaryRestrictions = null;

#[ORM\Column(type: 'boolean', nullable: true)]
private ?bool $attending = null;

#[ORM\Column(type: 'text', nullable: true)]
private ?string $notes = null;

// Add getters and setters for each property
public function getType(): ?string
{
    return $this->type;
}

public function setType(?string $type): self
{
    $this->type = $type;
    return $this;
}

// ... etc for all properties
```

---

## Step 3: Update API Endpoints

### Update Household API Response (`/api/households`)

**File:** `src/Controller/Api/HouseholdController.php`

Ensure the response includes guest details:

```php
return $this->json([
    'id' => $household->getId(),
    'displayName' => $household->getDisplayName(),
    'email' => $household->getEmail(),
    'guestCount' => count($household->getGuests()),
    'guests' => array_map(function($guest) {
        return [
            'id' => $guest->getId(),
            'firstName' => $guest->getFirstName(),
            'lastName' => $guest->getLastName(),
            'name' => $guest->getFirstName() . ' ' . $guest->getLastName(),
            'type' => $guest->getType(), // NEW
            'mealChoice' => $guest->getMealChoice(), // NEW
            'dietary' => $guest->getDietaryRestrictions(), // NEW
            'attending' => $guest->getAttending(), // NEW
            'notes' => $guest->getNotes(), // NEW
        ];
    }, $household->getGuests()->toArray()),
    // ... other fields
]);
```

### Update RSVP Submission Endpoint

**File:** `src/Controller/RsvpController.php`

When processing RSVP submission, save individual guest details:

```php
// Example: When processing RSVP form data
foreach ($rsvpData['guests'] as $guestData) {
    $guest = new Guest();
    $guest->setHousehold($household);
    $guest->setFirstName($guestData['firstName']);
    $guest->setLastName($guestData['lastName']);
    $guest->setType($guestData['type']); // 'ADULT' or 'CHILD'
    $guest->setMealChoice($guestData['mealChoice']);
    $guest->setDietaryRestrictions($guestData['dietary']);
    $guest->setAttending($guestData['attending'] ?? true);
    
    $entityManager->persist($guest);
}
```

---

## Step 4: Test the Flow

### 1. Test Frontend RSVP Form
1. Go to: `http://localhost:8000/invite/[token]`
2. Fill in guest details:
   - Adult names
   - Child names  
   - Meal choices
   - Dietary restrictions
3. Submit the form

### 2. Verify Data in Database
```sql
SELECT * FROM guest;
```

Should show:
- first_name, last_name
- type (ADULT/CHILD)
- meal_choice
- dietary_restrictions
- attending

### 3. Test Admin View
1. Go to: `http://localhost:8000/admin/guests`
2. Click "View Guest Details" on any household
3. Should see:
   - Individual guest names
   - Meal choices
   - Dietary restrictions

---

## Step 5: Additional Enhancements (Optional)

### Add to Seating Page
Update `/admin/seating` to show meal choices when assigning seats

### Add to RSVPs Page  
Show dietary restrictions in RSVP export for caterers

### Add Reporting
Create `/admin/reports/meals` to show:
- Total count by meal choice
- List of all dietary restrictions
- Seating requirements

---

## Current Status

✅ **Database Migration** - Created (needs to be run)  
✅ **Admin UI** - Updated to display guest details  
⚠️ **Entity** - Needs property additions  
⚠️ **API Endpoints** - Need to save/return guest details  
⚠️ **Migration Execution** - Needs to be run  

---

## Quick Start Commands

```bash
# 1. Run the migration
php bin/console doctrine:migrations:migrate

# 2. Check migration status
php bin/console doctrine:migrations:status

# 3. If needed, rollback
php bin/console doctrine:migrations:migrate prev
```

---

## Questions?

The structure is now in place. Once you:
1. Run the migration
2. Update the Guest entity
3. Update API endpoints to save/return data

The admin will automatically display all individual guest details with meal choices and dietary restrictions! 🎉
