# Email & Invites System Implementation

## ✅ What's Been Added

### 1. Database (Setting Entity)
Added SMTP configuration fields to `Setting` entity:
- `smtpHost` - SMTP server host
- `smtpPort` - SMTP port (default: 587)
- `smtpUsername` - SMTP username
- `smtpPassword` - SMTP password
- `smtpEncryption` - Encryption type (tls/ssl)
- `emailFromAddress` - From email address
- `emailFromName` - From name

### 2. Email Service
Created `src/Service/EmailService.php` with:
- **sendInviteEmail()** - Send single invite
- **sendBulkInvites()** - Send multiple invites
- **testSmtpConnection()** - Test SMTP settings
- Beautiful HTML email template

### 3. API Endpoints
Added to `AdminApiController`:
- `POST /api/email/test-smtp` - Test SMTP connection
- `POST /api/email/send-invite` - Send single invite
- `POST /api/email/send-bulk` - Send bulk invites

##  ⚠️ IMPORTANT: Database Migration Needed

Run these commands to add SMTP fields to database:

```bash
php bin/console make:migration
php bin/console doctrine:migrations:migrate
```

Or manually add columns:
```sql
ALTER TABLE setting ADD COLUMN smtp_host VARCHAR(255) NULL;
ALTER TABLE setting ADD COLUMN smtp_port INT NULL DEFAULT 587;
ALTER TABLE setting ADD COLUMN smtp_username VARCHAR(255) NULL;
ALTER TABLE setting ADD COLUMN smtp_password VARCHAR(255) NULL;
ALTER TABLE setting ADD COLUMN smtp_encryption VARCHAR(50) NULL DEFAULT 'tls';
ALTER TABLE setting ADD COLUMN email_from_address VARCHAR(255) NULL;
ALTER TABLE setting ADD COLUMN email_from_name VARCHAR(255) NULL;
```

## 🔧 TODO: Complete Implementation

### 1. Update Settings Page
Add SMTP section to `assets/controllers/react/admin/pages/Settings.js`:

```javascript
// Add SMTP Settings Section
<div className="space-y-4">
    <h3 className="text-lg font-semibold">📧 SMTP Email Settings</h3>
    
    <div className="grid grid-cols-2 gap-4">
        <div>
            <label>SMTP Host</label>
            <input
                type="text"
                value={settings.smtpHost || ''}
                onChange={(e) => setSettings({...settings, smtpHost: e.target.value})}
                placeholder="smtp.gmail.com"
            />
        </div>
        <div>
            <label>SMTP Port</label>
            <input
                type="number"
                value={settings.smtpPort || 587}
                onChange={(e) => setSettings({...settings, smtpPort: parseInt(e.target.value)})}
            />
        </div>
    </div>
    
    <div className="grid grid-cols-2 gap-4">
        <div>
            <label>SMTP Username</label>
            <input
                type="text"
                value={settings.smtpUsername || ''}
                onChange={(e) => setSettings({...settings, smtpUsername: e.target.value})}
            />
        </div>
        <div>
            <label>SMTP Password</label>
            <input
                type="password"
                value={settings.smtpPassword || ''}
                onChange={(e) => setSettings({...settings, smtpPassword: e.target.value})}
            />
        </div>
    </div>
    
    <div className="grid grid-cols-3 gap-4">
        <div>
            <label>Encryption</label>
            <select
                value={settings.smtpEncryption || 'tls'}
                onChange={(e) => setSettings({...settings, smtpEncryption: e.target.value})}
            >
                <option value="tls">TLS</option>
                <option value="ssl">SSL</option>
            </select>
        </div>
        <div>
            <label>From Email</label>
            <input
                type="email"
                value={settings.emailFromAddress || ''}
                onChange={(e) => setSettings({...settings, emailFromAddress: e.target.value})}
            />
        </div>
        <div>
            <label>From Name</label>
            <input
                type="text"
                value={settings.emailFromName || ''}
                onChange={(e) => setSettings({...settings, emailFromName: e.target.value})}
            />
        </div>
    </div>
    
    <Button onClick={testSmtp}>Test SMTP Connection</Button>
</div>
```

### 2. Create Invites Page
Create `assets/controllers/react/admin/pages/Invites.js`:

```javascript
import React, { useState, useEffect } from 'react';
import { Button } from '@/components/ui/button';

export default function Invites() {
    const [events, setEvents] = useState([]);
    const [households, setHouseholds] = useState([]);
    const [selectedEvent, setSelectedEvent] = useState('');
    const [selectedHouseholds, setSelectedHouseholds] = useState([]);
    const [sending, setSending] = useState(false);

    useEffect(() => {
        fetch('/api/events').then(r => r.json()).then(setEvents);
        fetch('/api/households').then(r => r.json()).then(setHouseholds);
    }, []);

    const sendInvites = async () => {
        setSending(true);
        const event = events.find(e => e.id === parseInt(selectedEvent));
        
        const recipients = selectedHouseholds.map(id => {
            const household = households.find(h => h.id === id);
            return {
                email: household.email,
                name: household.displayName,
                eventName: event?.name,
                eventSlug: event?.slug
            };
        });

        const response = await fetch('/api/email/send-bulk', {
            method: 'POST',
            headers: { 'Content-Type': 'application/json' },
            body: JSON.stringify({ recipients, eventSlug: event?.slug })
        });

        const result = await response.json();
        alert(`✓ Sent: ${result.sent}, ❌ Failed: ${result.failed}`);
        setSending(false);
        setSelectedHouseholds([]);
    };

    return (
        <div>
            <h1 className="text-3xl font-bold mb-6">📧 Send Invitations</h1>
            
            <div className="bg-card rounded-lg border p-6 mb-6">
                <h3 className="font-semibold mb-4">Select Event</h3>
                <select
                    value={selectedEvent}
                    onChange={(e) => setSelectedEvent(e.target.value)}
                    className="w-full px-4 py-2 border rounded"
                >
                    <option value="">All Events (/rsvp)</option>
                    {events.map(event => (
                        <option key={event.id} value={event.id}>
                            {event.name} (/rsvp/{event.slug})
                        </option>
                    ))}
                </select>
            </div>

            <div className="bg-card rounded-lg border p-6">
                <h3 className="font-semibold mb-4">Select Recipients</h3>
                <div className="space-y-2 mb-4">
                    {households.filter(h => h.email).map(household => (
                        <label key={household.id} className="flex items-center gap-2">
                            <input
                                type="checkbox"
                                checked={selectedHouseholds.includes(household.id)}
                                onChange={(e) => {
                                    if (e.target.checked) {
                                        setSelectedHouseholds([...selectedHouseholds, household.id]);
                                    } else {
                                        setSelectedHouseholds(selectedHouseholds.filter(id => id !== household.id));
                                    }
                                }}
                            />
                            <span>{household.displayName} ({household.email})</span>
                        </label>
                    ))}
                </div>

                <div className="flex gap-2">
                    <Button 
                        onClick={sendInvites}
                        disabled={sending || selectedHouseholds.length === 0}
                    >
                        {sending ? 'Sending...' : `Send to ${selectedHouseholds.length} Recipients`}
                    </Button>
                    <Button variant="outline" onClick={() => setSelectedHouseholds([])}>
                        Clear Selection
                    </Button>
                </div>
            </div>
        </div>
    );
}
```

### 3. Update Sidebar Navigation
In `assets/controllers/react/admin/App.js`, add Invites menu:

```javascript
import Invites from './pages/Invites';

// In routes
<Route path="/invites" element={<Invites />} />

// In sidebar
<Link to="/invites" className={linkClass('/invites')}>
    <span>📧</span>
    {expanded && <span>Invites</span>}
</Link>
```

### 4. Update AdminApiController Settings
Add SMTP fields to settings update:

```php
if (isset($data['smtpHost'])) $settings->setSmtpHost($data['smtpHost']);
if (isset($data['smtpPort'])) $settings->setSmtpPort($data['smtpPort']);
if (isset($data['smtpUsername'])) $settings->setSmtpUsername($data['smtpUsername']);
if (isset($data['smtpPassword'])) $settings->setSmtpPassword($data['smtpPassword']);
if (isset($data['smtpEncryption'])) $settings->setSmtpEncryption($data['smtpEncryption']);
if (isset($data['emailFromAddress'])) $settings->setEmailFromAddress($data['emailFromAddress']);
if (isset($data['emailFromName'])) $settings->setEmailFromName($data['emailFromName']);
```

## 📧 SMTP Configuration Examples

### Gmail
```
Host: smtp.gmail.com
Port: 587
Encryption: TLS
Username: your-email@gmail.com
Password: App Password (not your regular password!)
```

**Gmail App Password:** https://myaccount.google.com/apppasswords

### Mailtrap (Testing)
```
Host: sandbox.smtp.mailtrap.io
Port: 2525
Encryption: TLS
Username: (from Mailtrap)
Password: (from Mailtrap)
```

### Sendgrid
```
Host: smtp.sendgrid.net
Port: 587
Encryption: TLS
Username: apikey
Password: (Your Sendgrid API key)
```

## 🚀 How to Use

### 1. Configure SMTP
1. Go to **Admin → Settings**
2. Scroll to **SMTP Email Settings**
3. Enter your SMTP details
4. Click **Test SMTP Connection**
5. Save settings

### 2. Send Invites
1. Go to **Admin → Invites** (new menu item)
2. Select event (or leave blank for all events)
3. Check households to invite
4. Click **Send to X Recipients**
5. Wait for confirmation

### 3. What Guests Receive
Beautiful HTML email with:
- Couple names
- Event details
- Wedding date
- **RSVP Now** button with unique link
- Personal greeting

## 🔒 Security Notes

- SMTP passwords stored in database (encrypt in production!)
- Only ROLE_ADMIN can send emails
- Rate limiting recommended for production
- Use App Passwords (not regular passwords) for Gmail

## 🎨 Email Template

The email includes:
- ✅ Pink gradient header
- ✅ Couple names and wedding info
- ✅ Event-specific details
- ✅ Big "RSVP Now" button
- ✅ Direct link to RSVP page
- ✅ Mobile responsive design

## 📊 Email Tracking (Future)

Consider adding:
- Email open tracking
- Click tracking
- Bounce handling
- Resend functionality
- Email history log

## 🐛 Troubleshooting

**"SMTP connection failed"**
- Check host/port/credentials
- Try port 465 with SSL instead of 587/TLS
- Enable "Less secure app access" (Gmail)
- Use App Password (Gmail)

**"Failed to send email"**
- Verify from email matches SMTP username
- Check firewall/antivirus blocking port
- Try Mailtrap for testing first

**Email not received**
- Check spam folder
- Verify recipient email address
- Check SMTP sending limits
- Review email service logs

---

**Next Steps:**
1. Run database migration
2. Update Settings page with SMTP fields
3. Create Invites page
4. Add to sidebar navigation
5. Configure SMTP settings
6. Test with Mailtrap
7. Send real invites!
